// -*- c++ -*-
/// @file LtlGraph.C
/*
 * This file is based on code originally written by Mauno Rnkk,
 * Copyright  1998 Mauno Rnkk <mronkko@abo.fi>.
 *
 * Modifications by Heikki Tauriainen <Heikki.Tauriainen@hut.fi>
 * and Marko Mkel <Marko.Makela@hut.fi>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "LtlGraph.h"

/// An empty formula set
static const std::set<const class Ltl*> e;
/// An empty atomic proposition set
static const std::set<const class LtlAtom*> eatom;

LtlGraph::LtlGraph (const class Ltl& formula) :
  m_next (1), m_nodes ()
{
  std::set<unsigned> incoming;
  incoming.insert (0);

  std::set<const class Ltl*> new_set;
  new_set.insert (&formula);

  expand (1, *new class LtlGraphNode (incoming, new_set, e, eatom, e));
}

void
LtlGraph::expand (unsigned state,
		  class LtlGraphNode& node)
{
  if (!node.m_new.empty ()) {
    // expand a newly introduced proposition
    std::set<const class Ltl*>::iterator i = node.m_new.begin ();
    const class Ltl* eta = *i;
    node.m_old.insert (eta);
    node.m_new.erase (i);
    eta->expand (state, node, *this);
  }
  else {
    // no new propositions: merge the node with the graph
    // search for an existing node having the same 'old' and 'next' sets
    iterator i;
    for (i = m_nodes.begin (); i != m_nodes.end (); i++)
      if (i->second.m_old == node.m_old && i->second.m_next == node.m_next)
	break;
    if (i != m_nodes.end ()) {
      // merge the node with an existing one
      for (std::set<unsigned>::const_iterator j = node.m_incoming.begin ();
	   j != node.m_incoming.end(); j++)
	i->second.m_incoming.insert (*j);
      delete &node;
    }
    else {
      // a genuinely new node: insert it to the graph and expand further
      m_nodes.insert (std::pair<unsigned,class LtlGraphNode> (m_next, node));
      std::set<unsigned> incoming;
      incoming.insert (m_next++);
      class LtlGraphNode* node2 =
	new class LtlGraphNode (incoming, node.m_next, e, eatom, e);
      delete &node;
      expand (m_next, *node2);
    }
  }
}
