// -*- c++ -*-
/// @file Ltl.C
/*
 * This file is based on code originally written by Mauno Rnkk,
 * Copyright  1998 Mauno Rnkk <mronkko@abo.fi>.
 *
 * Modifications by Heikki Tauriainen <Heikki.Tauriainen@hut.fi>
 * and Marko Mkel <Marko.Makela@hut.fi>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "Ltl.h"
#include "LtlGraph.h"
#include <assert.h>

Ltl::Store Ltl::m_store;

bool
Ltl::operator< (const class Ltl& other) const
{
  if (getKind () < other.getKind ()) return true;
  if (other.getKind () < getKind ()) return false;
  switch (getKind ()) {
  case Atom:
    return static_cast<const class LtlAtom&>(*this) <
      static_cast<const class LtlAtom&>(other);
  case Constant:
    return static_cast<const class LtlConstant&>(*this) <
      static_cast<const class LtlConstant&>(other);
  case Junct:
    return static_cast<const class LtlJunct&>(*this) <
      static_cast<const class LtlJunct&>(other);
  case Iff:
    return static_cast<const class LtlIff&>(*this) <
      static_cast<const class LtlIff&>(other);
  case Future:
    return static_cast<const class LtlFuture&>(*this) <
      static_cast<const class LtlFuture&>(other);
  case Until:
    return static_cast<const class LtlUntil&>(*this) <
      static_cast<const class LtlUntil&>(other);
  }
  assert (false);
  return false;
}

void
LtlAtom::expand (unsigned state,
		 class LtlGraphNode& node,
		 class LtlGraph& graph) const
{
  class Ltl& f = negClone ();
  if (node.m_old.find (&f) == node.m_old.end ()) {
    node.m_atomic.insert (this);
    graph.expand (state, node);
  }
  else
    delete &node;
}

void
LtlConstant::expand (unsigned state,
		     class LtlGraphNode& node,
		     class LtlGraph& graph) const
{
  if (!m_true) {
    delete &node;
    return;
  }
  graph.expand (state, node);
}

void
LtlJunct::expand (unsigned state,
		  class LtlGraphNode& node,
		  class LtlGraph& graph) const
{
  if (m_con) {
    if (node.m_old.find (&m_left) == node.m_old.end ())
      node.m_new.insert (&m_left);
    if (node.m_old.find (&m_right) == node.m_old.end ())
      node.m_new.insert (&m_right);
  }
  else {
    std::set<const class Ltl*> new_set (node.m_new);
    if (node.m_old.find (&m_left) == node.m_old.end ())
      new_set.insert (&m_left);
    if (node.m_old.find (&m_right) == node.m_old.end ())
      node.m_new.insert (&m_right);
    graph.add (*new class LtlGraphNode (node.m_incoming, new_set, node.m_old,
					node.m_atomic, node.m_next));
  }
  graph.expand (state, node);
}

void
LtlIff::expand (unsigned state,
		class LtlGraphNode& node,
		class LtlGraph& graph) const
{
  std::set<const class Ltl*> new_set (node.m_new);
  class Ltl& neg_left = m_left.negClone ();
  class Ltl& neg_right = m_right.negClone ();
  if (node.m_old.find (&m_left) == node.m_old.end ())
    node.m_new.insert (&m_left);
  if (new_set.find (&neg_left) == new_set.end ())
    new_set.insert (&neg_left);
  if (m_iff) {
    if (node.m_old.find (&m_right) == node.m_old.end ())
      node.m_new.insert (&m_right);
    if (new_set.find (&neg_right) == new_set.end())
      new_set.insert (&neg_right);
  }
  else {
    if (node.m_old.find (&neg_right) == node.m_old.end ())
      node.m_new.insert (&neg_right);
    if (new_set.find (&m_right) == new_set.end ())
      new_set.insert (&m_right);
  }
  graph.add (*new class LtlGraphNode (node.m_incoming, new_set, node.m_old,
				      node.m_atomic, node.m_next));
  graph.expand (state, node);
}

void
LtlUntil::expand (unsigned state,
		  class LtlGraphNode& node,
		  class LtlGraph& graph) const
{
  if (m_release) {
    if (node.m_old.find (&m_right) == node.m_old.end ())
      node.m_new.insert (&m_right);
    std::set<const class Ltl*> new_set (node.m_new);
    if (node.m_old.find (&m_left) == node.m_old.end ())
      node.m_new.insert (&m_left);
    std::set<const class Ltl*> next_set (node.m_next);
    next_set.insert (this);
    graph.add (*new class LtlGraphNode (node.m_incoming, new_set, node.m_old,
					node.m_atomic, next_set));
  }
  else {
    std::set<const class Ltl*> new_set (node.m_new);
    if (node.m_old.find (&m_left) == node.m_old.end ())
      new_set.insert (&m_left);
    if (node.m_old.find (&m_right) == node.m_old.end ())
      node.m_new.insert (&m_right);
    std::set<const class Ltl*> next_set (node.m_next);
    next_set.insert(this);
    graph.add (*new class LtlGraphNode (node.m_incoming, new_set, node.m_old,
					node.m_atomic, next_set));
  }
  graph.expand (state, node);
}

void
LtlFuture::expand (unsigned state,
		   class LtlGraphNode& node,
		   class LtlGraph& graph) const
{
  switch (m_op) {
  case next:
    node.m_next.insert (&m_formula);
    graph.expand (state, node);
    return;
  case finally:
    {
      std::set<const class Ltl*> new_set (node.m_new);
      if (node.m_old.find (&m_formula) == node.m_old.end ())
	new_set.insert (&m_formula);
      graph.add (*new class LtlGraphNode (node.m_incoming,
					  new_set,
					  node.m_old,
					  node.m_atomic,
					  node.m_next));
    }
    break;
  case globally:
    if (node.m_old.find (&m_formula) == node.m_old.end ())
      node.m_new.insert (&m_formula);
    break;
#ifndef NDEBUG
  default:
    assert (false);
#endif // NDEBUG
  }
  node.m_next.insert (this);
  graph.expand (state, node);
}
