#!/bin/sh
#
# File:        liveness.sh
# Date:        13.03.2005
# Author:      Topi Pohjolainen
# Description: Liveness requirements generation from LTL into Promela. As such
#              the LTL syntax that can be given to the Spin's translator is
#              poorly limited for our needs, thus we need to augment the
#              translator's output ourselves.
#
TRANSLATE=ltl2ba

if [[ -z $1 ]]; then
    echo "Usage: ./liveness.sh <num_workers>"
    exit
else
    WORKERS=$1
fi

CONTROLLERS=1

#
# NOTES:
#
# 1. Run flag cannot differ from zero if the process stays in idle. This is 
#    builtin in the model.
#
# 2. Spin's translator cannot handle array indexing array[] as it is only
#    capable of handling '[' and ']' as LTL operators. We give array
#    reference 'array[identifier]' as 'array__identifier___' and use 'sed'
#    to recover the array representation once the translator has done its job.
#
# 3. Spin has built-in "enabled(p)" routine for determining if process 'p' is
#    blocked or not. But like the array[] construct, enabled() is not 
#    supported either. Therefore we turn to sed..., _LPAR_ = '(', _RPAR_ = ')'
#
# 4. Referring to the states of native Spin processes is done by
#    'ProcessBody[pid]@label'. However, '@'-character is not accepted by the
#    translator, and with sed _AT_ = '@'.
#
# 5. The translator runs easily out of memory if literals have long names 
#    (~30 characters), therefore we use 'sed' again to truncate the names
#    for the translator and expand them again once the translator has done
#    its job.
#
C_PID=1
SH_PID=2
ST_PID=3
FIRST_WORKER_PID=4

################################################################################
# Preconditions (fairness) for liveness claims
################################################################################
#
# Fairness constraints for controllers (strongly fair)
#     for each controller c in model: 
#         []<>enabled(c) -> []<>controllersSelected[c]
#
F="![] (("
for (( I=0; $CONTROLLERS - $I; I = I + 1)); do
    F="$F ([]<>(cE && !cI) -> []<>cS)"

    if (( $CONTROLLERS > ($I + 1))); then F="$F &&"; fi
done

#
# Fairness constraints for the shutdown scheduler (strongly fair)
#     []<>enabled(shutdown_sched) -> []<>startupSchedulerSelected
#
F="$F && ([]<>(sHE && !sHI) -> []<>sHS)"

#
# Fairness constraints for the startup scheduler (strongly fair)
#     <>[]enabled(startup_sched) -> []<>startupSchedulerSelected
#
F="$F && ([]<>(sTE && !sTI) -> []<>sTS)"

#
# Fairness constraints for workers' transitions (weakly fair)
#     for each worker w in model:
#         <>[]enabled(w) -> []<>workersSelected[w]
#
for (( I=0; $WORKERS - $I; I = I + 1)); do
    TMP=`expr $FIRST_WORKER_PID + $I`

    F="$F && (<>[](enabled_LPAR_${TMP}_RPAR_ || worker__${TMP}___wA) -> "
    F="$F []<>workersSelected__${I}___)"
done

F="$F )"

################################################################################
# Liveness claims
################################################################################
#
# Once a controller leaves its idle state for "create/destroy cycle", it must 
# eventually get back to the idle state.
#
# for each controller c in model: 
#     []<>controller[w]@CONTROLLER_IDLE
#
F="$F -> ([]<>cI"

#
# Once a scheduler leaves its idle state, it must eventually get back to
# the idle state.
#
# []<>startupScheduler[3]@STARTUP_SCHEDULER_IDLE
# []<>shutdownScheduler[2]@SHUTDOWN_SCHEDULER_IDLE
#
F="$F && []<>sTI"
F="$F && []<>sHI"
F="$F ))"

#
# Disabling '-a, disable trick in (A)ccepting conditions' and
#           '-p, disable a-(P)osteriori simplification'
# makes the translator to work with larger conditions
#
$TRANSLATE -f "$F" | sed                                              \
               -e 's/___/]/g'                                         \
               -e 's/__/[/g'                                          \
               -e 's/_LPAR_/(/g'                                      \
               -e 's/_RPAR_/)/g'                                      \
               -e 's/sHI/shutdownScheduler@SHUTDOWN_SCHEDULER_IDLE/g' \
               -e 's/sHE/enabled(2)/g'                                \
               -e 's/sHS/shutdownSchedulerSelected/g'                 \
               -e 's/sTE/enabled(3)/g'                                \
               -e 's/sTS/startupSchedulerSelected/g'                  \
               -e 's/sTI/startupScheduler@STARTUP_SCHEDULER_IDLE/g'   \
               -e 's/cI/controller@CONTROLLER_IDLE/g'                 \
               -e 's/cS/controllersSelected[0]/g'                     \
               -e 's/cE/enabled(1)/g'                                 \
               -e 's/wA/@WORKER_CONTENT_ACQUIRED/g'                   \
               -e 's/wI/@WORKER_IDLE/g'

# EOF
