#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "l2s.h"

/*------------------------------------------------------------------------*/
/* printpod application                                                   */
/*------------------------------------------------------------------------*/

typedef struct L2s_Printpod {
  int id;
  FILE *in;
  int closein;
  FILE *out;
  int closeout;
  FILE *err;
  int closeerr;
} L2s_Printpod;

/*------------------------------------------------------------------------*/

L2s_Printpod *L2s_Printpod_new() {
  L2s_Printpod *app;

  app = (L2s_Printpod *) malloc (sizeof(L2s_Printpod));
  assert(app);
  app->id = -1;
  app->in = stdin;
  app->closein = 0;
  app->out = stdout;
  app->closeout = 0;
  app->err = stderr;
  app->closeerr = 0;
  return app;
}

/*------------------------------------------------------------------------*/

void L2s_Printpod_delete(L2s_Printpod *app) {
  assert(app);
  if (app->closein)
    fclose(app->in);
  if (app->closeout)
    fclose(app->out);
  if (app->closeerr)
    fclose(app->err);
  free(app);
}

/*------------------------------------------------------------------------*/

#define L2S_PRINTPOD_USAGE \
"printpod: Prints past operator depth of an LTL formula to stdout. Syntax of\n"\
"          formulae is currently restricted:\n"\
"          - no brackets around main formula,\n"\
"          - brackets around each subformula,\n"\
"          - atoms extend to the next closing bracket\n"\
"\n"\
"Usage: printpod <ifile>\n"\
"\n"\
"Where:  <ifile>       is the file from which the LTL Formula to be translated\n"\
"                      is read in.\n"

static void
L2s_Printpod_usage (L2s_Printpod *app)
{
  assert(app);
  fprintf(app->err, L2S_PRINTPOD_USAGE);
}

/*------------------------------------------------------------------------*/

int
L2s_Printpod_main (int argc, char *argv[])
{
  L2s_Printpod *app;
  int i;
  char c;
  L2s_String *str;
  L2s_SimpleParser *pars;
  L2s_PLTLBFormula *phi;
  L2s_PLTLBNode *node;

  app = L2s_Printpod_new();
  i = 1;
  if (argc <= i) {
    L2s_Printpod_usage(app);
    L2s_Printpod_delete(app);
    exit(1);
  } else {
    app->in = fopen(argv[i], "r");
    if (!app->in) {
      fprintf(app->err, "Error: cannot open %s for reading\n", argv[i]);
      L2s_Printpod_delete(app);
      exit(1);
    }
    app->closein = 1;
    str = L2s_String_new();
    while ((c = fgetc(app->in)) != EOF) {
      if (c != ' ' && c != '\n' && c != '\t') {
	L2s_String_appendC(str, c);
      }
    }
    fclose(app->in);
    app->closein = 0;
  }

  pars = L2s_SimpleParser_new();
  L2s_SimpleParser_setLog(pars, app->err);
  L2s_SimpleParser_setInput(pars, str);
  L2s_SimpleParser_parse(pars);
  phi = L2s_SimpleParser_getPhi(pars);
  if (phi) {
    node = L2s_PLTLBFormula_getTop(phi);
    assert(node);
    fprintf(app->out, "%u\n", L2s_PLTLBNode_pod(node));
    L2s_SimpleParser_delete(pars);
    L2s_Printpod_delete(app);
    exit (0);
  } else {
    L2s_Printpod_delete(app);
    exit (1);
  }

  assert (0);
  return 0;
}

/*------------------------------------------------------------------------*/

int main (int argc, char *argv[]) {
  return L2s_Printpod_main(argc, argv);
}

/*------------------------------------------------------------------------*/
