#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "l2s.h"

/*------------------------------------------------------------------------*/
/* very simple parser for LTL formulae

   format
   - no brackets around formula,
   - brackets around each subformula,
   - atoms extend to the next closing bracket                             */
/*------------------------------------------------------------------------*/

struct L2s_SimpleParser
{
  L2s_String *input;
  int i;
  L2s_PLTLBFormula *phi;
  FILE *log; /* error msgs */
};

/*------------------------------------------------------------------------*/

L2s_SimpleParser *
L2s_SimpleParser_new()
{
  L2s_SimpleParser *pars;

  pars = (L2s_SimpleParser *) malloc (sizeof (L2s_SimpleParser));
  assert (pars);
  pars->input = NULL;
  pars->i = 0;
  pars->phi = NULL;
  pars->log = stderr;
  return pars;
}

/*------------------------------------------------------------------------*/

void
L2s_SimpleParser_setLog(L2s_SimpleParser *pars, FILE *log)
{
  assert(pars);
  assert(log);
  pars->log = log;
}

/*------------------------------------------------------------------------*/

void
L2s_SimpleParser_setInput(L2s_SimpleParser *pars, L2s_String *input)
{
  assert(pars);
  assert(input);
  pars->input = input;
}

/*------------------------------------------------------------------------*/

L2s_PLTLBFormula *
L2s_SimpleParser_getPhi(L2s_SimpleParser *pars)
{
  assert(pars);
  return pars->phi;
}

/*------------------------------------------------------------------------*/

static void
L2s_SimpleParser_error(L2s_SimpleParser *pars, char *error) {
  L2s_String *msg;
  int i, j;

  msg = L2s_String_new();
  L2s_String_append(msg, "Syntax error in formula, ");
  L2s_String_append(msg, error);
  L2s_String_append(msg, ": ");
  i = L2s_String_len(msg) + pars->i;
  L2s_String_appendS(msg, pars->input);
  L2s_String_fprint(msg, pars->log);
  fprintf(pars->log, "\n");
  L2s_String_delete(msg);
  msg = L2s_String_new();
  for (j = 0; j < i; j++) {
    L2s_String_append(msg, " ");
  }
  L2s_String_append(msg, "^");
  L2s_String_fprint(msg, pars->log);
  fprintf(pars->log, "\n");
  L2s_String_delete(msg);
}

/*------------------------------------------------------------------------*/

static int
L2s_SimpleParser_check(L2s_SimpleParser *pars, char expected) {
  char c, s[11];

  assert(pars);
  if (pars->i < L2s_String_len(pars->input)) {
    c = L2s_String_charAt(pars->input, pars->i);
    if (c == expected)
      return 1;
  }
  sprintf(s, "expected '%c'", expected);
  L2s_SimpleParser_error(pars, s);
  return 0;
}

/*------------------------------------------------------------------------*/

static int
L2s_SimpleParser_prefixIsUnary(L2s_SimpleParser *pars)
{
  char c;

  assert(pars);
  c = L2s_String_charAt(pars->input, pars->i);
  return ((c == '!') ||
	  (c == 'X') ||
	  (c == 'F') ||
	  (c == 'G') ||
	  (c == 'Y') ||
	  (c == 'Z') ||
	  (c == 'O') ||
	  (c == 'H'));
}

/*------------------------------------------------------------------------*/

static int
L2s_SimpleParser_prefixIsBinary(L2s_SimpleParser *pars)
{
  char c, *s;
  int i, res;
  L2s_String *pre;

  assert(pars);
  i = pars->i;
  res = 0;
  c = L2s_String_charAt(pars->input, i);
  res = ((c == '|') ||
	 (c == '&') ||
	 (c == 'U') ||
	 (c == 'V') ||
	 (c == 'S') ||
	 (c == 'T'));
  if (!res && i+1 < L2s_String_len(pars->input)) {
    pre = L2s_String_substring(pars->input, i, i+1);
    s = L2s_String_toCharPtr(pre);
    res = !strcmp(s, "->");
    free(s);
    L2s_String_delete(pre);
  }
  if (!res && i+2 < L2s_String_len(pars->input)) {
    pre = L2s_String_substring(pars->input, i, i+2);
    s = L2s_String_toCharPtr(pre);
    res = !strcmp(s, "<->");
    free(s);
    L2s_String_delete(pre);
  }
  return res;
}

/*------------------------------------------------------------------------*/

static int
L2s_SimpleParser_prefixIsOpenBracket(L2s_SimpleParser *pars)
{
  char c;

  assert(pars);
  c = L2s_String_charAt(pars->input, pars->i);
  return (c == '(');
}

/*------------------------------------------------------------------------*/

static L2s_PLTLBOperator
L2s_SimpleParser_readUnary(L2s_SimpleParser *pars)
{
  char c;
  L2s_PLTLBOperator op;

  assert(pars);
  c = L2s_String_charAt(pars->input, pars->i);
  switch (c) {
  case '!':
    op = NOT;
    break;
  case 'X':
    op = X;
    break;
  case 'F':
    op = F;
    break;
  case 'G':
    op = G;
    break;
  case 'Y':
    op = Y;
    break;
  case 'Z':
    op = Z;
    break;
  case 'O':
    op = O;
    break;
  case 'H':
    op = H;
    break;
  default:
    assert(0);
    break;
  }
  return op;
}

/*------------------------------------------------------------------------*/

static L2s_PLTLBOperator
L2s_SimpleParser_readBinary(L2s_SimpleParser *pars)
{
  char c;
  L2s_PLTLBOperator op;

  assert(pars);
  c = L2s_String_charAt(pars->input, pars->i);
  switch (c) {
  case '|':
    op = OR;
    break;
  case '&':
    op = AND;
    break;
  case '-':
    op = IMPLICATION;
    break;
  case '<':
    op = EQUIVALENCE;
    break;
  case 'U':
    op = U;
    break;
  case 'V':
    op = V;
    break;
  case 'S':
    op = S;
    break;
  case 'T':
    op = T;
    break;
  default:
    assert(0);
    break;
  }
  return op;
}

/*------------------------------------------------------------------------*/

static L2s_String *
L2s_SimpleParser_readAtom(L2s_SimpleParser *pars)
{
  int j;
  L2s_String *atom;

  assert(pars);
  assert((pars->i < L2s_String_len(pars->input)) &&
	 (L2s_String_charAt(pars->input, pars->i) != ')') &&
	 (L2s_String_charAt(pars->input, pars->i) != '('));
  j = pars->i;
  while (j < L2s_String_len(pars->input) &&
	 L2s_String_charAt(pars->input, j) != ')' &&
	 L2s_String_charAt(pars->input, j) != '(')
    j++;
  atom = L2s_String_substring(pars->input, pars->i, j-1);
  pars->i = j;
  return atom;
}

/*------------------------------------------------------------------------*/

static L2s_PLTLBNode *
L2s_SimpleParser_parseFormula(L2s_SimpleParser *pars)
{
  L2s_PLTLBNode *phi, *psi1, *psi2;
  L2s_PLTLBOperator op;
  L2s_String *atom;

  assert(pars);
  if (pars->i >= L2s_String_len(pars->input)) {
    L2s_SimpleParser_error(pars, "expected formula");
    phi = NULL;
  } else if (L2s_SimpleParser_prefixIsUnary(pars)) {
    op = L2s_SimpleParser_readUnary(pars);
    pars->i++;
    if (!L2s_SimpleParser_check(pars, '('))
      return NULL;
    pars->i++;
    psi1 = L2s_SimpleParser_parseFormula(pars);
    if (!psi1)
      return NULL;
    if (!L2s_SimpleParser_check(pars, ')'))
      return NULL;
    pars->i++;
    phi = L2s_PLTLBNode_new();
    L2s_PLTLBNode_setOp(phi, op);
    L2s_PLTLBNode_setLeft(phi, psi1);
  } else if (L2s_SimpleParser_prefixIsOpenBracket(pars)) {
    pars->i++;
    psi1 = L2s_SimpleParser_parseFormula(pars);
    if (!psi1)
      return NULL;
    if (!L2s_SimpleParser_check(pars, ')'))
      return NULL;
    pars->i++;
    if (pars->i >= L2s_String_len(pars->input) || !L2s_SimpleParser_prefixIsBinary(pars)) {
      L2s_SimpleParser_error(pars, "expected binary operator");
      return NULL;
    }
    op = L2s_SimpleParser_readBinary(pars);
    pars->i++;
    if (op == IMPLICATION)
      pars->i++;
    if (op == EQUIVALENCE)
      pars->i+=2;
    if (!L2s_SimpleParser_check(pars, '('))
      return NULL;
    pars->i++;
    psi2 = L2s_SimpleParser_parseFormula(pars);
    if (!psi2)
      return NULL;
    if (!L2s_SimpleParser_check(pars, ')'))
      return NULL;
    pars->i++;
    phi = L2s_PLTLBNode_new();
    L2s_PLTLBNode_setOp(phi, op);
    L2s_PLTLBNode_setLeft(phi, psi1);
    L2s_PLTLBNode_setRight(phi, psi2);
  } else {
    if (pars->i >= L2s_String_len(pars->input) ||
	(L2s_String_charAt(pars->input, pars->i) == ')') ||
	(L2s_String_charAt(pars->input, pars->i) == '(')) {
      L2s_SimpleParser_error(pars, "expected atom");
      return NULL;
    }
    atom = L2s_SimpleParser_readAtom(pars);
    phi = L2s_PLTLBNode_new();
    L2s_PLTLBNode_setOp(phi, ATOM);
    L2s_PLTLBNode_setAtom(phi, atom);
  }
  return phi;
}

/*------------------------------------------------------------------------*/

void
L2s_SimpleParser_parse(L2s_SimpleParser *pars)
{
  L2s_PLTLBNode *top;

  assert(pars);
  assert(pars->input);
  assert(pars->i == 0);
  assert(pars->phi == NULL);
  top = L2s_SimpleParser_parseFormula(pars);
  if (top && pars->i != L2s_String_len(pars->input)) {
    L2s_SimpleParser_error(pars, "expected end of formula");
  } else if (top) {
    pars->phi = L2s_PLTLBFormula_new();
    L2s_PLTLBFormula_setTop(pars->phi, top);
  }
}

/*------------------------------------------------------------------------*/

void L2s_SimpleParser_delete(L2s_SimpleParser *pars) {
  assert(pars);
  if (pars->input)
    L2s_String_delete(pars->input);
  if (pars->phi)
    L2s_PLTLBFormula_delete(pars->phi);
  free(pars);
}

/*------------------------------------------------------------------------*/
/* unit tests                                                             */
/*------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
  parse formulae
*/
static int
L2s_test_SimpleParser_parse (FILE * log, char *s) {
  L2s_SimpleParser *pars;
  L2s_String *str, *str2;
  L2s_PLTLBFormula *phi;

  pars = L2s_SimpleParser_new();
  L2s_SimpleParser_setLog(pars, log);
  str = L2s_String_new();
  L2s_String_append(str, s);
  L2s_SimpleParser_setInput(pars, str);
  L2s_SimpleParser_parse(pars);
  phi = L2s_SimpleParser_getPhi(pars);
  if (phi) {
    str2 = L2s_PLTLBFormula_toString(phi);
    L2s_String_fprint(str2, log);
    fprintf(log, "\n");
    L2s_String_delete(str2);
  }
  L2s_SimpleParser_delete(pars);
  return 1;
}

/*--------------------------------------------------------------------------
  parse p
*/
int
L2s_test_SimpleParser_parse_p (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "p");
}

/*--------------------------------------------------------------------------
  parse !p
*/
int
L2s_test_SimpleParser_parse_NOTp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "!(p)");
}

/*--------------------------------------------------------------------------
  parse p | q
*/
int
L2s_test_SimpleParser_parse_pORq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)|(q)");
}

/*--------------------------------------------------------------------------
  parse p & q
*/
int
L2s_test_SimpleParser_parse_pANDq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)&(q)");
}

/*--------------------------------------------------------------------------
  parse p -> q
*/
int
L2s_test_SimpleParser_parse_pIMPLICATIONq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)->(q)");
}

/*--------------------------------------------------------------------------
  parse p <-> q
*/
int
L2s_test_SimpleParser_parse_pEQUIVALENCEq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)<->(q)");
}

/*--------------------------------------------------------------------------
  parse X p
*/
int
L2s_test_SimpleParser_parse_Xp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "X(p)");
}

/*--------------------------------------------------------------------------
  parse p U q
*/
int
L2s_test_SimpleParser_parse_pUq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)U(q)");
}

/*--------------------------------------------------------------------------
  parse p V q
*/
int
L2s_test_SimpleParser_parse_pVq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)V(q)");
}

/*--------------------------------------------------------------------------
  parse F p
*/
int
L2s_test_SimpleParser_parse_Fp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "F(p)");
}

/*--------------------------------------------------------------------------
  parse G p
*/
int
L2s_test_SimpleParser_parse_Gp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "G(p)");
}

/*--------------------------------------------------------------------------
  parse Y p
*/
int
L2s_test_SimpleParser_parse_Yp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "Y(p)");
}

/*--------------------------------------------------------------------------
  parse Z p
*/
int
L2s_test_SimpleParser_parse_Zp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "Z(p)");
}

/*--------------------------------------------------------------------------
  parse p S q
*/
int
L2s_test_SimpleParser_parse_pSq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)S(q)");
}

/*--------------------------------------------------------------------------
  parse p T q
*/
int
L2s_test_SimpleParser_parse_pTq (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "(p)T(q)");
}

/*--------------------------------------------------------------------------
  parse O p
*/
int
L2s_test_SimpleParser_parse_Op (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "O(p)");
}

/*--------------------------------------------------------------------------
  parse H p
*/
int
L2s_test_SimpleParser_parse_Hp (FILE * log)
{
  return L2s_test_SimpleParser_parse(log, "H(p)");
}

/*------------------------------------------------------------------------*/
