#include <assert.h>
#include <stdlib.h>
#include "l2s.h"

/*------------------------------------------------------------------------*/
/* PLTLB formula node                                                     */
/*------------------------------------------------------------------------*/

struct L2s_PLTLBNode
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *left;
  L2s_PLTLBNode *right;
  L2s_String *atom;
};

/*------------------------------------------------------------------------*/

L2s_PLTLBNode *
L2s_PLTLBNode_new ()
{
  L2s_PLTLBNode *node;

  node = (L2s_PLTLBNode *) malloc (sizeof (L2s_PLTLBNode));
  assert (node);
  node->left = NULL;
  node->right = NULL;
  node->atom = NULL;
  return node;
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBNode_setOp (L2s_PLTLBNode * node, L2s_PLTLBOperator op)
{
  assert (node);
  node->op = op;
}

/*------------------------------------------------------------------------*/

L2s_PLTLBOperator
L2s_PLTLBNode_getOp (L2s_PLTLBNode * node)
{
  assert (node);
  return node->op;
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBNode_setLeft (L2s_PLTLBNode * node, L2s_PLTLBNode * left)
{
  assert (node);
  assert (left);
  node->left = left;
}

/*------------------------------------------------------------------------*/

L2s_PLTLBNode *
L2s_PLTLBNode_getLeft (L2s_PLTLBNode * node)
{
  assert (node);
  return node->left;
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBNode_setRight (L2s_PLTLBNode * node, L2s_PLTLBNode * right)
{
  assert (node);
  assert (right);
  node->right = right;
}

/*------------------------------------------------------------------------*/

L2s_PLTLBNode *
L2s_PLTLBNode_getRight (L2s_PLTLBNode * node)
{
  assert (node);
  return node->right;
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBNode_setAtom (L2s_PLTLBNode * node, L2s_String * atom)
{
  assert (node);
  assert (atom);
  node->atom = atom;
}

/*------------------------------------------------------------------------*/

L2s_String *
L2s_PLTLBNode_getAtom (L2s_PLTLBNode * node)
{
  assert (node);
  return node->atom;
}

/*------------------------------------------------------------------------*/

int
L2s_PLTLBNode_pod (L2s_PLTLBNode * node)
{
  int res;

  assert (node);
  switch (node->op)
    {
    case ATOM:
    case TRUE:
    case FALSE:
      res = 0;
      break;
    case NOT:
    case X:
    case F:
    case G:
      res = L2s_PLTLBNode_pod (node->left);
      break;
    case OR:
    case AND:
    case IMPLICATION:
    case EQUIVALENCE:
    case U:
    case V:
      res = max (L2s_PLTLBNode_pod (node->left),
		 L2s_PLTLBNode_pod (node->right));
      break;
    case Y:
    case Z:
    case O:
    case H:
      res = 1 + L2s_PLTLBNode_pod (node->left);
      break;
    case S:
    case T:
      res = 1 + max (L2s_PLTLBNode_pod (node->left),
		     L2s_PLTLBNode_pod (node->right));
      break;
    default:
      assert (0);
      break;
    }
  return res;
}

/*------------------------------------------------------------------------*/

L2s_String *
L2s_PLTLBNode_toString (L2s_PLTLBNode * phi)
{
  L2s_PLTLBOperator op;
  L2s_String *strphi, *strtmp;

  assert (phi);
  op = phi->op;
  if (L2s_PLTLBOperator_isNullary(op)) {
    if (op == TRUE) {
      strphi = L2s_String_new();
      L2s_String_append(strphi, "1");
    } else if (op == FALSE) {
      strphi = L2s_String_new();
      L2s_String_append(strphi, "0");
    } else {
      assert(op == ATOM);
      assert(phi->atom);
      strphi = L2s_String_clone(phi->atom);
    }
  } else if (L2s_PLTLBOperator_isUnary(op)) {
    assert(phi->left);
    strphi = L2s_PLTLBOperator_toString(op);
    L2s_String_append(strphi, "(");
    strtmp = L2s_PLTLBNode_toString(phi->left);
    L2s_String_appendS(strphi, strtmp);
    L2s_String_delete(strtmp);
    L2s_String_append(strphi, ")");
  } else {
    assert(L2s_PLTLBOperator_isBinary(op));
    assert(phi->left);
    assert(phi->right);
    strphi =  L2s_String_new();
    L2s_String_append(strphi, "(");
    strtmp = L2s_PLTLBNode_toString(phi->left);
    L2s_String_appendS(strphi, strtmp);
    L2s_String_delete(strtmp);
    L2s_String_append(strphi, ")");
    strtmp = L2s_PLTLBOperator_toString(op);
    L2s_String_appendS(strphi, strtmp);
    L2s_String_delete(strtmp);
    L2s_String_append(strphi, "(");
    strtmp = L2s_PLTLBNode_toString(phi->right);
    L2s_String_appendS(strphi, strtmp);
    L2s_String_delete(strtmp);
    L2s_String_append(strphi, ")");
  }
  return strphi;
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBNode_delete (L2s_PLTLBNode * node)
{
  assert (node);
  if (L2s_PLTLBOperator_isNullary (node->op))
    {
      assert (!node->left);
      assert (!node->right);
      if (node->op == ATOM)
	{
	  assert (node->atom);
	  L2s_String_delete (node->atom);
	}
      else
	{
	  assert (!node->atom);
	}
    }
  else if (L2s_PLTLBOperator_isUnary (node->op))
    {
      assert (node->left);
      assert (!node->right);
      assert (!node->atom);
    }
  else
    {
      assert (L2s_PLTLBOperator_isBinary (node->op));
      assert (node->left);
      assert (node->right);
      assert (!node->atom);
    }
  free (node);
}

/*------------------------------------------------------------------------*/
