#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "l2s.h"

/*------------------------------------------------------------------------*/
/* ltl2tsmv application                                                   */
/*------------------------------------------------------------------------*/

typedef struct L2s_Ltl2tsmv {
  int id;
  FILE *in;
  int closein;
  FILE *out;
  int closeout;
  FILE *err;
  int closeerr;
} L2s_Ltl2tsmv;

/*------------------------------------------------------------------------*/

L2s_Ltl2tsmv *L2s_Ltl2tsmv_new() {
  L2s_Ltl2tsmv *app;

  app = (L2s_Ltl2tsmv *) malloc (sizeof(L2s_Ltl2tsmv));
  assert(app);
  app->id = -1;
  app->in = stdin;
  app->closein = 0;
  app->out = stdout;
  app->closeout = 0;
  app->err = stderr;
  app->closeerr = 0;
  return app;
}

/*------------------------------------------------------------------------*/

void L2s_Ltl2tsmv_delete(L2s_Ltl2tsmv *app) {
  assert(app);
  if (app->closein)
    fclose(app->in);
  if (app->closeout)
    fclose(app->out);
  if (app->closeerr)
    fclose(app->err);
  free(app);
}

/*------------------------------------------------------------------------*/

#define L2S_LTL2TSMV_USAGE \
"ltl2tsmv: Converts an LTL formula to a fragment of an SMV program.\n"\
"          Similar to ltl2smv from NuSMV distribution but generates\n"\
"          automaton that is tight also for past time LTL. Usage is\n"\
"          similar but syntax of formulae is currently restricted:\n"\
"          - no brackets around main formula,\n"\
"          - brackets around each subformula,\n"\
"          - atoms extend to the next closing bracket\n"\
"\n"\
"Usage: ltl2tsmv # <ifile> [<ofile>]\n"\
"\n"\
"Where:  #             is the counter to be used in _LTL#_SPECF_N_.\n"\
"        <ifile>       is the file from which the LTL Formula to be translated\n"\
"                      is read in.\n"\
"        <ofile>       is the file in which the SMV code corresponding to the\n"\
"                      tableau of LTL Formula is written in. If not specified\n"\
"                      then stdout is used.\n"

static void
L2s_Ltl2tsmv_usage (L2s_Ltl2tsmv *app)
{
  assert(app);
  fprintf(app->err, L2S_LTL2TSMV_USAGE);
}

/*------------------------------------------------------------------------*/

int
L2s_Ltl2tsmv_main (int argc, char *argv[])
{
  L2s_Ltl2tsmv *app;
  int i, n, id;
  char c, *s;
  L2s_String *str, *pre, *mod, *res;
  L2s_SimpleParser *pars;
  L2s_PLTLBFormula *phi;
  L2s_Encoder *enc; 

  app = L2s_Ltl2tsmv_new();
  i = 1;
  if (argc <= i) {
    L2s_Ltl2tsmv_usage(app);
    L2s_Ltl2tsmv_delete(app);
    exit(1);
  } else {
    n = sscanf(argv[i], "%d", &id);
    if (!n) {
      fprintf(app->err, "Error: counter must be number larger or equal to 0\n");
      L2s_Ltl2tsmv_delete(app);
      exit(1);
    } else if (id < 0) {
      fprintf(app->err, "Error: counter must be larger or equal to 0\n");
      L2s_Ltl2tsmv_delete(app);
      exit(1);
    }
    app->id = id;
  }
  i++;
  if (argc <= i) {
    L2s_Ltl2tsmv_usage(app);
    L2s_Ltl2tsmv_delete(app);
    exit(1);
  } else {
    app->in = fopen(argv[i], "r");
    if (!app->in) {
      fprintf(app->err, "Error: cannot open %s for reading\n", argv[i]);
      L2s_Ltl2tsmv_delete(app);
      exit(1);
    }
    app->closein = 1;
    str = L2s_String_new();
    while ((c = fgetc(app->in)) != EOF) {
      if (c != ' ' && c != '\n' && c != '\t') {
	L2s_String_appendC(str, c);
      }
    }
    fclose(app->in);
    app->closein = 0;
  }
  i++;
  if (i < argc) {
    app->out = fopen(argv[i], "w");
    if (!app->out) {
      fprintf(app->err, "Error: cannot open %s for writing\n", argv[i]);
      L2s_Ltl2tsmv_delete(app);
      exit(1);
    }
    app->closeout = 1;
  }

  pars = L2s_SimpleParser_new();
  L2s_SimpleParser_setLog(pars, app->err);
  L2s_SimpleParser_setInput(pars, str);
  L2s_SimpleParser_parse(pars);
  phi = L2s_SimpleParser_getPhi(pars);
  if (phi) {
    enc = L2s_Encoder_new();
    L2s_Encoder_setEncoding(enc, CGHTight);
    s = (char *) malloc (sizeof(char) * (10 + 10)); /* 32 bit int */
    sprintf(s, "ltl_spec_%d", app->id);
    mod = L2s_String_new();
    L2s_String_append(mod, s);
    L2s_Encoder_setModulename(enc, mod);
    free(s);
    s = (char *) malloc (sizeof(char) * (12 + 10)); /* 32 bit int */
    sprintf(s, "LTL_%d_SPECF_", app->id);
    pre = L2s_String_new();
    L2s_String_append(pre, s);
    L2s_Encoder_setVarprefix(enc, pre);
    free(s);
    L2s_Encoder_setPhi(enc, phi);
    L2s_Encoder_encode(enc);
    res = L2s_Encoder_getRes(enc);
    L2s_String_fprint(res, app->out);
    L2s_String_delete(res);
    L2s_Encoder_delete(enc);
    L2s_SimpleParser_delete(pars);
    L2s_Ltl2tsmv_delete(app);
    exit (0);
  } else {
    L2s_Ltl2tsmv_delete(app);
    exit (1);
  }

  assert (0);
  return 0;
}

/*------------------------------------------------------------------------*/

int main (int argc, char *argv[]) {
  return L2s_Ltl2tsmv_main(argc, argv);
}

/*------------------------------------------------------------------------*/
