#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include "l2s.h"

/*------------------------------------------------------------------------*/
/* ltl2l2s application                                                   */
/*------------------------------------------------------------------------*/

typedef struct L2s_Ltl2l2s {
  int maxunroll;
  FILE *in;
  int closein;
  FILE *out;
  int closeout;
  FILE *err;
  int closeerr;
} L2s_Ltl2l2s;

/*------------------------------------------------------------------------*/

L2s_Ltl2l2s *L2s_Ltl2l2s_new() {
  L2s_Ltl2l2s *app;

  app = (L2s_Ltl2l2s *) malloc (sizeof(L2s_Ltl2l2s));
  assert(app);
  app->maxunroll = INT_MAX;
  app->in = stdin;
  app->closein = 0;
  app->out = stdout;
  app->closeout = 0;
  app->err = stderr;
  app->closeerr = 0;
  return app;
}

/*------------------------------------------------------------------------*/

void L2s_Ltl2l2s_delete(L2s_Ltl2l2s *app) {
  assert(app);
  if (app->closein)
    fclose(app->in);
  if (app->closeout)
    fclose(app->out);
  if (app->closeerr)
    fclose(app->err);
  free(app);
}

/*------------------------------------------------------------------------*/

#define L2S_LTL2L2S_USAGE \
"ltl2l2s: Converts an LTL formula to a fragment of an SMV program. To be\n"\
"         used only with corresponding l2s translation. Syntax of formulae\n"\
"         is currently restricted:\n"\
"          - no brackets around main formula,\n"\
"          - brackets around each subformula,\n"\
"          - atoms extend to the next closing bracket\n"\
"\n"\
"Usage: ltl2l2s [--tight|--no-tight|--maxunroll k] <ifile> [<ofile>]\n"\
"\n"\
"Where:  --tight         produces tight encoding (shortest counterexamples)\n"\
"        --no-tight      produces non-tight encoding\n"\
"        --maxunroll k   produces encoding with at most k unrollings\n"\
"                        (--maxunroll 0 is equivalent to --no-tight)\n"\
"        <ifile>         is the file from which the LTL Formula to be\n"\
"                        translated is read in.\n"\
"        <ofile>         is the file in which the SMV code corresponding to\n"\
"                        the tableau of LTL Formula is written in. If not\n"\
"                        specified then stdout is used.\n"

static void
L2s_Ltl2l2s_usage (L2s_Ltl2l2s *app)
{
  assert(app);
  fprintf(app->err, L2S_LTL2L2S_USAGE);
}

/*------------------------------------------------------------------------*/

int
L2s_Ltl2l2s_main (int argc, char *argv[])
{
  L2s_Ltl2l2s *app;
  int i;
  char c;
  L2s_String *str, *res;
  L2s_SimpleParser *pars;
  L2s_PLTLBFormula *phi;
  L2s_Encoder *enc; 

  app = L2s_Ltl2l2s_new();
  i = 1;
  if (argc <= i) {
    L2s_Ltl2l2s_usage(app);
    L2s_Ltl2l2s_delete(app);
    exit(1);
  } else {
    if (!strcmp(argv[i], "--tight")) {
      app->maxunroll = INT_MAX;
      i++;
    } else if (!strcmp(argv[i], "--no-tight")) {
      app->maxunroll = 0;
      i++;
    } else if (!strcmp(argv[i], "--maxunroll")) {
      i++;
      if (argc <= i) {
	L2s_Ltl2l2s_usage(app);
	L2s_Ltl2l2s_delete(app);
	exit(1);
      } else if ((sscanf(argv[i], "%d", &(app->maxunroll)) != 1) || (app->maxunroll < 0)) {
	fprintf(app->err, "Error: --maxunroll requires integer argument >= 0\n");
	L2s_Ltl2l2s_delete(app);
	exit(1);
      } else {
	i++;
      }
    }
  }
  if (argc <= i) {
    L2s_Ltl2l2s_usage(app);
    L2s_Ltl2l2s_delete(app);
    exit(1);
  } else {
    app->in = fopen(argv[i], "r");
    if (!app->in) {
      fprintf(app->err, "Error: cannot open %s for reading\n", argv[i]);
      L2s_Ltl2l2s_delete(app);
      exit(1);
    }
    app->closein = 1;
    str = L2s_String_new();
    while ((c = fgetc(app->in)) != EOF) {
      if (c != ' ' && c != '\n' && c != '\t') {
	L2s_String_appendC(str, c);
      }
    }
    fclose(app->in);
    app->closein = 0;
  }
  i++;
  if (i < argc) {
    app->out = fopen(argv[i], "w");
    if (!app->out) {
      fprintf(app->err, "Error: cannot open %s for writing\n", argv[i]);
      L2s_Ltl2l2s_delete(app);
      exit(1);
    }
    app->closeout = 1;
  }

  pars = L2s_SimpleParser_new();
  L2s_SimpleParser_setLog(pars, app->err);
  L2s_SimpleParser_setInput(pars, str);
  L2s_SimpleParser_parse(pars);
  phi = L2s_SimpleParser_getPhi(pars);
  if (phi) {
    enc = L2s_Encoder_new();
    L2s_Encoder_setEncoding(enc, BMCNdet);
    L2s_Encoder_setPhi(enc, phi);
    L2s_Encoder_setMaxunroll(enc, app->maxunroll);
    L2s_Encoder_encode(enc);
    res = L2s_Encoder_getRes(enc);
    L2s_String_fprint(res, app->out);
    L2s_String_delete(res);
    L2s_Encoder_delete(enc);
    L2s_SimpleParser_delete(pars);
    L2s_Ltl2l2s_delete(app);
    exit (0);
  } else {
    L2s_Ltl2l2s_delete(app);
    exit (1);
  }

  assert (0);
  return 0;
}

/*------------------------------------------------------------------------*/

int main (int argc, char *argv[]) {
  return L2s_Ltl2l2s_main(argc, argv);
}

/*------------------------------------------------------------------------*/
