#include <assert.h>
#include <stdlib.h>
#include "l2s.h"

/*------------------------------------------------------------------------*/
/* PLTLB formula                                                          */
/*------------------------------------------------------------------------*/

struct L2s_PLTLBFormula
{
  L2s_PLTLBNode *top;
};

/*------------------------------------------------------------------------*/

L2s_PLTLBFormula *
L2s_PLTLBFormula_new ()
{
  L2s_PLTLBFormula *f;

  f = (L2s_PLTLBFormula *) malloc (sizeof (L2s_PLTLBFormula));
  assert (f);
  f->top = NULL;
  return f;
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBFormula_setTop (L2s_PLTLBFormula *f, L2s_PLTLBNode *top)
{
  assert(f);
  assert(top);
  f->top = top;
}

/*------------------------------------------------------------------------*/

L2s_PLTLBNode *
L2s_PLTLBFormula_getTop (L2s_PLTLBFormula *f)
{
  assert(f);
  return f->top;
}

/*------------------------------------------------------------------------*/

L2s_String *
L2s_PLTLBFormula_toString (L2s_PLTLBFormula * f)
{
  assert (f);
  return L2s_PLTLBNode_toString(f->top);
}

/*------------------------------------------------------------------------*/

static void
L2s_PLTLBFormula_deleterec (L2s_PLTLBFormula * f, L2s_PLTLBNode *n)
{
  L2s_PLTLBOperator op;

  assert (f);
  assert (n);
  op = L2s_PLTLBNode_getOp(n);
  if (L2s_PLTLBOperator_isNullary (op))
    {
      /* nop */
    }
  else if (L2s_PLTLBOperator_isUnary (op))
    {
      L2s_PLTLBFormula_deleterec (f, L2s_PLTLBNode_getLeft(n));
    }
  else
    {
      assert (L2s_PLTLBOperator_isBinary (op));
      L2s_PLTLBFormula_deleterec (f, L2s_PLTLBNode_getLeft(n));
      L2s_PLTLBFormula_deleterec (f, L2s_PLTLBNode_getRight(n));
    }
  L2s_PLTLBNode_delete(n);
}

/*------------------------------------------------------------------------*/

void
L2s_PLTLBFormula_delete (L2s_PLTLBFormula * f)
{
  assert (f);
  L2s_PLTLBFormula_deleterec(f, f->top);
  free (f);
}

/*------------------------------------------------------------------------*/
/* unit tests                                                             */
/*------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
  print formulae with temporal or boolean unary operator
*/
static int
L2s_test_PLTLBFormula_printUnary (FILE * log, L2s_PLTLBOperator op)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1, *n2;
  L2s_String *s1, *str;

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  n2 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n2, op);
  L2s_PLTLBNode_setLeft (n2, n1);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n2);

  str = L2s_PLTLBFormula_toString (f);
  L2s_String_fprint(str, log);
  fprintf(log, "\n");

  L2s_String_delete (str);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  print formulae with temporal or boolean unary operator
*/
static int
L2s_test_PLTLBFormula_printBinary (FILE * log, L2s_PLTLBOperator op)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1, *n2, *n3;
  L2s_String *s1, *s2, *str;

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  s2 = L2s_String_new ();
  L2s_String_append (s2, "q");
  n2 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n2, ATOM);
  L2s_PLTLBNode_setAtom (n2, s2);
  n3 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n3, op);
  L2s_PLTLBNode_setLeft (n3, n1);
  L2s_PLTLBNode_setRight (n3, n2);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n3);

  str = L2s_PLTLBFormula_toString (f);
  L2s_String_fprint(str, log);
  fprintf(log, "\n");

  L2s_String_delete (str);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  print p
*/
int
L2s_test_PLTLBFormula_print_p (FILE * log)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1;
  L2s_String *s1, *str;

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n1);

  str = L2s_PLTLBFormula_toString (f);
  L2s_String_fprint(str, log);
  fprintf(log, "\n");

  L2s_String_delete (str);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  print !p
*/
int
L2s_test_PLTLBFormula_print_NOTp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, NOT);
}

/*--------------------------------------------------------------------------
  print p | q
*/
int
L2s_test_PLTLBFormula_print_pORq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, OR);
}

/*--------------------------------------------------------------------------
  print p & q
*/
int
L2s_test_PLTLBFormula_print_pANDq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, AND);
}

/*--------------------------------------------------------------------------
  print p -> q
*/
int
L2s_test_PLTLBFormula_print_pIMPLICATIONq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, IMPLICATION);
}

/*--------------------------------------------------------------------------
  print p <-> q
*/
int
L2s_test_PLTLBFormula_print_pEQUIVALENCEq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, EQUIVALENCE);
}

/*--------------------------------------------------------------------------
  print X p
*/
int
L2s_test_PLTLBFormula_print_Xp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, X);
}

/*--------------------------------------------------------------------------
  print p U q
*/
int
L2s_test_PLTLBFormula_print_pUq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, U);
}

/*--------------------------------------------------------------------------
  print p V q
*/
int
L2s_test_PLTLBFormula_print_pVq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, V);
}

/*--------------------------------------------------------------------------
  print F p
*/
int
L2s_test_PLTLBFormula_print_Fp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, F);
}

/*--------------------------------------------------------------------------
  print G p
*/
int
L2s_test_PLTLBFormula_print_Gp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, G);
}

/*--------------------------------------------------------------------------
  print Y p
*/
int
L2s_test_PLTLBFormula_print_Yp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, Y);
}

/*--------------------------------------------------------------------------
  print Z p
*/
int
L2s_test_PLTLBFormula_print_Zp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, Z);
}

/*--------------------------------------------------------------------------
  print p S q
*/
int
L2s_test_PLTLBFormula_print_pSq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, S);
}

/*--------------------------------------------------------------------------
  print p T q
*/
int
L2s_test_PLTLBFormula_print_pTq (FILE * log)
{
  return L2s_test_PLTLBFormula_printBinary(log, T);
}

/*--------------------------------------------------------------------------
  print O p
*/
int
L2s_test_PLTLBFormula_print_Op (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, O);
}

/*--------------------------------------------------------------------------
  print H p
*/
int
L2s_test_PLTLBFormula_print_Hp (FILE * log)
{
  return L2s_test_PLTLBFormula_printUnary(log, H);
}

/*------------------------------------------------------------------------*/
