#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "l2s.h"
#include "encoder.h"

/*------------------------------------------------------------------------*/
/* tight Hamaguchi-like encoding                                          */
/*------------------------------------------------------------------------*/

static char *
L2s_CGHTightEncoder_getOnloopName (L2s_Encoder *enc) {
  L2s_String *str;
  char *s;
  
  assert(enc);
  str = L2s_String_clone(enc->varprefix);
  L2s_String_append(str, "onloop");
  s = L2s_String_toCharPtr(str);
  L2s_String_delete(str);
  return s;
}

/*------------------------------------------------------------------------*/

static char *
L2s_CGHTightEncoder_getUpName (L2s_Encoder *enc) {
  L2s_String *str;
  char *s;
  
  assert(enc);
  str = L2s_String_clone(enc->varprefix);
  L2s_String_append(str, "up");
  s = L2s_String_toCharPtr(str);
  L2s_String_delete(str);
  return s;
}

/*------------------------------------------------------------------------*/
static char *
L2s_CGHTightEncoder_getName (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			     int it)
{
  char *s, *t;

  s = (char *) malloc (sizeof (char) * (4 + 2 * 10));	/* 32 Bit int */
  assert (s);
  t = L2s_String_toCharPtr(L2s_Encoder_getVarprefix(enc));
  sprintf (s, "%s%d_%d", t, it, id);
  free(t);
  return s;
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_commentformula(L2s_Encoder * enc, L2s_PLTLBNode * phi) {
  L2s_String *str;

  assert (enc);
  assert (phi);
#if 0
  L2s_String_append(enc->res, "--\n-- ");
  str = L2s_PLTLBNode_toString(phi);
  L2s_String_appendS(enc->res, str);
  L2s_String_delete(str);
  L2s_String_append(enc->res, "\n--\n");  
#endif
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_declarevars (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
				 int from, int to)
{
  int i;
  char *namephi;

  assert (enc);
  assert (phi);
  assert (to >= from);
  L2s_String_append (enc->res, "VAR\n");
  for (i = 0; i <= to; i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      L2s_String_append (enc->res, "  ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ": boolean;\n");
      free (namephi);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encatom (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id)
{
  char *name;

  assert (enc);
  assert (phi);
  assert (L2s_PLTLBNode_getOp(phi) == ATOM);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  name = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  L2s_String_append (enc->res, "DEFINE\n  ");
  L2s_String_append (enc->res, name);
  L2s_String_append (enc->res, " := (");
  L2s_String_appendS (enc->res, L2s_PLTLBNode_getAtom (phi));
  L2s_String_append (enc->res, ");\n");
  free (name);
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encconst (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id)
{
  L2s_PLTLBOperator op;
  char *name;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == TRUE || op == FALSE);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  name = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  L2s_String_append (enc->res, "DEFINE\n  ");
  L2s_String_append (enc->res, name);
  L2s_String_append (enc->res, " := ");
  L2s_String_append (enc->res, (op == TRUE) ? "1" : "0");
  L2s_String_append (enc->res, ";\n");
  free (name);
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encnot (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			    int idpsi1)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1;
  int i;
  char *namephi, *namepsi1;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == NOT);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  for (i = 0; i <= L2s_PLTLBNode_pod (phi); i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, i);
      L2s_String_append (enc->res, "DEFINE\n  ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " := !");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ";\n");
      free (namephi);
      free (namepsi1);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encbinarybool (L2s_Encoder * enc, L2s_PLTLBNode * phi,
				   int id, int idpsi1, int idpsi2)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1, *psi2;
  int i;
  char *namephi, *namepsi1, *namepsi2, ops[6];

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  for (i = 0; i <= L2s_PLTLBNode_pod (phi); i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1 =
	L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				     min (i, L2s_PLTLBNode_pod (psi1)));
      namepsi2 =
	L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				     min (i, L2s_PLTLBNode_pod (psi2)));
      switch (op)
	{
	case OR:
	  sprintf (ops, " | ");
	  break;
	case AND:
	  sprintf (ops, " & ");
	  break;
	case IMPLICATION:
	  sprintf (ops, " -> ");
	  break;
	case EQUIVALENCE:
	  sprintf (ops, " <-> ");
	  break;
	default:
	  assert (0);
	  break;
	}
      L2s_String_append (enc->res, "DEFINE\n  ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " := ");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ops);
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res, ";\n");
      free (namephi);
      free (namepsi1);
      free (namepsi2);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encX (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			  int idpsi1)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1;
  int i, pod;
  char *nameonloop, *nameup, *namephi, *namepsi1, *namepsi1ip1;
  
  assert (enc);
  assert (phi);
  nameonloop = L2s_CGHTightEncoder_getOnloopName(enc);
  nameup = L2s_CGHTightEncoder_getUpName(enc);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == X);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  pod = L2s_PLTLBNode_pod (phi);
  L2s_CGHTightEncoder_declarevars (enc, phi, id, 0, pod);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, 0);
  L2s_String_append (enc->res, "TRANS\n  !");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> next(");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, "))\n");
  free (namephi);
  free (namepsi1);
  for (i = 0; i < pod; i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, i);
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & !");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> next(");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, "))\n");
      free (namephi);
      free (namepsi1);
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1ip1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, i+1);
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & ");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> next(");
      L2s_String_append (enc->res, namepsi1ip1);
      L2s_String_append (enc->res, "))\n");
      free (namephi);
      free (namepsi1ip1);
    }
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, pod);
  namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, pod);
  L2s_String_append (enc->res, "TRANS\n  ");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> next(");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, "))\n");
  free (namephi);
  free (namepsi1);
  free(nameonloop);
  free(nameup);
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encUV (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			   int idpsi1, int idpsi2)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1, *psi2;
  int i, pod;
  char *nameonloop, *nameup, *namephi, *namephiip1, *namepsi1, *namepsi2;

  assert (enc);
  assert (phi);
  nameonloop = L2s_CGHTightEncoder_getOnloopName(enc);
  nameup = L2s_CGHTightEncoder_getUpName(enc);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == U || op == V);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  pod = L2s_PLTLBNode_pod (phi);
  L2s_CGHTightEncoder_declarevars (enc, phi, id, 0, pod);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, 0);
  namepsi2 = L2s_CGHTightEncoder_getName (enc, psi2, idpsi2, 0);
  L2s_String_append (enc->res, "TRANS\n  !");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ")))\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  for (i = 0; i < pod; i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1 =
	L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				     min (i, L2s_PLTLBNode_pod (psi1)));
      namepsi2 =
	L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				     min (i, L2s_PLTLBNode_pod (psi2)));
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & !");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> ");
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ")))\n");
      free (namephi);
      free (namepsi1);
      free (namepsi2);
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namephiip1 = L2s_CGHTightEncoder_getName (enc, phi, id, i+1);
      namepsi1 =
	L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				     min (i, L2s_PLTLBNode_pod (psi1)));
      namepsi2 =
	L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				     min (i, L2s_PLTLBNode_pod (psi2)));
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & ");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> ");
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
      L2s_String_append (enc->res, namephiip1);
      L2s_String_append (enc->res, ")))\n");
      free (namephi);
      free (namephiip1);
      free (namepsi1);
      free (namepsi2);
    }
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, pod);
  namepsi1 =
    L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				 min (pod, L2s_PLTLBNode_pod (psi1)));
  namepsi2 =
    L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				 min (pod, L2s_PLTLBNode_pod (psi2)));
  L2s_String_append (enc->res, "TRANS\n  ");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ")))\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, pod);
  namepsi2 = L2s_CGHTightEncoder_getName (enc, psi2, idpsi2, min(pod, L2s_PLTLBNode_pod(psi2)));
  L2s_String_append (enc->res, "FAIRNESS\n  ");
  L2s_String_append (enc->res, (op == U) ? "!" : "");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, (op == U) ? " | " : " | !");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, "\n");
  free(namephi);
  free(namepsi2);
  free(nameonloop);
  free(nameup);
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encYZ (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
                           int idpsi1)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1;
  int i, pod;
  char *nameonloop, *nameup, *namephi, *namephiip1, *namepsi1;
  
  assert (enc);
  assert (phi);
  nameonloop = L2s_CGHTightEncoder_getOnloopName(enc);
  nameup = L2s_CGHTightEncoder_getUpName(enc);
  op = L2s_PLTLBNode_getOp(phi);
  assert ((op == Y) || (op == Z));
  psi1 = L2s_PLTLBNode_getLeft(phi);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  pod = L2s_PLTLBNode_pod (phi);
  L2s_CGHTightEncoder_declarevars (enc, phi, id, 0, pod);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  L2s_String_append (enc->res, "INIT\n  ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, (op == Y) ? " <-> 0\n" : " <-> 1\n");
  free (namephi);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, 0);
  L2s_String_append (enc->res, "TRANS\n  !");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ") <-> ");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, ")\n");
  free (namephi);
  free (namepsi1);
  for (i = 0; i < pod; i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, min(i,L2s_PLTLBNode_pod(psi1)));
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & !");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (next(");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ") <-> ");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ")\n");
      free (namephi);
      free (namepsi1);
      namephiip1 = L2s_CGHTightEncoder_getName (enc, phi, id, i+1);
      namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, min(i,L2s_PLTLBNode_pod(psi1)));
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & ");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (next(");
      L2s_String_append (enc->res, namephiip1);
      L2s_String_append (enc->res, ") <-> ");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ")\n");
      free (namephiip1);
      free (namepsi1);
    }
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, pod);
  namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, min(pod,L2s_PLTLBNode_pod(psi1)));
  L2s_String_append (enc->res, "TRANS\n  ");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ") <-> ");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, ")\n");
  free (namephi);
  free (namepsi1);
  free(nameonloop);
  free(nameup);
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encST (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			   int idpsi1, int idpsi2)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1, *psi2;
  int i, pod;
  char *nameonloop, *nameup, *namephi, *namephiip1, *namepsi1, *namepsi1ip1, *namepsi2, *namepsi2ip1;

  assert (enc);
  assert (phi);
  nameonloop = L2s_CGHTightEncoder_getOnloopName(enc);
  nameup = L2s_CGHTightEncoder_getUpName(enc);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == S || op == T);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  L2s_CGHTightEncoder_commentformula(enc, phi);
  pod = L2s_PLTLBNode_pod (phi);
  L2s_CGHTightEncoder_declarevars (enc, phi, id, 0, pod);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  namepsi2 = L2s_CGHTightEncoder_getName (enc, psi2, idpsi2, 0);
  L2s_String_append (enc->res, "INIT\n  ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, "\n");
  free (namephi);
  free (namepsi2);
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, 0);
  namepsi1 = L2s_CGHTightEncoder_getName (enc, psi1, idpsi1, 0);
  namepsi2 = L2s_CGHTightEncoder_getName (enc, psi2, idpsi2, 0);
  L2s_String_append (enc->res, "TRANS\n  !");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ") <-> next(");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, (op == S) ? ") | (next(" : ") & (next(");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == S) ? ") & " : ") | ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, "))\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  for (i = 0; i < pod; i++)
    {
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namepsi1 =
	L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				     min (i, L2s_PLTLBNode_pod (psi1)));
      namepsi2 =
	L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				     min (i, L2s_PLTLBNode_pod (psi2)));
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & !");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (next(");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ") <-> next(");
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res,
			 (op == S) ? ") | (next(" : ") & (next(");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, (op == S) ? ") & " : ") | ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, "))\n");
      free (namephi);
      free (namepsi1);
      free (namepsi2);
      namephi = L2s_CGHTightEncoder_getName (enc, phi, id, i);
      namephiip1 = L2s_CGHTightEncoder_getName (enc, phi, id, i+1);
      namepsi1ip1 =
	L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				     min (i+1, L2s_PLTLBNode_pod (psi1)));
      namepsi2ip1 =
	L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				     min (i+1, L2s_PLTLBNode_pod (psi2)));
      L2s_String_append (enc->res, "TRANS\n  (");
      L2s_String_append (enc->res, nameonloop);
      L2s_String_append (enc->res, " & ");
      L2s_String_append (enc->res, nameup);
      L2s_String_append (enc->res, ") -> (next(");
      L2s_String_append (enc->res, namephiip1);
      L2s_String_append (enc->res, ") <-> next(");
      L2s_String_append (enc->res, namepsi2ip1);
      L2s_String_append (enc->res,
			 (op == S) ? ") | (next(" : ") & (next(");
      L2s_String_append (enc->res, namepsi1ip1);
      L2s_String_append (enc->res, (op == S) ? ") & " : ") | ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, "))\n");
      free (namephi);
      free (namephiip1);
      free (namepsi1ip1);
      free (namepsi2ip1);
    }
  namephi = L2s_CGHTightEncoder_getName (enc, phi, id, pod);
  namepsi1 =
    L2s_CGHTightEncoder_getName (enc, psi1, idpsi1,
				 min (pod, L2s_PLTLBNode_pod (psi1)));
  namepsi2 =
    L2s_CGHTightEncoder_getName (enc, psi2, idpsi2,
				 min (pod, L2s_PLTLBNode_pod (psi2)));
  L2s_String_append (enc->res, "TRANS\n  ");
  L2s_String_append (enc->res, nameonloop);
  L2s_String_append (enc->res, " -> (next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ") <-> next(");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res,
		     (op == S) ? ") | (next(" : ") & (next(");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == S) ? ") & " : ") | ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, "))\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  free(nameonloop);
  free(nameup);
}

/*------------------------------------------------------------------------*/

static void
L2s_CGHTightEncoder_encrec (L2s_Encoder * enc, L2s_PLTLBNode * phi)
{
  int id, idpsi1, idpsi2;
  L2s_PLTLBNode *psi1, *psi2, *sigma, *sigma1;

  assert (enc);
  assert (phi);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  switch (L2s_PLTLBNode_getOp (phi))
    {
    case ATOM:
      id = ++enc->id;
      L2s_CGHTightEncoder_encatom (enc, phi, id);
      break;
    case TRUE:
    case FALSE:
      id = ++enc->id;
      L2s_CGHTightEncoder_encconst (enc, phi, id);
      break;
    case NOT:
      L2s_CGHTightEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      id = ++enc->id;
      L2s_CGHTightEncoder_encnot (enc, phi, id, idpsi1);
      break;
    case OR:
    case AND:
    case IMPLICATION:
    case EQUIVALENCE:
      L2s_CGHTightEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      L2s_CGHTightEncoder_encrec (enc, psi2);
      idpsi2 = enc->id;
      id = ++enc->id;
      L2s_CGHTightEncoder_encbinarybool (enc, phi, id, idpsi1, idpsi2);
      break;
    case X:
      L2s_CGHTightEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      id = ++enc->id;
      L2s_CGHTightEncoder_encX (enc, phi, id, idpsi1);
      break;
    case U:
    case V:
      L2s_CGHTightEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      L2s_CGHTightEncoder_encrec (enc, psi2);
      idpsi2 = enc->id;
      id = ++enc->id;
      L2s_CGHTightEncoder_encUV (enc, phi, id, idpsi1, idpsi2);
      break;
    case F:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, TRUE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, U);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_CGHTightEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    case G:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, FALSE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, V);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_CGHTightEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    case Y:
    case Z:
      L2s_CGHTightEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      id = ++enc->id;
      L2s_CGHTightEncoder_encYZ (enc, phi, id, idpsi1);
      break;
    case S:
    case T:
      L2s_CGHTightEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      L2s_CGHTightEncoder_encrec (enc, psi2);
      idpsi2 = enc->id;
      id = ++enc->id;
      L2s_CGHTightEncoder_encST (enc, phi, id, idpsi1, idpsi2);
      break;
    case O:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, TRUE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, S);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_CGHTightEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    case H:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, FALSE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, T);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_CGHTightEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    default:
      assert (0);
      break;
    }
}

/*------------------------------------------------------------------------*/

void
L2s_CGHTightEncoder_enc (L2s_Encoder * enc)
{
  char *onloopname, *upname, *name;

  assert (enc);
  assert (enc->phi);
  onloopname = L2s_CGHTightEncoder_getOnloopName(enc);
  upname = L2s_CGHTightEncoder_getUpName(enc);
#if 0
  L2s_String_append(enc->res, "--\n-- begin property automaton\n--\n");
#endif
  if (enc->modulename) {
    L2s_String_append(enc->res, "MODULE ");
    L2s_String_appendS(enc->res, enc->modulename);
    L2s_String_append(enc->res, "\n");
  }
#if 0
  L2s_String_append(enc->res, "--\n-- generic variables\n--\n");
#endif
  L2s_String_append(enc->res, "VAR\n  ");
  L2s_String_append(enc->res, upname);
  L2s_String_append(enc->res, ": boolean;\n");
  L2s_String_append(enc->res, "  ");
  L2s_String_append(enc->res, onloopname);
  L2s_String_append(enc->res, ": boolean;\n");
  L2s_String_append(enc->res, "TRANS\n  ");
  L2s_String_append(enc->res, onloopname);
  L2s_String_append(enc->res, " -> next(");
  L2s_String_append(enc->res, onloopname);
  L2s_String_append(enc->res, ")\n");
  L2s_String_append(enc->res, "FAIRNESS\n  ");
  L2s_String_append(enc->res, onloopname);
  L2s_String_append(enc->res, " & ");
  L2s_String_append(enc->res, upname);
  L2s_String_append(enc->res, "\n");
  L2s_CGHTightEncoder_encrec (enc, L2s_PLTLBFormula_getTop(enc->phi));
  name = L2s_CGHTightEncoder_getName(enc, L2s_PLTLBFormula_getTop(enc->phi), enc->id, 0);
#if 0
  L2s_String_append(enc->res, "--\n-- property\n--\n");
#endif
  L2s_String_append(enc->res, "DEFINE\n  ");
  L2s_String_appendS(enc->res, enc->propertyvarname);
  L2s_String_append(enc->res, " := ");
  L2s_String_append(enc->res, name);
  L2s_String_append(enc->res, ";\n");
  L2s_String_append(enc->res, "INIT\n  ");
  L2s_String_appendS(enc->res, enc->propertyvarname);
  L2s_String_append(enc->res, "\n");
#if 0
  L2s_String_append(enc->res, "--\n-- end property automaton\n--\n");
#endif
  free(name);
  free(onloopname);
  free(upname);
}

/*------------------------------------------------------------------------*/
/* unit tests                                                             */
/*------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
  encode formulae with temporal or boolean unary operator
*/
static int
L2s_test_CGHTightEncoder_unary (FILE * log, L2s_PLTLBOperator op)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1, *n2;
  L2s_String *s1;
  L2s_Encoder *enc;

  enc = L2s_Encoder_new ();
  L2s_Encoder_setEncoding(enc, CGHTight);

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  n2 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n2, op);
  L2s_PLTLBNode_setLeft (n2, n1);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n2);

  L2s_Encoder_setPhi (enc, f);
  L2s_CGHTightEncoder_enc (enc);
  L2s_String_fprint(enc->res, log);

  L2s_String_delete(enc->res);
  L2s_Encoder_delete (enc);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  encode formulae with temporal or boolean binary operator
*/
static int
L2s_test_CGHTightEncoder_binary (FILE * log, L2s_PLTLBOperator op)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1, *n2, *n3;
  L2s_String *s1, *s2;
  L2s_Encoder *enc;

  enc = L2s_Encoder_new ();
  L2s_Encoder_setEncoding(enc, CGHTight);

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  s2 = L2s_String_new ();
  L2s_String_append (s2, "q");
  n2 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n2, ATOM);
  L2s_PLTLBNode_setAtom (n2, s2);
  n3 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n3, op);
  L2s_PLTLBNode_setLeft (n3, n1);
  L2s_PLTLBNode_setRight (n3, n2);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n3);

  L2s_Encoder_setPhi (enc, f);
  L2s_CGHTightEncoder_enc (enc);
  L2s_String_fprint (enc->res, log);

  L2s_String_delete(enc->res);
  L2s_Encoder_delete (enc);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  encode p
*/
int
L2s_test_CGHTightEncoder_p (FILE * log)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1;
  L2s_String *s1;
  L2s_Encoder *enc;

  enc = L2s_Encoder_new ();
  L2s_Encoder_setEncoding(enc, CGHTight);

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n1);

  L2s_Encoder_setPhi (enc, f);
  L2s_CGHTightEncoder_enc (enc);
  L2s_String_fprint(enc->res, log);

  L2s_String_delete(enc->res);
  L2s_Encoder_delete (enc);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  encode !p
*/
int
L2s_test_CGHTightEncoder_NOTp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, NOT);
}

/*--------------------------------------------------------------------------
  encode p | q
*/
int
L2s_test_CGHTightEncoder_pORq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, OR);
}

/*--------------------------------------------------------------------------
  encode p & q
*/
int
L2s_test_CGHTightEncoder_pANDq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, AND);
}

/*--------------------------------------------------------------------------
  encode p -> q
*/
int
L2s_test_CGHTightEncoder_pIMPLICATIONq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, IMPLICATION);
}

/*--------------------------------------------------------------------------
  encode p <-> q
*/
int
L2s_test_CGHTightEncoder_pEQUIVALENCEq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, EQUIVALENCE);
}

/*--------------------------------------------------------------------------
  encode X p
*/
int
L2s_test_CGHTightEncoder_Xp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, X);
}

/*--------------------------------------------------------------------------
  encode p U q
*/
int
L2s_test_CGHTightEncoder_pUq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, U);
}

/*--------------------------------------------------------------------------
  encode p V q
*/
int
L2s_test_CGHTightEncoder_pVq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, V);
}

/*--------------------------------------------------------------------------
  encode F p
*/
int
L2s_test_CGHTightEncoder_Fp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, F);
}

/*--------------------------------------------------------------------------
  encode G p
*/
int
L2s_test_CGHTightEncoder_Gp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, G);
}

/*--------------------------------------------------------------------------
  encode Y p
*/
int
L2s_test_CGHTightEncoder_Yp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, Y);
}

/*--------------------------------------------------------------------------
  encode Z p
*/
int
L2s_test_CGHTightEncoder_Zp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, Z);
}

/*--------------------------------------------------------------------------
  encode p S q
*/
int
L2s_test_CGHTightEncoder_pSq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, S);
}

/*--------------------------------------------------------------------------
  encode p T q
*/
int
L2s_test_CGHTightEncoder_pTq (FILE * log)
{
  return L2s_test_CGHTightEncoder_binary(log, T);
}

/*--------------------------------------------------------------------------
  encode O p
*/
int
L2s_test_CGHTightEncoder_Op (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, O);
}

/*--------------------------------------------------------------------------
  encode H p
*/
int
L2s_test_CGHTightEncoder_Hp (FILE * log)
{
  return L2s_test_CGHTightEncoder_unary(log, H);
}

/*-------------------------------------------------------------------------*/
