#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "l2s.h"
#include "encoder.h"

/*------------------------------------------------------------------------*/
/* nondeterministic encoding described in TACAS'05 sub                    */
/*------------------------------------------------------------------------*/

static int
L2s_BMCNdetEncoder_pod (L2s_Encoder * enc, L2s_PLTLBNode * phi)
{
  assert(enc);
  assert(phi);
  return min(L2s_PLTLBNode_pod (phi), L2s_Encoder_getMaxunroll(enc));
}

/*------------------------------------------------------------------------*/

static char *
L2s_BMCNdetEncoder_getName (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			    int it, int l2s)
{
  char *s, *t;

  assert(enc);
  assert(phi);
  t = L2s_String_toCharPtr(L2s_Encoder_getVarprefix(enc));
  s = (char *) malloc (sizeof (char) * (14 + strlen(t) + 2 * 10)); /* 32 Bit int */
  assert (s);
  sprintf (s, "%s%s%010d_%010d", l2s ? "__l2s_copy__" : "", t, it, id);
  free(t);
  return s;
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_commentformula(L2s_Encoder * enc, L2s_PLTLBNode * phi) {
  L2s_String *str;

  assert (enc);
  assert (phi);
  L2s_String_append(enc->res, "--\n-- ");
  str = L2s_PLTLBNode_toString(phi);
  L2s_String_appendS(enc->res, str);
  L2s_String_delete(str);
  L2s_String_append(enc->res, "\n--\n");  
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_declarevars (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
				int from, int to)
{
  int i;
  char *namephi;

  assert (enc);
  assert (phi);
  assert (to >= from);
  L2s_String_append (enc->res, "VAR\n");
  for (i = 0; i <= to; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      L2s_String_append (enc->res, "  ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ": boolean;\n");
      free (namephi);
      if (i >= from)
	{
	  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 1);
	  L2s_String_append (enc->res, "  ");
	  L2s_String_append (enc->res, namephi);
	  L2s_String_append (enc->res, ": boolean;\n");
	  free (namephi);
	}
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_loopendvars (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
				int from, int to)
{
  int i;
  char *namephi, *namephil2s;

  assert (enc);
  assert (phi);
  assert (from > 0);
  for (i = from - 1; i < to; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namephil2s = L2s_BMCNdetEncoder_getName (enc, phi, id, i + 1, 1);
      L2s_String_append (enc->res, "INVAR\n  __l2s_looped__ -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> ");
      L2s_String_append (enc->res, namephil2s);
      L2s_String_append (enc->res, ")\n");
      free (namephi);
      free (namephil2s);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encatom (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id)
{
  char *name;

  assert (enc);
  assert (phi);
  assert (L2s_PLTLBNode_getOp(phi) == ATOM);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  name = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  L2s_String_append (enc->res, "DEFINE\n  ");
  L2s_String_append (enc->res, name);
  L2s_String_append (enc->res, " := (");
  L2s_String_appendS (enc->res, L2s_PLTLBNode_getAtom (phi));
  L2s_String_append (enc->res, ");\n");
  free (name);
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encconst (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id)
{
  L2s_PLTLBOperator op;
  char *name;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == TRUE || op == FALSE);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  name = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  L2s_String_append (enc->res, "DEFINE\n  ");
  L2s_String_append (enc->res, name);
  L2s_String_append (enc->res, " := ");
  L2s_String_append (enc->res, (op == TRUE) ? "1" : "0");
  L2s_String_append (enc->res, ";\n");
  free (name);
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encnot (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			   int idpsi1)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1;
  int i, podphi;
  char *namephi, *namepsi1;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == NOT);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  podphi = L2s_BMCNdetEncoder_pod (enc, phi);
  for (i = 0; i <= podphi; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, i, 0);
      L2s_String_append (enc->res, "DEFINE\n  ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " := !");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ";\n");
      free (namephi);
      free (namepsi1);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encbinarybool (L2s_Encoder * enc, L2s_PLTLBNode * phi,
				  int id, int idpsi1, int idpsi2)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1, *psi2;
  int i, podphi, podpsi1, podpsi2;
  char *namephi, *namepsi1, *namepsi2, ops[6];

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  podphi = L2s_BMCNdetEncoder_pod (enc, phi);
  for (i = 0; i <= podphi; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      podpsi1 = L2s_BMCNdetEncoder_pod (enc, psi1);
      podpsi2 = L2s_BMCNdetEncoder_pod (enc, psi2);
      namepsi1 =
	L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1,
				min (i, podpsi1), 0);
      namepsi2 =
	L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2,
				min (i, podpsi2), 0);
      switch (op)
	{
	case OR:
	  sprintf (ops, " | ");
	  break;
	case AND:
	  sprintf (ops, " & ");
	  break;
	case IMPLICATION:
	  sprintf (ops, " -> ");
	  break;
	case EQUIVALENCE:
	  sprintf (ops, " <-> ");
	  break;
	default:
	  assert (0);
	  break;
	}
      L2s_String_append (enc->res, "DEFINE\n  ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " := ");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ops);
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res, ";\n");
      free (namephi);
      free (namepsi1);
      free (namepsi2);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_savevars (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			     int from, int to)
{
  int i;
  char *namephi, *namephil2s;

  assert (enc);
  assert (phi);
  assert (to >= from);
  for (i = from; i <= to; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namephil2s = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 1);
      L2s_String_append (enc->res, "TRANS\n  __l2s_save__ & !__l2s_saved__ -> (next(");
      L2s_String_append (enc->res, namephil2s);
      L2s_String_append (enc->res, ") <-> ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ")\n");
      L2s_String_append (enc->res, "TRANS\n  __l2s_saved__ & !__l2s_looped__ -> (next(");
      L2s_String_append (enc->res, namephil2s);
      L2s_String_append (enc->res, ") <-> ");
      L2s_String_append (enc->res, namephil2s);
      L2s_String_append (enc->res, ")\n");
      free (namephi);
      free (namephil2s);
    }
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encX (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			 int idpsi1)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1;
  int i, podphi;
  char *namephi, *namephil2s, *namepsi1;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == X);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  podphi = L2s_BMCNdetEncoder_pod (enc, phi);
  L2s_BMCNdetEncoder_declarevars (enc, phi, id, (podphi > 0) ? 1 : 0, podphi);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, 0, 0);
  L2s_String_append (enc->res, "TRANS\n  !__l2s_looped__ -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> next(");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, "))\n");
  free (namephi);
  free (namepsi1);
  for (i = 1; i <= podphi; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, i, 0);
      L2s_String_append (enc->res, "TRANS\n  (__l2s_save__ | __l2s_saved__) & !__l2s_looped__ -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> next(");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, "))\n");
      free (namephi);
      free (namepsi1);
    }
  L2s_BMCNdetEncoder_loopendvars (enc, phi, id, 1, podphi);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi, 0);
  namephil2s = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi, 1);
  L2s_String_append (enc->res, "INVAR\n  __l2s_looped__ -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namephil2s);
  L2s_String_append (enc->res, ")\n");
  free (namephi);
  free (namephil2s);
  L2s_BMCNdetEncoder_savevars (enc, phi, id, (podphi > 0) ? 1 : 0, podphi);
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encUV (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			  int idpsi1, int idpsi2)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1, *psi2;
  int i, podphi, podpsi1, podpsi2;
  char *namephi, *namepsi1, *namepsi2;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == U || op == V);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  podphi = L2s_BMCNdetEncoder_pod (enc, phi);
  podpsi1 = L2s_BMCNdetEncoder_pod (enc, psi1);
  podpsi2 = L2s_BMCNdetEncoder_pod (enc, psi2);
  L2s_BMCNdetEncoder_declarevars (enc, phi, id, 1, podphi + 1);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, 0, 0);
  namepsi2 = L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2, 0, 0);
  L2s_String_append (enc->res, "TRANS\n  !__l2s_looped__ -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ")))\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  for (i = 1; i <= podphi; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namepsi1 =
	L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1,
				min (i, podpsi1), 0);
      namepsi2 =
	L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2,
				min (i, podpsi2), 0);
      L2s_String_append (enc->res, "TRANS\n  (__l2s_save__ | __l2s_saved__) & !__l2s_looped__ -> (");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, " <-> ");
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ")))\n");
      free (namephi);
      free (namepsi1);
      free (namepsi2);
    }
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi + 1, 0);
  namepsi1 =
    L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1,
			    min (podphi, podpsi1), 0);
  namepsi2 =
    L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2,
			    min (podphi, podpsi2), 0);
  L2s_String_append (enc->res, "TRANS\n  (__l2s_save__ | __l2s_saved__) & !__l2s_looped__ -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, (op == U) ? " | (" : " & (");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == U) ? " & next(" : " | next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ")))\n");
  L2s_String_append (enc->res, "INVAR\n  (__l2s_save__ | __l2s_saved__) & __l2s_looped__ -> (");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, ")\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  L2s_BMCNdetEncoder_loopendvars (enc, phi, id, 1, podphi + 1);
  L2s_BMCNdetEncoder_savevars (enc, phi, id, 1, podphi + 1);
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encYZ (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
                          int idpsi1)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1;
  int i, podphi, podpsi1;
  char *namephi, *namephil2s, *namepsi1;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert ((op == Y) || (op == Z));
  psi1 = L2s_PLTLBNode_getLeft(phi);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  podphi = L2s_BMCNdetEncoder_pod (enc, phi);
  podpsi1 = L2s_BMCNdetEncoder_pod (enc, psi1);
  L2s_BMCNdetEncoder_declarevars (enc, phi, id, (podphi > 0) ? 1 : 0, podphi);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  L2s_String_append (enc->res, "INIT\n  ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, (op == Y) ? " <-> 0\n" : " <-> 1\n");
  free (namephi);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, 0, 0);
  L2s_String_append (enc->res, "TRANS\n  !__l2s_looped__ -> (next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ") <-> ");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, ")\n");
  free (namephi);
  free (namepsi1);
  for (i = 1; i <= podphi; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, min(i, podpsi1), 0);
      L2s_String_append (enc->res, "TRANS\n  (__l2s_save__ | __l2s_saved__) & !__l2s_looped__ -> (next(");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ") <-> ");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, ")\n");
      free (namephi);
      free (namepsi1);
    }
  L2s_BMCNdetEncoder_loopendvars (enc, phi, id, 1, podphi);
  if (L2s_BMCNdetEncoder_pod(enc, phi) < L2s_PLTLBNode_pod(phi)) {
    /* only check for stabilization if number of unrollings might not
       be enough */
    namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi, 0);
    namephil2s = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi, 1);
    L2s_String_append (enc->res, "INVAR\n  __l2s_looped__ -> (");
    L2s_String_append (enc->res, namephi);
    L2s_String_append (enc->res, " <-> ");
    L2s_String_append (enc->res, namephil2s);
    L2s_String_append (enc->res, ")\n");
    free (namephi);
    free (namephil2s);
  }
  L2s_BMCNdetEncoder_savevars (enc, phi, id, (podphi > 0) ? 1 : 0, podphi);
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encST (L2s_Encoder * enc, L2s_PLTLBNode * phi, int id,
			  int idpsi1, int idpsi2)
{
  L2s_PLTLBOperator op;
  L2s_PLTLBNode *psi1, *psi2;
  int i, podphi, podpsi1, podpsi2;
  char *namephi, *namephil2s, *namepsi1, *namepsi2;

  assert (enc);
  assert (phi);
  op = L2s_PLTLBNode_getOp(phi);
  assert (op == S || op == T);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  L2s_BMCNdetEncoder_commentformula(enc, phi);
  podphi = L2s_BMCNdetEncoder_pod (enc, phi);
  podpsi1 = L2s_BMCNdetEncoder_pod (enc, psi1);
  podpsi2 = L2s_BMCNdetEncoder_pod (enc, psi2);
  L2s_BMCNdetEncoder_declarevars (enc, phi, id, (podphi > 0) ? 1 : 0, podphi);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  namepsi2 = L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2, 0, 0);
  L2s_String_append (enc->res, "INIT\n  ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, " <-> ");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, "\n");
  free (namephi);
  free (namepsi2);
  namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, 0, 0);
  namepsi1 = L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1, 0, 0);
  namepsi2 = L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2, 0, 0);
  L2s_String_append (enc->res, "TRANS\n  !__l2s_looped__ -> (next(");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, ") <-> next(");
  L2s_String_append (enc->res, namepsi2);
  L2s_String_append (enc->res, (op == S) ? ") | (next(" : ") & (next(");
  L2s_String_append (enc->res, namepsi1);
  L2s_String_append (enc->res, (op == S) ? ") & " : ") | ");
  L2s_String_append (enc->res, namephi);
  L2s_String_append (enc->res, "))\n");
  free (namephi);
  free (namepsi1);
  free (namepsi2);
  for (i = 1; i <= podphi; i++)
    {
      namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, i, 0);
      namepsi1 =
	L2s_BMCNdetEncoder_getName (enc, psi1, idpsi1,
				min (i, podpsi1), 0);
      namepsi2 =
	L2s_BMCNdetEncoder_getName (enc, psi2, idpsi2,
				min (i, podpsi2), 0);
      L2s_String_append (enc->res, "TRANS\n  (__l2s_save__ | __l2s_saved__) & !__l2s_looped__ -> (next(");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, ") <-> next(");
      L2s_String_append (enc->res, namepsi2);
      L2s_String_append (enc->res,
			 (op == S) ? ") | (next(" : ") & (next(");
      L2s_String_append (enc->res, namepsi1);
      L2s_String_append (enc->res, (op == S) ? ") & " : ") | ");
      L2s_String_append (enc->res, namephi);
      L2s_String_append (enc->res, "))\n");
      free (namephi);
      free (namepsi1);
      free (namepsi2);
    }
  L2s_BMCNdetEncoder_loopendvars (enc, phi, id, 1, podphi);
  if (L2s_BMCNdetEncoder_pod(enc, phi) < L2s_PLTLBNode_pod(phi)) {
    /* only check for stabilization if number of unrollings might not
       be enough */
    namephi = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi, 0);
    namephil2s = L2s_BMCNdetEncoder_getName (enc, phi, id, podphi, 1);
    L2s_String_append (enc->res, "INVAR\n  __l2s_looped__ -> (");
    L2s_String_append (enc->res, namephi);
    L2s_String_append (enc->res, " <-> ");
    L2s_String_append (enc->res, namephil2s);
    L2s_String_append (enc->res, ")\n");
    free (namephi);
    free (namephil2s);
  }
  L2s_BMCNdetEncoder_savevars (enc, phi, id, (podphi > 0) ? 1 : 0, podphi);
}

/*------------------------------------------------------------------------*/

static void
L2s_BMCNdetEncoder_encrec (L2s_Encoder * enc, L2s_PLTLBNode * phi)
{
  int id, idpsi1, idpsi2;
  L2s_PLTLBNode *psi1, *psi2, *sigma, *sigma1;

  assert (enc);
  assert (phi);
  psi1 = L2s_PLTLBNode_getLeft(phi);
  psi2 = L2s_PLTLBNode_getRight(phi);
  switch (L2s_PLTLBNode_getOp (phi))
    {
    case ATOM:
      id = ++enc->id;
      L2s_BMCNdetEncoder_encatom (enc, phi, id);
      break;
    case TRUE:
    case FALSE:
      id = ++enc->id;
      L2s_BMCNdetEncoder_encconst (enc, phi, id);
      break;
    case NOT:
      L2s_BMCNdetEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      id = ++enc->id;
      L2s_BMCNdetEncoder_encnot (enc, phi, id, idpsi1);
      break;
    case OR:
    case AND:
    case IMPLICATION:
    case EQUIVALENCE:
      L2s_BMCNdetEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      L2s_BMCNdetEncoder_encrec (enc, psi2);
      idpsi2 = enc->id;
      id = ++enc->id;
      L2s_BMCNdetEncoder_encbinarybool (enc, phi, id, idpsi1, idpsi2);
      break;
    case X:
      L2s_BMCNdetEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      id = ++enc->id;
      L2s_BMCNdetEncoder_encX (enc, phi, id, idpsi1);
      break;
    case U:
    case V:
      L2s_BMCNdetEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      L2s_BMCNdetEncoder_encrec (enc, psi2);
      idpsi2 = enc->id;
      id = ++enc->id;
      L2s_BMCNdetEncoder_encUV (enc, phi, id, idpsi1, idpsi2);
      break;
    case F:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, TRUE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, U);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_BMCNdetEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    case G:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, FALSE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, V);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_BMCNdetEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    case Y:
    case Z:
      L2s_BMCNdetEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      id = ++enc->id;
      L2s_BMCNdetEncoder_encYZ (enc, phi, id, idpsi1);
      break;
    case S:
    case T:
      L2s_BMCNdetEncoder_encrec (enc, psi1);
      idpsi1 = enc->id;
      L2s_BMCNdetEncoder_encrec (enc, psi2);
      idpsi2 = enc->id;
      id = ++enc->id;
      L2s_BMCNdetEncoder_encST (enc, phi, id, idpsi1, idpsi2);
      break;
    case O:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, TRUE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, S);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_BMCNdetEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    case H:
      sigma1 = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma1, FALSE);
      sigma = L2s_PLTLBNode_new ();
      L2s_PLTLBNode_setOp (sigma, T);
      L2s_PLTLBNode_setLeft (sigma, sigma1);
      L2s_PLTLBNode_setRight (sigma, L2s_PLTLBNode_getLeft (phi));
      L2s_BMCNdetEncoder_encrec (enc, sigma);
      L2s_PLTLBNode_delete(sigma1);
      L2s_PLTLBNode_delete(sigma);
      break;
    default:
      assert (0);
      break;
    }
}

/*------------------------------------------------------------------------*/

void
L2s_BMCNdetEncoder_enc (L2s_Encoder * enc)
{
  char *name;

  assert (enc);
  assert (enc->phi);
  L2s_String_append(enc->res, "--\n-- begin property automaton\n--\n");
  if (enc->modulename) {
    L2s_String_append(enc->res, "MODULE ");
    L2s_String_appendS(enc->res, enc->modulename);
    L2s_String_append(enc->res, "\n");
  }
  L2s_BMCNdetEncoder_encrec (enc, L2s_PLTLBFormula_getTop(enc->phi));
  name = L2s_BMCNdetEncoder_getName(enc, L2s_PLTLBFormula_getTop(enc->phi), enc->id, 0, 0);
  L2s_String_append(enc->res, "--\n-- property\n--\n");
  L2s_String_append(enc->res, "DEFINE\n  ");
  L2s_String_appendS(enc->res, enc->propertyvarname);
  L2s_String_append(enc->res, " := ");
  L2s_String_append(enc->res, name);
  L2s_String_append(enc->res, ";\n");
  L2s_String_append(enc->res, "INIT\n  ");
  L2s_String_appendS(enc->res, enc->propertyvarname);
  L2s_String_append(enc->res, "\n");
  L2s_String_append(enc->res, "--\n-- end property automaton\n--\n");
  free(name);
}

/*------------------------------------------------------------------------*/
/* unit tests                                                             */
/*------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
  encode formulae with temporal or boolean unary operator
*/
static int
L2s_test_BMCNdetEncoder_unary (FILE * log, L2s_PLTLBOperator op)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1, *n2;
  L2s_String *s1;
  L2s_Encoder *enc;

  enc = L2s_Encoder_new ();
  L2s_Encoder_setEncoding(enc, BMCNdet);

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  n2 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n2, op);
  L2s_PLTLBNode_setLeft (n2, n1);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n2);

  L2s_Encoder_setPhi (enc, f);
  L2s_BMCNdetEncoder_enc (enc);
  L2s_String_fprint(enc->res, log);

  L2s_String_delete(enc->res);
  L2s_Encoder_delete (enc);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  encode formulae with temporal or boolean binary operator
*/
static int
L2s_test_BMCNdetEncoder_binary (FILE * log, L2s_PLTLBOperator op)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1, *n2, *n3;
  L2s_String *s1, *s2;
  L2s_Encoder *enc;

  enc = L2s_Encoder_new ();
  L2s_Encoder_setEncoding(enc, BMCNdet);

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  s2 = L2s_String_new ();
  L2s_String_append (s2, "q");
  n2 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n2, ATOM);
  L2s_PLTLBNode_setAtom (n2, s2);
  n3 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n3, op);
  L2s_PLTLBNode_setLeft (n3, n1);
  L2s_PLTLBNode_setRight (n3, n2);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n3);

  L2s_Encoder_setPhi (enc, f);
  L2s_BMCNdetEncoder_enc (enc);
  L2s_String_fprint (enc->res, log);

  L2s_String_delete(enc->res);
  L2s_Encoder_delete (enc);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  encode p
*/
int
L2s_test_BMCNdetEncoder_p (FILE * log)
{
  L2s_PLTLBFormula *f;
  L2s_PLTLBNode *n1;
  L2s_String *s1;
  L2s_Encoder *enc;

  enc = L2s_Encoder_new ();
  L2s_Encoder_setEncoding(enc, BMCNdet);

  s1 = L2s_String_new ();
  L2s_String_append (s1, "p");
  n1 = L2s_PLTLBNode_new ();
  L2s_PLTLBNode_setOp (n1, ATOM);
  L2s_PLTLBNode_setAtom (n1, s1);
  f = L2s_PLTLBFormula_new();
  L2s_PLTLBFormula_setTop (f, n1);

  L2s_Encoder_setPhi (enc, f);
  L2s_BMCNdetEncoder_enc (enc);
  L2s_String_fprint(enc->res, log);

  L2s_String_delete(enc->res);
  L2s_Encoder_delete (enc);
  L2s_PLTLBFormula_delete (f);

  return 1;
}

/*--------------------------------------------------------------------------
  encode !p
*/
int
L2s_test_BMCNdetEncoder_NOTp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, NOT);
}

/*--------------------------------------------------------------------------
  encode p | q
*/
int
L2s_test_BMCNdetEncoder_pORq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, OR);
}

/*--------------------------------------------------------------------------
  encode p & q
*/
int
L2s_test_BMCNdetEncoder_pANDq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, AND);
}

/*--------------------------------------------------------------------------
  encode p -> q
*/
int
L2s_test_BMCNdetEncoder_pIMPLICATIONq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, IMPLICATION);
}

/*--------------------------------------------------------------------------
  encode p <-> q
*/
int
L2s_test_BMCNdetEncoder_pEQUIVALENCEq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, EQUIVALENCE);
}

/*--------------------------------------------------------------------------
  encode X p
*/
int
L2s_test_BMCNdetEncoder_Xp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, X);
}

/*--------------------------------------------------------------------------
  encode p U q
*/
int
L2s_test_BMCNdetEncoder_pUq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, U);
}

/*--------------------------------------------------------------------------
  encode p V q
*/
int
L2s_test_BMCNdetEncoder_pVq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, V);
}

/*--------------------------------------------------------------------------
  encode F p
*/
int
L2s_test_BMCNdetEncoder_Fp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, F);
}

/*--------------------------------------------------------------------------
  encode G p
*/
int
L2s_test_BMCNdetEncoder_Gp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, G);
}

/*--------------------------------------------------------------------------
  encode Y p
*/
int
L2s_test_BMCNdetEncoder_Yp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, Y);
}

/*--------------------------------------------------------------------------
  encode Z p
*/
int
L2s_test_BMCNdetEncoder_Zp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, Z);
}

/*--------------------------------------------------------------------------
  encode p S q
*/
int
L2s_test_BMCNdetEncoder_pSq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, S);
}

/*--------------------------------------------------------------------------
  encode p T q
*/
int
L2s_test_BMCNdetEncoder_pTq (FILE * log)
{
  return L2s_test_BMCNdetEncoder_binary(log, T);
}

/*--------------------------------------------------------------------------
  encode O p
*/
int
L2s_test_BMCNdetEncoder_Op (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, O);
}

/*--------------------------------------------------------------------------
  encode H p
*/
int
L2s_test_BMCNdetEncoder_Hp (FILE * log)
{
  return L2s_test_BMCNdetEncoder_unary(log, H);
}

/*-------------------------------------------------------------------------*/
