#!/usr/bin/perl

my $VERSION="0.12";
################################################################################
# Name:    l2s.pl
# Date:    30.11.2005
#
# Author:  Viktor Schuppan, ETH Zuerich
#
# Description:
#
# l2s.pl transforms an smv src-file into an equivalent dst-file with
# symbolic loop detection. src-file must already have been flattened
# by NuSMV. Specifications are handled by convert.bash. Remaining
# specifications are discarded. Compassion constraints are not
# supported (as this would require more than mere pattern matching in
# the parsing step). A list of variables that should neither be copied
# nor compared can be given in var-file. The "parsing" is based on
# inspection of a number of examples after flattening, hence, don't
# expect too much here.
#
# Command-line
#
# l2s.pl [-nusmv|-csmv] src-file [-var var-file] dst-file
################################################################################

use strict;

#------------------------------------------------------------------------------#
# global variables
#
my $mc;             # brand of model checker: "nusmv" or "csmv"
my %nocopyvarnames; # these variables are neither copied nor compared
my @varnames;       # names of variables
my @vartypes;       # types of variables
my @justice;        # justice constraints

#------------------------------------------------------------------------------#
# usage
#
sub usage() {
    print "usage: l2s.pl [-nusmv|-csmv] src-file [-var var-file] dst-file\n";
    exit 1;
}

#------------------------------------------------------------------------------#
# clean line
#
sub clean_line($) {
    my ($line) = shift;

    # remove comments
    $line =~ s/^--.*$//;

    # approximation to variable syntax only (see NuSMV user manual
    # sect 2.4.1 on input file syntax); remove leading and trailing
    # whitespace
    $line =~ s/^\s*([A-Za-z_][A-Za-z0-9_\$\#\-\\.[\]]*?)\s*$/\1/;

    return $line;
}

#------------------------------------------------------------------------------#
# get variables neither to be copied nor compared
#
sub get_nocopy_vars() {
    my $line;

    while ($line = <VAR>) {
	$line = clean_line $line;
	if ($line ne "") {
	    $nocopyvarnames{$line} = 1;
	}
    }
}

#------------------------------------------------------------------------------#
# extract variables and justice constraints and copy original file on
# the way; don't copy justice constraints and specifications though;
# compassion constraints are currently discarded totally
#
sub parse_and_copy() {
    my $state;
    my $line;
    my $name;
    my $type;

    $state="default";
    while ($line = <SRC>) {
	if ($line =~ /^(ASSIGN|TRANS|INIT|INVAR|DEFINE|MODULE main)/) {
	    $state="default";
	} elsif ($line =~ /^(VAR|IVAR)/) {
	    $state="var";
	} elsif ($line =~ /^(FAIRNESS|JUSTICE)/) {
	    $state="justice";
	} elsif ($line =~ /^COMPASSION/) {
	    $state="compassion";
	} elsif ($line =~ /^(SPEC|INVARSPEC|LTLSPEC|COMPUTE)/) {
	    $state="spec";
	} elsif ($state eq "var" && $line =~ /^\s*([A-Za-z_][A-Za-z0-9_\$\#\-\\.[\]]*)\s*:\s*(.*?)\s*;\s*$/) {
	    $name=$1;
	    $type=$2;

	    # NuSMV encodes __l2s_renamed___process_selector_ as constant if
	    # only one process; would result in error for optnone
	    if ($line !~ /^__l2s_renamed___process_selector_ : {main};$/ &&
		$nocopyvarnames{$name} eq "") {
		push @varnames, $name;
		push @vartypes, $type;
	    }

	} elsif ($state eq "justice" && $line =~ /^\s*(.*?)\s*$/) {
	    push @justice, $1;
	}
	if ($state eq "default" || $state eq "var") {
	    # hack: IVARs must not appear in INVAR statements, hence
	    # if IVAR is to be copied and compared it must be declared
	    # a VAR. We assume there is only one IVAr, namely,
	    # __l2s_renamed___process_selector_
	    if ($line =~ /^IVAR/ && $nocopyvarnames{__l2s_renamed___process_selector_} eq "") {
		print DST "VAR\n";
	    } else {
		print DST $line;
	    }
	}
    }
}

#------------------------------------------------------------------------------#
# add l2s part
#
sub add_l2s() {
    print DST "-- l2s section begin\n";

    # declare variables
    print DST "VAR\n";
    for (my $i = 0; $i < scalar(@varnames); $i++) {
	print DST "  __l2s_copy__$varnames[$i]: $vartypes[$i];\n";
    }
    for (my $i = 0; $i < scalar(@justice); $i++) {
	print DST "  __l2s_justice${i}__: boolean;\n";
    }
    print DST "  __l2s_save__: boolean;\n";
    print DST "  __l2s_saved__: boolean;\n";
    print DST "  __l2s_looped__: boolean;\n";

    # assign l2s copies
    if (scalar(@varnames) > 0) {
	print DST "ASSIGN\n";
	for (my $i = 0; $i < scalar(@varnames); $i++) {
	    print DST "  next(__l2s_copy__$varnames[$i]) := case __l2s_save__ & !__l2s_saved__: $varnames[$i]; 1: __l2s_copy__$varnames[$i]; esac;\n";
	}
    }

    # handle justice
    if (scalar(@justice) > 0) {
	print DST "ASSIGN\n";
	for (my $i = 0; $i < scalar(@justice); $i++) {
	    print DST "  init(__l2s_justice${i}__) := 0; next(__l2s_justice${i}__) := __l2s_justice${i}__ | (__l2s_save__ | __l2s_saved__) & ($justice[$i]);\n";
	}
    }

    # put parts together
    print DST "ASSIGN\n";
    print DST "  init(__l2s_saved__) := 0;\n";
    print DST "  next(__l2s_saved__) := __l2s_save__ | __l2s_saved__;\n";
    print DST "INVAR\n";
    print DST "  __l2s_looped__ -> __l2s_saved__\n";
    for (my $i = 0; $i < scalar(@justice); $i++) {
	print DST "  & __l2s_justice${i}__\n";
    }
    for (my $i = 0; $i < scalar(@varnames); $i++) {
	print DST "  & __l2s_copy__$varnames[$i] = $varnames[$i]";
	print DST "\n";
    }

    print DST "-- l2s section end\n";
}

#------------------------------------------------------------------------------#
# main
#
{
    my $argno=0;
    my $srcfile;
    my $varfile="";
    my $dstfile;
    
    print "l2s.pl ${VERSION}\n";
    
#------------------------------------------------------------------------------#
# read parameters
#
    if ($ARGV[$argno] eq "-h" || $ARGV[$argno] eq "--help") {
	usage;
    }
    
    $mc="nusmv";
    if ($ARGV[$argno] eq "-nusmv") {
	$mc="nusmv";
	$argno++;
    } elsif ($ARGV[$argno] eq "-csmv") {
	$mc="csmv";
	$argno++;
    }

    if ($ARGV[$argno] ne "") {
	$srcfile=$ARGV[$argno];
	$argno++;
    } else {
	usage;
    }
    
    if ($ARGV[$argno] eq "-var") {
	$argno++;
	if ($ARGV[$argno] ne "") {
	    $varfile=$ARGV[$argno];
	    $argno++;
	} else {
	    usage;
	}
    }

    if ($ARGV[$argno] ne "") {
	$dstfile=$ARGV[$argno];
	$argno++;
    } else {
	usage;
    }
    
    if ($ARGV[$argno] ne "") {
	usage;
    }
    
#------------------------------------------------------------------------------#
# do work
#
    # open files
    open(SRC, $srcfile) or die "error: can't open $srcfile for reading: $!";
    if ($varfile ne "") {
	open(VAR, $varfile) or die "error: can't open $varfile for reading: $!";
    }
    open(DST, ">$dstfile") or die "error: can't open $dstfile for writing: $!";

    # really work
    if ($varfile ne "") {
	get_nocopy_vars;
    }
    parse_and_copy;
    add_l2s;

    # close files
    close(SRC);
    if ($varfile ne "") {
	close(VAR);
    }
    close(DST);

#------------------------------------------------------------------------------#
# clean up and exit
#
    exit 0;
}
