#!/usr/bin/perl

my $VERSION="0.11";
################################################################################
# Name:    interord.pl
# Date:    13.03.2005
#
# Author:  Viktor Schuppan, ETH Zuerich
#
# Description:
#
# interord.pl reorders the variables in src-file such that l2s copies
# come after their respective originals and writes the result to
# dst-file. If a (partial) order is given in ref-file, that order is
# applied. The variables in ref-file (and their copies) come first,
# then the rest from src-file. Variable names are matched exact, i.e.,
# either both src-file and ref-file contain individual bits of a given
# variable or both contain the variable as a whole.
#
# Command-line
#
# interord.pl src-file [-ref ref-file] dst-file
################################################################################

use strict;

#------------------------------------------------------------------------------#
# usage
#
sub usage() {
    print "usage: interord.pl src-file [-ref ref-file] dst-file\n";
    exit 1;
}

#------------------------------------------------------------------------------#
# clean line
#
sub clean_line($) {
    my ($line) = shift;

    # remove comments
    $line =~ s/^--.*$//;

    # approximation to variable syntax only (see NuSMV user manual
    # sect 2.4.1 on input file syntax); remove leading and trailing
    # whitespace
    $line =~ s/^\s*([A-Za-z_][A-Za-z0-9_\$\#\-\\.[\]]*?)\s*$/\1/;

    return $line;
}

#------------------------------------------------------------------------------#
# search l2s copy of srcline in srclines and push onto dstlines if found
#
sub search_l2s_copy($\@\@) {
    my $srcline = shift;
    my $srclinesref = shift;
    my $dstlinesref = shift;
    my $jsrc = 0;
    my $foundl2s = 0;
    my $srcline2;

    while (!$foundl2s && $jsrc < scalar(@$srclinesref)) {
	$srcline2 = @$srclinesref[$jsrc];
	if ($srcline2 =~ /^__l2s_copy__\Q${srcline}\E$/) {
	    push @$dstlinesref, $srcline2;
	    @$srclinesref[$jsrc] = "";
	    $foundl2s = 1;
	}
	$jsrc++;
    }
}

#------------------------------------------------------------------------------#
# main
#
{
    my $argno;
    my $haveref;
    my @srclines;
    my @reflines;
    my @dstlines;
    my @proplines;
    my $isrc;
    my $refline;
    my $srcline;
    my $dstline;
    my $propline;
    
    $argno = 0;
    $haveref = 0;
    
    print "interord.pl $VERSION\n";

#------------------------------------------------------------------------------#
# read parameters
#
    if ($ARGV[$argno] eq "-h" || $ARGV[$argno] eq "--help" || $ARGV[$argno] eq "") {
	usage;
    }

    open(SRCORD, $ARGV[$argno]) or die "Can't open $ARGV[$argno] for reading: $!";
    $argno++;

    if ($ARGV[$argno] eq "-ref") {
	$argno++;
	if ($ARGV[$argno] eq "") {
	    usage;
	}
	open(REFORD, $ARGV[$argno]) or die "Can't open $ARGV[$argno] for reading: $!";
	$haveref = 1;
	$argno++;
    }
    if ($ARGV[$argno] eq "") {
	usage;
    }

    open(DSTORD, ">$ARGV[$argno]") or die "Can't open $ARGV[$argno] for writing: $!";
    $argno++;

    if ($ARGV[$argno] ne "") {
	usage;
    }

#------------------------------------------------------------------------------#
# read and clean input
#
    while ($srcline = <SRCORD>) {
	push @srclines, clean_line($srcline);
    }
    if ($haveref == 1) {
	while ($refline = <REFORD>) {
	    push @reflines, clean_line($refline);
        }
    }
    
#------------------------------------------------------------------------------#
# process variables in reference order first
# 
    for $refline (@reflines) {
        # search for refline in src
	$isrc = 0;
	while ($isrc < scalar(@srclines)) {
	    $srcline = $srclines[$isrc];

	    # match if equal
	    if ($srcline =~ /^\Q${refline}\E$/) {
		push @dstlines, $srcline;
		$srclines[$isrc] = "";
		search_l2s_copy($srcline,@srclines,@dstlines);
	    }
	    $isrc++;
        }
    }

#------------------------------------------------------------------------------#
# process remaining variables
#
    $isrc = 0;
    while ($isrc < scalar(@srclines)) {
        $srcline = $srclines[$isrc];
	if ($srcline ne "" &&
	    # the following are processed separately
	    $srcline !~ /^(LTL_0_SPECF|z)_[\d]+_[\d]+$/ &&
	    $srcline !~ /^__l2s_copy/ &&
	    $srcline !~ /^__l2s_save__$/ &&
	    $srcline !~ /^__l2s_saved__$/ &&
	    $srcline !~ /^__l2s_looped__$/) {

	    push @dstlines, $srcline;
            $srclines[$isrc] = "";
	    search_l2s_copy($srcline,@srclines,@dstlines);
        } elsif ($srcline =~ /^(LTL_0_SPECF|z)_[\d]+_[\d]+$/) {
	    push @proplines, $srcline;
	    $srclines[$isrc] = "";
        }
        $isrc++;
    }
    #
    # use this for clustering of generations: foreach $propline (sort @proplines) {
    #
    foreach $propline (@proplines) {
	push @dstlines, $propline;
	search_l2s_copy($propline,@srclines,@dstlines);
    }

#------------------------------------------------------------------------------#
# write result, close files
#
    
    # these come first
    print DSTORD "__l2s_save__\n__l2s_saved__\n__l2s_looped__\n";

    # write final order
    foreach $dstline (@dstlines) {
        print DSTORD "$dstline\n";
    }

    close SRCORD;
    if ($haveref == 1) {
	close REFORD;
    }
    close DSTORD;

#------------------------------------------------------------------------------#
# clean up and exit
#
    exit 0;
}
