#!/bin/bash

VERSION=0.16b
#
# modified from 0.16 for lmcs journal experiments:
# - invoke ltl2smv explicitly also for ltl,bmc/nusmv/not tight
# - use 1 instead of 0 as index for ltl2smv/ltl2tsmv
# - use "LTLSPEC\n !G 1" instead of "SPEC\n !EG 1" for ltl,bmc
# - modified BASE
#
################################################################################
# Name:    convert.bash
# Date:    21.02.2006
#
# Author:  Viktor Schuppan, ETH Zuerich
#
# Description:
#
# convert.bash takes an smv model and an LTL property and generates an
# smv file suitable for verification with BDD-based invariant checking
# via translation from liveness to safety (-l2s*), BDD-based normal
# LTL model checking (-ltl), or BMC (-bmc). Target model checkers are
# either NuSMV (-nusmv) or Cadence SMV (-csmv). The model should not
# contain any specification itself. The specification is encoded as
# follows
# - ltl,bmc/not tight: ltl2smv
# - ltl/tight: ltl2tsmv
# - l2scgh/not tight: ltl2smv
# - l2scgh/tight: ltl2tsmv
# - l2sbmc/not tight: ltl2l2s
# - l2sbmc/tight: ltl2l2s
# - l2sbmc/maxunroll k: ltl2l2s
# The combination -csmv -bmc is not available. -l2scgh has no
# -maxunroll, -bmc requires -notight.  For -ltl and -bmc a reference
# order is just copied, for l2s* an interleaved order is produced
# taking a reference order into account if one exists. An additional
# file may contain variables to be excluded from saving and comparing
# when using l2s (these are typically input and constant variables but
# potentially also variables not in the cone of influence of the
# property). The model should contain "MODULE main" on a single line
# as the definition of main is moved to the end of the file for -ltl
# and -bmc (see shift_main).
#
# File names are as follows:
# - input
#   - model:                    ${model}.smv
#   - property:                 ${property}.ltl
#   - reference variable order: ${model}.ord
#   - input variables:          ${model}.${property}.icvar
#   - variables not in coi:     ${model}.${property}.ncvar
#   - variables w/ distance > k
#     from coi                  ${model}.${property}.ncvar.${k}
# - output
#   - model:                    ${model}.${property}.${method}.${tight}.${varopt}.${mc}
#   - variable order:           ${model}.${property}.${method}.${tight}.${varopt}.ord
#
# Command-line
#
# convert.bash [-nusmv|-csmv]
#              [-l2sbmc|-l2scgh|-ltl|-bmc]
#              [-tight|-notight|-maxunroll k]
#              [-optnone|-optic|-optcoi|-optbcoi k]
#              model property
################################################################################

# common parameters
BASE=~/l2s4lmcs
source ${BASE}/bin/config.bash

#------------------------------------------------------------------------------#
# usage
#
usage() {
${CAT} <<EOF
usage: convert.bash [-nusmv|-csmv]
                    [-l2sbmc|-l2scgh|-ltl|-bmc]
                    [-tight|-notight|-maxunroll k]
                    [-optnone|-optic|-optcoi|-optbcoi k]
                    model property
       where -csmv has no -bmc,
             -l2scgh and -ltl have no -maxunroll, and
             -bmc must have -notight
EOF
  exit 1
}

#------------------------------------------------------------------------------#
# flatten hierarchy
#
flatten() {
  local modelfile=${1}
  local tmp=${TMPDIR}/tmp.flatten.convert.$$

  #
  # get flat model
  #
  ${NUSMV} -int <<EOF
read_model -i ${modelfile}
flatten_hierarchy
write_flat_model -o ${tmp}
quit
EOF
  mv ${tmp} ${modelfile}

  #
  # rename running and _process_selector_, otherwise some smv variants
  # don't process file again
  #
  ${SED} -e "s/running/__l2s_renamed__running/g" \
         -e "s/_process_selector_/__l2s_renamed___process_selector_/g" \
         <${modelfile} >${tmp}
  mv ${tmp} ${modelfile}
}

#------------------------------------------------------------------------------#
# get initial order
#
get_initial_order() {
  local modelfile=${1}
  local ordfile=${2}

  ${NUSMV} -int <<EOF
read_model -i ${modelfile}
flatten_hierarchy
encode_variables
write_order -o ${ordfile}
quit
EOF
}

#------------------------------------------------------------------------------#
# move declaration of main module to end of file
#
shift_main() {
  local modelfile=${1}
  local tmp1=${TMPDIR}/tmp1.shift_main.convert.$$
  local tmp2=${TMPDIR}/tmp2.shift_main.convert.$$

  ${SED} -n -e ":find_main
               /^[ \t]*MODULE[ \t][ \t]*main$/ {
                 w ${tmp2}
                 n
                 b write_main
               }
               {
                 w ${tmp1}
                 n
                 b find_main
               }
               :write_main
               /^MODULE/ {
                 w ${tmp1}
                 n
                 b write_rest
               }
               {
                 w ${tmp2}
                 n
                 b write_main
               }
               : write_rest
               {
                 w ${tmp1}
                 n
                 b write_rest
               }
               " ${modelfile}
  mv ${tmp1} ${modelfile}
  ${CAT} ${tmp2} >> ${modelfile}
  rm -f ${tmp2}
}

#------------------------------------------------------------------------------#
# add invariant
#
add_invariant() {
  local mc=${1}
  local dstmodelfile=${2}

  if [ "${mc}" == "nusmv" ]; then
      echo -n -e "INVARSPEC\n  !__l2s_looped__\n" >> ${dstmodelfile}
  elif [ "${mc}" == "csmv" ]; then
      echo -n -e "SPEC\n  AG !__l2s_looped__\n" >> ${dstmodelfile}
  fi
}

#------------------------------------------------------------------------------#
# generate order
#
generate_order_l2s() {
  local ordfile=${1}
  local dstmodelfile=${2}
  local dstordfile=${3}
  local tmp=${TMPDIR}/tmp.generate_order_l2s.convert.$$

  # get initial order
  get_initial_order ${dstmodelfile} ${dstordfile}
  # interleave model variables and l2s copies
  if [ -f "${ordfile}" ]; then
    ${INTERORD} ${dstordfile} -ref ${ordfile} ${tmp}
  else
    ${INTERORD} ${dstordfile} ${tmp}
  fi
  mv ${tmp} ${dstordfile}
}

#------------------------------------------------------------------------------#
# adapt syntax for Cadence SMV
#
# != -> ~=
# IVAR -> VAR
#
adapt_syntax_csmv() {
  local modelfile=${1}
  local tmp=${TMPDIR}/tmp.adapt_syntax_csmv.convert.$$

  ${SED} -e "s/!=/~=/g" \
         -e "s/^IVAR/VAR/g" ${modelfile} > ${tmp}
  mv ${tmp} ${modelfile}
}

#------------------------------------------------------------------------------#
# genl2sbmc
#
genl2sbmc() {
  local mc=${1}
  local tight=${2}
  local maxunroll=${3}
  local modelfile=${4}
  local ordfile=${5}
  local propertyfile=${6}
  local varfile=${7}
  local dstmodelfile=${8}
  local dstordfile=${9}
  local tmp=${TMPDIR}/tmp.genl2sbmc.convert.$$

  #
  # convert model
  #

  # flatten model
  cp ${modelfile} ${dstmodelfile}
  flatten ${dstmodelfile}
  # perform translation
  if [ "${varfile}" != "" ]; then
    ${L2S} -${mc} ${dstmodelfile} -var ${varfile} ${tmp}
  else 
    ${L2S} -${mc} ${dstmodelfile} ${tmp}
  fi
  mv ${tmp} ${dstmodelfile}
  # add property part
  echo "!(`${CAT} ${propertyfile}`)" > ${tmp}
  if [ "${tight}" == "tight" ]; then
    ${LTL2L2S} --tight ${tmp} >> ${dstmodelfile}
  elif [ "${tight}" == "notight" ]; then
    ${LTL2L2S} --no-tight ${tmp} >> ${dstmodelfile}
  elif [ "${tight}" == "maxunroll" ]; then
    ${LTL2L2S} --maxunroll ${maxunroll} ${tmp} >> ${dstmodelfile}
  fi
  rm ${tmp}
  # add invariant
  add_invariant ${mc} ${dstmodelfile}
  # generate order
  generate_order_l2s ${ordfile} ${dstmodelfile} ${dstordfile}
}

#------------------------------------------------------------------------------#
# genl2scgh
#
genl2scgh() {
  local mc=${1}
  local tight=${2}
  local maxunroll=${3}
  local modelfile=${4}
  local ordfile=${5}
  local propertyfile=${6}
  local varfile=${7}
  local dstmodelfile=${8}
  local dstordfile=${9}
  local tmp=${TMPDIR}/tmp.genl2scgh.convert.$$

  #
  # convert model
  #

  # flatten model
  cp ${modelfile} ${dstmodelfile}
  flatten ${dstmodelfile}
  # generate Bchi automaton for property
  echo "!(`${CAT} ${propertyfile}`)" > ${tmp}
  if [ "${tight}" == "tight" ]; then
    ${LTL2TSMV} 0 ${tmp} | ${GREP} -v "^MODULE ltl_spec_0" >> ${dstmodelfile}
  elif [ "${tight}" == "notight" ]; then
    ${LTL2SMV} 0 ${tmp} | ${GREP} -v "^MODULE ltl_spec_0" >> ${dstmodelfile}
  fi
  # perform translation
  if [ "${varfile}" != "" ]; then
    ${L2S} -${mc} ${dstmodelfile} -var ${varfile} ${tmp}
  else 
    ${L2S} -${mc} ${dstmodelfile} ${tmp}
  fi
  mv ${tmp} ${dstmodelfile}
  # add invariant
  add_invariant ${mc} ${dstmodelfile}
  # generate order
  generate_order_l2s ${ordfile} ${dstmodelfile} ${dstordfile}
}

#------------------------------------------------------------------------------#
# genltlorbmc
#
genltlorbmc() {
  local mc=${1}
  local tight=${2}
  local maxunroll=${3}
  local modelfile=${4}
  local ordfile=${5}
  local propertyfile=${6}
  local varfile=${7}
  local dstmodelfile=${8}
  local dstordfile=${9}
  local tmp=${TMPDIR}/tmp.genltlorbmc.convert.$$

  #
  # convert model
  #
  cp ${modelfile} ${dstmodelfile}
  shift_main ${dstmodelfile}
  echo "!(`${CAT} ${propertyfile}`)" > ${tmp}
  if [ "${tight}" == "tight" ]; then
    ${LTL2TSMV} 1 ${tmp} | ${GREP} -v "^MODULE ltl_spec_1" >> ${dstmodelfile}
  elif [ "${tight}" == "notight" ]; then
    ${LTL2SMV} 1 ${tmp} | ${GREP} -v "^MODULE ltl_spec_1" >> ${dstmodelfile}
  fi
  echo -n -e "LTLSPEC\n  !G 1\n" >> ${dstmodelfile}
  rm -f ${tmp}

  #
  # generate order  
  #
  if [ -f "${ordfile}" ]; then
    cp ${ordfile} ${dstordfile}
  fi
}

#------------------------------------------------------------------------------#
# main
#
echo "convert.bash ${VERSION}"

#------------------------------------------------------------------------------#
# read parameters
#
if [ "${1}" == "-h" -o "${1}" == "--help" ]; then
  usage
fi

mc=nusmv
if [ "${1}" == "-nusmv" ]; then
  mc=nusmv
  shift
elif [ "${1}" == "-csmv" ]; then
  mc=csmv
  shift
fi

method=l2sbmc
if [ "${1}" == "-l2sbmc" ]; then
  method=l2sbmc
  shift
elif [ "${1}" == "-l2scgh" ]; then
  method=l2scgh
  shift
elif [ "${1}" == "-ltl" ]; then
  method=ltl
  shift
elif [ "${1}" == "-bmc" ]; then
  if [ "${mc}" == "csmv" ]; then
    usage
  else
    method=bmc
    shift
  fi
fi

tight=tight
maxunroll=0
if [ "${1}" == "-tight" ]; then
  if [ "${method}" == "bmc" ]; then
    usage
  else
    tight=tight
    shift
  fi
elif [ "${1}" == "-notight" ]; then
  tight=notight
  shift
elif [ "${1}" == "-maxunroll" ]; then
  if [ "${method}" != "l2sbmc" ]; then
    usage
  else
    tight=maxunroll
    shift
    if [ "${1}" == "" ]; then
      usage
    else
      maxunroll=${1}
      shift
    fi
  fi
fi

varopt=none
if [ "${1}" == "-optnone" ]; then
  varopt=none
  shift
elif [ "${1}" == "-optic" ]; then
  if [ "${method}" != "l2sbmc" -a  "${method}" != "l2scgh" ]; then
    usage
  else
    varopt=ic
    shift
  fi
elif [ "${1}" == "-optcoi" ]; then
  if [ "${method}" != "l2sbmc" -a  "${method}" != "l2scgh" ]; then
    usage
  else
    varopt=coi
    shift
  fi
elif [ "${1}" == "-optbcoi" ]; then
  if [ "${method}" != "l2sbmc" -a  "${method}" != "l2scgh" ]; then
    usage
  else
    varopt=bcoi
    shift
    if [ "${1}" == "" ]; then
      usage
    else
      bcoi=${1}
      shift
    fi
  fi
fi

if [ "${1}" != "" ]; then
  model=${1}
  shift
else
  usage
fi

if [ "${1}" != "" ]; then
  property=${1}
  shift
else
  usage
fi

if [ "${1}" != "" ]; then
  usage
fi 

#------------------------------------------------------------------------------#
# do work
#
modelfile=${model}.smv
ordfile=${model}.ord
propertyfile=${property}.ltl
icvarfile=${model}.${property}.icvar
coivarfile=${model}.${property}.ncvar
bcoivarfile=${model}.${property}.ncvar.${bcoi}

if [ "${tight}" == "maxunroll" ]; then
  basename=${model}.${property}.${method}.${tight}${maxunroll}
else
  basename=${model}.${property}.${method}.${tight}
fi
if [ "${varopt}" == "bcoi" ]; then
  basename="${basename}.${varopt}${bcoi}"
else
  basename="${basename}.${varopt}"
fi
dstmodelfile=${basename}.${mc}
dstordfile=${basename}.ord
dstvarfile=${basename}.var

if [ "${varopt}" == "none" ]; then
  if [ "${method}" == "l2sbmc" -o "${method}" == "l2scgh" ]; then
    echo "" >${dstvarfile}
  fi
elif [ "${varopt}" == "ic" ]; then
  cp ${icvarfile} ${dstvarfile}
elif [ "${varopt}" == "coi" ]; then
  cp ${icvarfile} ${dstvarfile}
  ${CAT} ${coivarfile} >> ${dstvarfile}
elif [ "${varopt}" == "bcoi" ]; then
  cp ${icvarfile} ${dstvarfile}
  ${CAT} ${bcoivarfile} >> ${dstvarfile}
fi

case "${method}" in
  l2scgh)
    genl2scgh ${mc} ${tight} ${maxunroll} ${modelfile} ${ordfile} ${propertyfile} ${dstvarfile} ${dstmodelfile} ${dstordfile}
    ;;
  l2sbmc)
    genl2sbmc ${mc} ${tight} ${maxunroll} ${modelfile} ${ordfile} ${propertyfile} ${dstvarfile} ${dstmodelfile} ${dstordfile}
    ;;
  ltl|bmc)
    genltlorbmc ${mc} ${tight} ${maxunroll} ${modelfile} ${ordfile} ${propertyfile} ${dstvarfile} ${dstmodelfile} ${dstordfile}
    ;;
esac

#
# adapt syntax for Cadence SMV
#
if [ "${mc}" == "csmv" ]; then
  adapt_syntax_csmv ${dstmodelfile}
fi

#------------------------------------------------------------------------------#
# clean up and exit
#
exit 0
