/**CFile***********************************************************************

  FileName    [bmcTableauTimoLTLformula.c]

  PackageName [bmc]

  Synopsis    [Bmc.Tableau module]

  Description [This module contains all the operations related to the
               construction of tableaux for LTL formulas]

  SeeAlso     [bmcGen.c, bmcModel.c, bmcConv.c, bmcVarMgr.c]

  Author      [Timo Latvala]

  Copyright   [2004 Timo Latvala <timo.latvala@hut.fi>. See the file LGPL-2.1 for details.]

******************************************************************************/

#include "bmcUtils.h"
#include "bmcModel.h"
#include "parser/symbols.h"

#include "bmcInt.h"
#include "bmcHash.h"
#include "bmcNodeStack.h"
#include "bmcTableau.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Array for caching f(i) values*/
static be_ptr *bmc_cache_f;
/**Array for caching g(i) values*/
static be_ptr *bmc_cache_g;
/**Array for caching il(i) values*/
static be_ptr *bmc_cache_il;
/**Variable for storing the formula depth*/
static unsigned bmc_tab_past_depth;

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void bmc_cache_init(const int count, const int k, const unsigned pastdepth);

static void bmc_cache_delete();

static unsigned bmc_past_depth(const node_ptr ltl_wff);

static be_ptr bmc_cache_fetch_f(const node_ptr ltl_wff, const int time, const int k, const unsigned pastdepth, 
				hashPtr table);

static be_ptr bmc_cache_fetch_g(const node_ptr ltl_wff, const int time, const int k, const unsigned pastdepth, 
				hashPtr table);

static be_ptr bmc_cache_insert_f(const node_ptr ltl_wff, const int time, const int k, const unsigned pastdepth,
				 hashPtr table, be_ptr result);

static be_ptr bmc_cache_insert_g(const node_ptr ltl_wff, const int time, const int k, const unsigned pastdepth,
				 hashPtr table, be_ptr result);

static be_ptr bmc_cache_insert_il(const int time, const int k, be_ptr result);

static be_ptr bmc_cache_fetch_il(const int time, const int k);

static int
formulaMap(hashPtr table, const node_ptr ltl_wff, unsigned TLcount);

static be_ptr
bmcTimo_tableau_GF_FG_last(const BmcVarsMgr_ptr vars_mgr,
				 const node_ptr ltl_wff, 
				 const int k, 
				 const int l, const unsigned pastdepth, hashPtr table);
static be_ptr
last_g(const BmcVarsMgr_ptr vars_mgr,node_ptr ltl_wff,
       hashPtr table,const int l,const int k, const unsigned pastdepth); 

static be_ptr
last_f(const BmcVarsMgr_ptr vars_mgr,node_ptr ltl_wff,
       hashPtr table,const int l,const int k, const unsigned pastdepth); 

static be_ptr
get_f_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);


static be_ptr
get_G_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_F_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_U_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);
static be_ptr
get_V_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_S_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_T_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_H_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_O_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_ZY_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_g_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth);

static be_ptr
get_el_at_time(const BmcVarsMgr_ptr vars_mgr, const int time, const int k);

static be_ptr
AtMostOnce(const BmcVarsMgr_ptr vars_mgr, const int k);

static be_ptr
Loop(const BmcVarsMgr_ptr vars_mgr, const int k);

static be_ptr
get_il_at_time(const BmcVarsMgr_ptr vars_mgr, const int time, const int k);

static be_ptr
get_loop_exists(const BmcVarsMgr_ptr vars_mgr, const int k);

static be_ptr
bmc_tableauGetEventuallyIL_opt(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff,
			       const int k, const int l, const unsigned pastdepth, hashPtr table);

static be_ptr
bmc_tableauGetGloballyIL_opt(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff,
			     const int k, const int l, const unsigned pastdepth, hashPtr table);


/**AutomaticEnd***************************************************************/



/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [Given a wff expressed in ltl builds the model-independent
  tableau at 'time' of a path formula bounded by \[k, l\]]

  Description        [The function generates the necessary auxilliary predicates
                     (loop, atmostonce) and calls on get_f_at_time to generate
		     the tableau for the ltl formula.]

  SideEffects        []

  SeeAlso            [AtMostOnce, Loop, get_f_at_time]

******************************************************************************/

be_ptr
BmcInt_TimoTableau_GetAtTime(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff,
			     const int time, const int k, const int l)
{
  be_ptr tableau=Be_Truth(BmcVarsMgr_GetBeMgr(vars_mgr));
  bmc_tab_past_depth=bmc_past_depth(ltl_wff);
  bmc_cache_f=0;
  bmc_cache_g=0;
  bmc_cache_il=0;
  
  
  hashPtr formulatable=Bmc_Hash_new_htable();
  /**Allocate variables needed for the bmc translation*/
   Be_RbcManager_Reserve( BmcVarsMgr_GetBeMgr(vars_mgr),
			 /* reserve space for (curvars, inputvars, nextvars) +
			    (state 0) + k+1 times (input, state)*/

  			 (2 + 1 + k) * BmcVarsMgr_GetStateVarsNum(vars_mgr) +
			 (k+1)*BmcVarsMgr_GetInputVarsNum(vars_mgr) 
			  /**Reserve space for el[0..k-1]*/
 			 + k 
			 );
  /**Set up cache*/  
  int count=formulaMap(formulatable, ltl_wff,0);
  bmc_cache_init(count,k,bmc_tab_past_depth);

  /**No or single loopback*/   
  if(!Bmc_Utils_IsAllLoopbacks(l)) {
    tableau=get_f_at_time(vars_mgr,ltl_wff,formulatable,0,k,l,0);
    Bmc_Hash_delete_table(formulatable);
    bmc_cache_delete();
    return tableau;    
  }
 
  /**Generate loop condition*/
  be_ptr loop = Be_And(BmcVarsMgr_GetBeMgr(vars_mgr), 
		       Loop(vars_mgr, k), 
		       AtMostOnce(vars_mgr,k)); 
  /**require that the formula is true in the initial state*/
  tableau= get_f_at_time(vars_mgr,ltl_wff,formulatable,0,k,l,0);
  
  Bmc_Hash_delete_table(formulatable);
  bmc_cache_delete();
  /**loop_condition \wedge tableau*/
  return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr), tableau, loop);
}


/*---------------------------------------------------------------------------*/
/* Definition of static functions functions                                  */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************
   
Synopsis           [Construct f(k) for the GF-,F-,FG-, or G-operator]

Description        [Checks if the il-optimisation is enabled and generates
f(k) accordingly] 

SideEffects        []

SeeAlso            [bmc_tableau_GetEventuallyIL_opt, bmc_tableau_GetGloballyIL_opt]

******************************************************************************/



static be_ptr
bmcTimo_tableau_GF_FG_last(const BmcVarsMgr_ptr vars_mgr,
			   const node_ptr ltl_wff,
			   const int k, const int l, const unsigned pastdepth, 
			   hashPtr table)
{
  be_ptr tableau=Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  
  if(node_get_type(ltl_wff) == OP_FUTURE) {
    if(node_get_type(car(ltl_wff)) == OP_GLOBAL) {
      /**FG p*/
      return bmc_tableauGetGloballyIL_opt(vars_mgr,ltl_wff,k,l,pastdepth,table);
    }
    /** Fp*/
    return bmc_tableauGetEventuallyIL_opt(vars_mgr,ltl_wff,k,l,pastdepth,table);
  }    
  else if (node_get_type(ltl_wff) == OP_GLOBAL) {
      if(node_get_type(car(ltl_wff)) == OP_FUTURE) {
	/**GF p*/ 
	return bmc_tableauGetEventuallyIL_opt(vars_mgr,ltl_wff,k,l,pastdepth,table);
      }      
      /**G p*/
      return bmc_tableauGetGloballyIL_opt(vars_mgr,ltl_wff,k,l,pastdepth,table);
    }
  else {
    nusmv_assert(false);
  }    
  /**never reachable*/
  nusmv_assert(false);
  return tableau;
  
}

/**Function********************************************************************

  Synopsis           [Generate the last f(k) with g] 

  Description        [The function checks which loop setting is active 
                      and genrates f(k) accordingly.]
  
  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr 
last_g(const BmcVarsMgr_ptr vars_mgr,node_ptr ltl_wff,
	hashPtr table,const int l,const int k, const unsigned pastdepth)
{
  nusmv_assert(node_get_type(ltl_wff) == UNTIL || node_get_type(ltl_wff) == RELEASES); 
  if(Bmc_Utils_IsAllLoopbacks(l)) {
    be_ptr temp=Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
    int i;      
    /** f(k):= \vee_{i=0}^{k-1} el(i) & g(i+1)*/    
    for(i=k-1; i>=0; i--) {
      temp=Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr), temp,
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr), 
			get_g_at_time(vars_mgr,ltl_wff,table,i+1,k,l,pastdepth), 
			get_el_at_time(vars_mgr,i,k)));
    }
    return temp;
  }
  else if(Bmc_Utils_IsSingleLoopback(l)) {
    /**f(k):=g(l)*/
    return get_g_at_time(vars_mgr,ltl_wff,table,l,k,l,pastdepth);
  }
  else { /**No loopback*/
    /** f(k):= false*/
    return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  nusmv_assert(false);
  return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [Generate the last f(k) with f]

  Description        [The function checks which loop setting is active 
                      and genrates f(k) accordingly.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr 
last_f(const BmcVarsMgr_ptr vars_mgr,node_ptr ltl_wff,
	 hashPtr table,const int l,const int k, const unsigned pastdepth)
{
  if(Bmc_Utils_IsAllLoopbacks(l)) {
    be_ptr temp=Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
    int i;      
    /** \vee_{i=0}^{k-1} el(i) & f(i+1)*/    
    for(i=k-1; i>=0; i--) {
      temp=Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr), temp,
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,ltl_wff,table,i+1,k,l,pastdepth), 
			get_el_at_time(vars_mgr, i, k)));
    }
    return temp;
  }
  else if(Bmc_Utils_IsSingleLoopback(l)) {
    /**f(k):=f1(l)*/
    return get_f_at_time(vars_mgr,ltl_wff,table,l,k,l,pastdepth);
  }
  else { /**No loopback*/
    /** f(k):= false*/
    return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  nusmv_assert(false);
  return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [map temporal subformulas to an integer, returns the number subformulas
                      with temporal connectives]

  Description        [Stores the nodes of the ltl expression in a table and maps
                     each formula to an integer. Temporal subformulas are numbered
		     from 0...N while all other subformulas are mapped to -2]
		     
  SideEffects        []

  SeeAlso            []

******************************************************************************/

static int
formulaMap(hashPtr table, const node_ptr ltl_wff, unsigned TLcount) 
{
  /**Number of temporal connectives in the formula*/
  /*unsigned TLcount=Bmc_Hash_size(table);*/
  Bmc_Stack_ptr thestack=Bmc_Stack_new_stack();  
  Bmc_Stack_push(thestack, ltl_wff);

  while(Bmc_Stack_size(thestack) > 0) {
    node_ptr formula = Bmc_Stack_pop(thestack);
    
    if (Bmc_Hash_find(table, formula) !=-1) {      
      continue;
    }
    switch(node_get_type(formula)) {
    case TRUEEXP:
    case FALSEEXP:
    case BIT:
    case ARRAY:
    case DOT:
      Bmc_Hash_insert(table, formula, -2);
      break;
    case NOT:
      /* checks out that argument of NOT operator is actually a variable: */
      nusmv_assert( node_get_type(car(formula)) == DOT  ||
		    node_get_type(car(formula)) == BIT || 
		    node_get_type(car(formula)) == ARRAY);
      Bmc_Hash_insert(table, formula, -2);
      Bmc_Stack_push(thestack, car(formula)); 
      break;
    case AND:
    case OR:
    case IFF:
      Bmc_Hash_insert(table, formula, -2);
      Bmc_Stack_push(thestack, cdr(formula)); 
      Bmc_Stack_push(thestack, car(formula)); 
      break;
    case SINCE:
    case TRIGGERED:
    case UNTIL:
    case RELEASES:
      Bmc_Hash_insert(table, formula, TLcount++);
      Bmc_Stack_push(thestack, cdr(formula)); 
      Bmc_Stack_push(thestack, car(formula)); 
      break;
    case OP_PREC:       
    case OP_NOTPRECNOT: 
    case OP_ONCE:       
    case OP_HISTORICAL: 
    case OP_NEXT:
      Bmc_Hash_insert(table, formula, TLcount++);
      Bmc_Stack_push(thestack, car(formula)); 
      break;
    case OP_GLOBAL:
      if (node_get_type(car(formula)) == OP_FUTURE) {
	Bmc_Hash_insert(table, formula, TLcount++);
	Bmc_Stack_push(thestack, car(car(formula))); 
	break;
      }
      Bmc_Hash_insert(table, formula, TLcount++);
      Bmc_Stack_push(thestack, car(formula)); 
      break;
    case OP_FUTURE: /* EVENTUALLY */
      if (node_get_type(car(formula)) == OP_GLOBAL) {
	Bmc_Hash_insert(table, formula, TLcount++);
	Bmc_Stack_push(thestack, car(car(formula))); 
	break;
      }
      Bmc_Hash_insert(table, formula, TLcount++);
      Bmc_Stack_push(thestack, car(formula)); 
      break;
    default:
      nusmv_assert(false);
    }
  }
  Bmc_Stack_delete(thestack);
  return TLcount;
}



/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_g_at_time]

******************************************************************************/

static be_ptr
get_f_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth) 
{
  nusmv_assert((time <= k) && (time>=0) && pastdepth<=bmc_tab_past_depth);
  int data = Bmc_Hash_find(table,ltl_wff);
  nusmv_assert(data!=-1);
  
  if(data>=0) { /**A TL subformula, check cache*/
    if(bmc_cache_fetch_f(ltl_wff,time,k,pastdepth,table)!=0) {
      return bmc_cache_fetch_f(ltl_wff,time,k,pastdepth,table);
    }
  }
  
  switch (node_get_type(ltl_wff)) {
  case TRUEEXP:
    return Be_Truth(BmcVarsMgr_GetBeMgr(vars_mgr));
  case FALSEEXP:
    return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));       
  case AND:
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth));
  case OR:
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		 get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth));
  case IFF:
    return Be_Iff(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth));
  case BIT:
  case DOT: {
    if (time == k) {
      int var_idx = BmcVarsMgr_Name2VarIndex(vars_mgr, ltl_wff);
      if (BmcVarsMgr_IsBeIndexInInputBlock(vars_mgr, var_idx)) {
	/* input vars when time == max_time evaluate to false: */
	return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
      }
    }

    return BmcVarsMgr_Name2Timed(vars_mgr, ltl_wff, time, k);
  }
  case ARRAY:
    if (!Encoding_is_symbol_declared(BmcVarsMgr_GetSymbEncoding(vars_mgr),
				     ltl_wff)) {
      internal_error("Unexpected array node\n");      
    }
    
    if (!Encoding_is_symbol_boolean_var(BmcVarsMgr_GetSymbEncoding(vars_mgr), 
					ltl_wff)) {
      fprintf(nusmv_stderr, "Detected scalar array variable '");
      print_node(nusmv_stderr, ltl_wff);
      fprintf(nusmv_stderr, "'");
      internal_error("Scalar array variable has been found where a boolean "
		     "variable had to be used instead.\n"
		     "This might be due to a bug on your model.");
    }

    if (time == k) {
      int var_idx = BmcVarsMgr_Name2VarIndex(vars_mgr, ltl_wff);
      if (BmcVarsMgr_IsBeIndexInInputBlock(vars_mgr, var_idx)) {
	/* input vars when time == max_time evaluate to false: */
	return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
      }
    }

    return BmcVarsMgr_Name2Timed(vars_mgr, ltl_wff, time, k);
          
  case NOT: { 
    /* checks out that argument of NOT operator is actually a variable: */
    nusmv_assert( node_get_type(car(ltl_wff)) == DOT  ||
		  node_get_type(car(ltl_wff)) == BIT || 
		  node_get_type(car(ltl_wff)) == ARRAY);
    
    if (!Encoding_is_symbol_declared(BmcVarsMgr_GetSymbEncoding(vars_mgr),
				     car(ltl_wff))) {
      internal_error("Unexpected scalar or undefined node\n");      
    }
    
    if ((node_get_type(car(ltl_wff)) == ARRAY) && 
	(!Encoding_is_symbol_boolean_var(
					 BmcVarsMgr_GetSymbEncoding(vars_mgr), car(ltl_wff)))) {
      fprintf(nusmv_stderr, "Detected scalar array variable '");
      print_node(nusmv_stderr, car(ltl_wff));
      fprintf(nusmv_stderr, "'");
      internal_error("Scalar array variable has been found where a boolean "
		     "variable had to be used instead.\n"
		     "This might be due to a bug on your model.");
    }
    
    if (time == k) { 
      int var_idx = BmcVarsMgr_Name2VarIndex(vars_mgr, car(ltl_wff));
      if (BmcVarsMgr_IsBeIndexInInputBlock(vars_mgr, var_idx)) {
	/* input vars when time == max_time evaluate to false: */
	return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
      }
    }
    
    return Be_Not(BmcVarsMgr_GetBeMgr(vars_mgr), 
		  BmcVarsMgr_Name2Timed(vars_mgr, car(ltl_wff), time, k));
    
  }
  case OP_NEXT: { 
    if(time<k) {
      /**f(time):=f(time+1)*/
      return get_f_at_time(vars_mgr,car(ltl_wff),table,time+1,k,l,pastdepth);
    }
    /**time=k*/
    return last_f(vars_mgr,car(ltl_wff),table,l,k,
		    (pastdepth<bmc_past_depth(ltl_wff) ? pastdepth+1: bmc_past_depth(ltl_wff)));  
  }
  case OP_NOTPRECNOT:
  case OP_PREC: return get_ZY_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth);  
  case OP_GLOBAL: return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					    get_G_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case OP_FUTURE: return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					    get_F_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case UNTIL: return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					get_U_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case RELEASES:return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					  get_V_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case SINCE: return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					get_S_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case TRIGGERED: return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					    get_T_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case OP_ONCE:  return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
					   get_O_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));
  case OP_HISTORICAL: return bmc_cache_insert_f(ltl_wff,time,k,pastdepth,table,
						get_H_at_time(vars_mgr,ltl_wff,table,time,k,l,pastdepth));

  case IMPLIES:
      internal_error("'Implies' should had been nnf-ed away!\n");
  case ATOM:
  case EX:
  case AX:
  case EF:
  case AF:
  case EG:
  case AG:
  case EE:
  case AA:
  case EBF:
  case EBG:
  case ABF:
  case ABG:
  case BUNTIL:
  case MMIN:
  case MMAX:
  case APATH:
  case EPATH:
    internal_error( "f: Unexpected CTL operator, node type %d\n",
		      node_get_type(ltl_wff) );
  default:
    /* no other type are available here: */
    nusmv_assert(false);
  
  }
  nusmv_assert(false);
  return 0; /**Should never be reached*/  
}
/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the GLOBALLY 
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_G_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(time>=0 && time<=k);
  nusmv_assert(node_get_type(ltl_wff)==OP_GLOBAL);
  if (time<k) {
    if(node_get_type(car(ltl_wff)) == OP_FUTURE) {
      /**f(i):=f(k)*/
      return get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth);
    }
    /*f(time):=f1(time)\wedge f(time+1)*/
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		  get_f_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth));
  }
  /**time=k*/ 
  if(node_get_type(car(ltl_wff)) == OP_FUTURE) {
    /**GF p only depends on f(k,pastdepth(ltl_wff))*/
    return bmcTimo_tableau_GF_FG_last(vars_mgr,ltl_wff,k,l,bmc_past_depth(ltl_wff),table);
  }
  /*f(k):=f1(k)\wedge last f(k)*/
  return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		(pastdepth<bmc_past_depth(ltl_wff)) ? 
		last_f(vars_mgr,ltl_wff,table,l,k,pastdepth+1) :
		bmcTimo_tableau_GF_FG_last(vars_mgr,ltl_wff,k,l,bmc_past_depth(ltl_wff),table) );
}

/**Function********************************************************************

Synopsis           [Genrates a boolean expression which is true iff the ltl
formula ltl_wff is true at time, handles the FINALLY 
operator]

Description        [The function recursively traverses the formula and genrates
the needed boolean expression.]

SideEffects        []

SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_F_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(time>=0 && time<=k);
  nusmv_assert(node_get_type(ltl_wff)==OP_FUTURE);
  if (time<k) {
    if(node_get_type(car(ltl_wff)) == OP_GLOBAL) {
      /**f(i):=f(k)*/
      return get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth);	   
    }
    /*f(time):=f1(time)\vee f(time+1)*/
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		 get_f_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth));
  }  
  /**time=k*/
  if(node_get_type(car(ltl_wff)) == OP_GLOBAL) {
    /**FG p only depends on f(k,pastdepth(ltl_wff))*/
    return bmcTimo_tableau_GF_FG_last(vars_mgr,ltl_wff,k,l,bmc_past_depth(ltl_wff),table);
  }
  /**f(k)=f1(k)\vee last f(k)*/
  return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
	       get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
	       (pastdepth<bmc_past_depth(ltl_wff)) ? 
	       last_f(vars_mgr,ltl_wff,table,l,k,pastdepth+1) :
	       bmcTimo_tableau_GF_FG_last(vars_mgr,ltl_wff,k,l,bmc_past_depth(ltl_wff),table) );        
}

/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the UNTIL 
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_U_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  const unsigned pd=bmc_past_depth(ltl_wff);
  nusmv_assert(time>=0 && time<=k);
  nusmv_assert(node_get_type(ltl_wff)==UNTIL);
  if (time<k) {
    /*f(time):=f2(time)\vee (f1(time)\wedge f(time+1))*/
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			get_f_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth)));
  }     
  /**time=k*/
  if (pastdepth<bmc_past_depth(ltl_wff)) {
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			last_f(vars_mgr,ltl_wff,table,l,k,pastdepth+1)));
  }
  /**pastdepth==n_\psi*/
  return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
	       get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pd),
	       Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		      get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pd),
		      last_g(vars_mgr,ltl_wff,table,l,k,pd)));
}

/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the RELEASE 
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/


static be_ptr
get_V_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  const unsigned pd=bmc_past_depth(ltl_wff);
  nusmv_assert(time>=0 && time<=k);
  nusmv_assert(node_get_type(ltl_wff)==RELEASES);

  if (time<k) {
    /*f(time):=f2(time)\wedge (f1(time)\vee f(time+1))*/
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			get_f_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth)));
  }
  /**time=k*/
  if (pastdepth<bmc_past_depth(ltl_wff)) {
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			last_f(vars_mgr,ltl_wff,table,l,k,pastdepth+1)));
  }
  /**pastdepth==n_\psi*/
  return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pd),
		Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		      get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pd),
		      last_g(vars_mgr,ltl_wff,table,l,k,pd)));
}

/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the SINCE 
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_S_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(node_get_type(ltl_wff)==SINCE && time<k+1);
  if(time==0 && pastdepth==0) {
    return get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth); 
  }
  else if (pastdepth==0 && time>0 && time<=k ) {
    /*f(i):=f2(i)\vee (f1(i) \wedge f(i-1))*/
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth)));
  }
  else if (pastdepth>0 && time>1 && time<=k) {
    /**f(i,d):=f2(i,d)\vee(f1(i,d)\wedge (el(i)->f(k,d-1), f(i-1,d)))*/ 
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			Be_Ite(BmcVarsMgr_GetBeMgr(vars_mgr),
			       get_el_at_time(vars_mgr,time-1,k),
			       get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1),
			       get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth))));
  } 
  else if(pastdepth>0 && time==1) {
    /**f(1,d):=f2(i,d)\vee(f1(i,d)\wedge f(k,d-1))*/ 
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		 Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1)));    
  }
  nusmv_assert(false);
  return 0; /**Should not be reached*/
}

/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the TRIGGER 
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_T_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(node_get_type(ltl_wff)==TRIGGERED && time<k+1);
  if(time==0 && pastdepth==0) {
    return get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth); 
  }
  else if (pastdepth==0 && time>0 && time<=k ) {
    /*f(time):=f2(time)\wedge (f1(time)\vee f(time-1))*/
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth)));
  }
  else if (pastdepth>0 && time>1 && time<=k) {
    /**f(i,d):=f2(i,d)\wedge(f1(i,d)\vee (el(i)->f(k,d-1)),f(i-1,d))*/ 
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			Be_Ite(BmcVarsMgr_GetBeMgr(vars_mgr),
			       get_el_at_time(vars_mgr,time-1,k),
			       get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1),
			       get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth)))); 
  } 
  else if(pastdepth>0 && time==1) {
    /**f(i,d):=f2(1,d)\wedge(f1(1,d)\vee f(k,d-1))*/ 
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
		  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
			get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1)));
  }
  nusmv_assert(false);
  return 0; /**Should not be reached*/
}

/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the HISTORICALLY
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_H_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(node_get_type(ltl_wff)==OP_HISTORICAL && time<k+1);
  if(time==0 && pastdepth==0) {
    /**f1(i)*/
    return get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth);   
  }
  else if (pastdepth==0 && time>0 && time<=k ) {
    /**f(i):=f1(i)\wedge f(i-1)*/
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		  get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth));
  }
  else if (pastdepth>0 && time>1 && time<=k) {
    /**f(i,d):= f1(i,d)\wedge (el(i)->f(k,d-1),f(i-1,d))*/
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		  Be_Ite(BmcVarsMgr_GetBeMgr(vars_mgr),
			 get_el_at_time(vars_mgr,time-1,k),
			 get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1),
			 get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth)));
  }
  else if(pastdepth>0 && time==1) {
    /**f(1,d):= f1(i,d)\wedge f(k,d-1)*/
    return Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		  get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1));      
  }
  nusmv_assert(false);
  return 0; /**Should not be reached*/
}

/**Function********************************************************************

  Synopsis           [Genrates a boolean expression which is true iff the ltl
                      formula ltl_wff is true at time, handles the ONCE 
		      operator]

  Description        [The function recursively traverses the formula and genrates
                      the needed boolean expression.]

  SideEffects        []

  SeeAlso            [get_f_at_time, get_g_at_time]

******************************************************************************/

static be_ptr
get_O_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(node_get_type(ltl_wff)==OP_ONCE && time<k+1);
  if(time==0 && pastdepth==0) {
    /**f1(i)*/
    return get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth);   
  }
  else if (pastdepth==0 && time>0 && time<=k ) {
    /**f(i):=f1(i)\vee f(i-1)*/
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		 get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth));
  }
  else if (pastdepth>0 && time>1 && time<=k) {
    /**f(i,d):= f1(i,d)\vee (el(i)->f(k,d-1),f(i-1,d))*/
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		 Be_Ite(BmcVarsMgr_GetBeMgr(vars_mgr),
			get_el_at_time(vars_mgr,time-1,k),
			get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1),
			get_f_at_time(vars_mgr,ltl_wff,table,time-1,k,l,pastdepth)));
  }
  else if(pastdepth>0 && time==1) {
    /**f(1,d):= f1(1,d)\vee f(k,d-1)*/
    return Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
		 get_f_at_time(vars_mgr,ltl_wff,table,k,k,l,pastdepth-1));
  }
  nusmv_assert(false);
  return 0; /**Should not be reached*/
}

static be_ptr
get_ZY_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{
  nusmv_assert(node_get_type(ltl_wff)==OP_PREC || node_get_type(ltl_wff)==OP_NOTPRECNOT);
  nusmv_assert(time<k+1);
  
  if(time==0 && pastdepth==0 && node_get_type(ltl_wff)==OP_NOTPRECNOT) {
    return Be_Truth(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  if(time==0 && pastdepth==0) {
    return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  else if (pastdepth==0 && time>0 && time<k+1 ) {
    /**f(i):=f1(i-1)*/
    return get_f_at_time(vars_mgr,car(ltl_wff),table,time-1,k,l,pastdepth);
  }
  else if (pastdepth>0 && time>1 && time<k+1) {
    /**f(i,d):=el(i)->f1(i-1,d), f(k,d-1)*/
    return Be_Ite(BmcVarsMgr_GetBeMgr(vars_mgr),
	   get_el_at_time(vars_mgr,time-1,k),
	   get_f_at_time(vars_mgr,car(ltl_wff),table,k,k,l,pastdepth-1),
	   get_f_at_time(vars_mgr,car(ltl_wff),table,time-1,k,l,pastdepth));
  }
  else if(pastdepth>0 && time==1) {
    return get_f_at_time(vars_mgr,car(ltl_wff),table,k,k,l,pastdepth-1);
  }
  nusmv_assert(false);
  return 0; /**Should not be reached*/
}


/**Function********************************************************************

  Synopsis           []

  Description        [Returns a pointer to the g_i(time) variable]

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr
get_g_at_time(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff, hashPtr table,
	      const int time, const int k, const int l, const unsigned pastdepth)
{

  nusmv_assert((time < k+1) && (time>=0));
  nusmv_assert(pastdepth==bmc_past_depth(ltl_wff));
  int data = Bmc_Hash_find(table, ltl_wff);
  nusmv_assert(data!=-1);

  if(data>=0) { /**A TL subformula, check cache*/
    if(bmc_cache_fetch_g(ltl_wff,time,k,pastdepth,table)!=0) {
      return bmc_cache_fetch_g(ltl_wff,time,k,pastdepth,table);
    }
  }
  
  switch (node_get_type(ltl_wff)) {
  case TRUEEXP:
  case FALSEEXP:
  case DOT:
  case NOT:
  case AND:
  case OR:
  case IFF:
  case OP_NEXT:
    /**The next operator and the normal 
     * Boolean connectives do not use g-variables*/
    nusmv_assert(false);
    break;
    
  case OP_GLOBAL:
    nusmv_assert(false);
    if (time<k) {
      /*g(time):=f1(time)\wedge g(time+1)*/
      return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
				Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
				       get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
					     get_g_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth)));
    }
    /**time=k*/
    return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
			      get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth));
   
  case OP_FUTURE: /* EVENTUALLY */
    nusmv_assert(false);
    if (time<k) {
      /*g(time):=f1(time)\vee g(time+1))*/
      return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
				Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
				      get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
				      get_g_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth)));
    }     
    /**time=k*/
    return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
			      get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth));
    
  case UNTIL:
    if (time<k) {
	/*g(time):=f2(time)\vee (f1(time)\wedge g(time+1))*/
      return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
				Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
				      get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
				      Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
					     get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
					     get_g_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth))));
    }     
    /**time=k*/
    return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
			      get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth));
    
  case RELEASES:
    if (time<k) {
      /*g(time):=f2(time)\wedge (f1(time)\vee g(time+1))*/
      return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
				Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
				       get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth),
				       Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
					     get_f_at_time(vars_mgr,car(ltl_wff),table,time,k,l,pastdepth),
					     get_g_at_time(vars_mgr,ltl_wff,table,time+1,k,l,pastdepth))));
    }
    /**time=k*/
    return bmc_cache_insert_g(ltl_wff,time,k,pastdepth,table,
			      get_f_at_time(vars_mgr,cdr(ltl_wff),table,time,k,l,pastdepth));

  case IMPLIES:
    internal_error("'Implies' should had been nnf-ed away!\n");
    
  case ATOM:
  case ARRAY:
  case EX:
  case AX:
  case EF:
  case AF:
  case EG:
  case AG:
  case EE:
  case AA:
  case EBF:
  case EBG:
  case ABF:
  case ABG:
  case BUNTIL:
  case MMIN:
  case MMAX:
  case APATH:
  case EPATH:
    internal_error( "g: Unexpected CTL operator, node type %d\n",
		    node_get_type(ltl_wff) );
    
  default:
    /* no other type are available here: */
    nusmv_assert(false);
  }
  
  nusmv_assert(false);
  return 0; /**Should never be reached*/
}

/**Function********************************************************************

  Synopsis           [Returns a pointer to the el(time) variable]

  Description        [The variables el(time) describe if the state s_time 
                      should be equivalent with s_k]

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr
get_el_at_time(const BmcVarsMgr_ptr vars_mgr, const int time, const int k)
{
  nusmv_assert((time < k) && (time>=0));
  /*  int offset=(3+BmcVarsMgr_GetMaxTime(vars_mgr))*BmcVarsMgr_GetStateVarsNum(vars_mgr) + 
      (BmcVarsMgr_GetMaxTime(vars_mgr)+1)*BmcVarsMgr_GetInputVarsNum(vars_mgr) + time;*/

     int offset=(3+k)*BmcVarsMgr_GetStateVarsNum(vars_mgr) + 
       (k+1)*BmcVarsMgr_GetInputVarsNum(vars_mgr) + time;

  return Be_Index2Var(BmcVarsMgr_GetBeMgr(vars_mgr), offset);
}


/**Function********************************************************************

  Synopsis           [Creates an expression which allows at most one el_i to be true]

  Description        []

  SideEffects        []

  SeeAlso            [get_el_at_time]

******************************************************************************/
static be_ptr
AtMostOnce(const BmcVarsMgr_ptr vars_mgr, const int k)
{  

  int i;
  be_ptr smaller_exists_i = Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  be_ptr at_most_once = Be_Truth(BmcVarsMgr_GetBeMgr(vars_mgr));


  for(i=1; i<k; i++) {
    /*smaller_exists_i:= smaller_exists_{i-1} \vee el_{i-1}*/
    smaller_exists_i = Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr), 
			     smaller_exists_i, 
			     get_el_at_time(vars_mgr, i-1, k));

    /*bad_index_i:=smaller_exists_i -> \neg el(time)*/
    be_ptr bad_index_i=Be_Implies(BmcVarsMgr_GetBeMgr(vars_mgr),
				  smaller_exists_i,
				  Be_Not(BmcVarsMgr_GetBeMgr(vars_mgr),
					 get_el_at_time(vars_mgr, i, k)));
    /**\wedge_{i=1}^{k-1} SE_i -> \neg el_i*/
    at_most_once= Be_And(BmcVarsMgr_GetBeMgr(vars_mgr), bad_index_i, at_most_once);
  }
  return at_most_once;
}

/**Function********************************************************************

  Synopsis           [Creates the expression: \wedge_{i=0}^{k-1} el_i=> (s_i <=> s_k)]

  Description        [The expression when coupled with AtMostOnce forces state i
                      to be equivalent with state k, if el_i is true.

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr 
Loop(const BmcVarsMgr_ptr vars_mgr, const int k)
{
  be_ptr loop_constraints = Be_Truth(BmcVarsMgr_GetBeMgr(vars_mgr));
  int i;
  
  /**set \wedge_{i=0}^{k-1} el(i) => Equiv(s_i, s_k)*/
  for(i=k; i--; ) {
    loop_constraints=Be_And(BmcVarsMgr_GetBeMgr(vars_mgr), loop_constraints,
			    Be_Implies(BmcVarsMgr_GetBeMgr(vars_mgr),
				       get_el_at_time(vars_mgr,i,k),
				       Bmc_Tableau_GetLoopCondition(vars_mgr,k,i)));
    
  }
  
  return loop_constraints;
}

/**Function********************************************************************

  Synopsis           [Returns a pointer to the il(time) variable]

  Description        [The il(i) variable describes if the state 'i' is a 
                      a state of the loop.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr
get_il_at_time(const BmcVarsMgr_ptr vars_mgr, const int time, const int k) 
{
  /**The variables are only allocated if the il-optimisation is active*/
  nusmv_assert(time<=k && time>0);

  /**check cache*/
  if(bmc_cache_fetch_il(time,k)!=0) {
    return bmc_cache_fetch_il(time,k);
  }

  if(time==1) {
    return bmc_cache_insert_il(time, k, get_el_at_time(vars_mgr,0,k));
  }

  return bmc_cache_insert_il(time, k,
			  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr), 
				get_il_at_time(vars_mgr,time-1,k),
				get_el_at_time(vars_mgr,time-1,k))
	       );
}

/**Function********************************************************************

  Synopsis           [Returns a pointer to the le variable]

  Description        [The le variable is true if a loop exists.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static be_ptr
get_loop_exists(const BmcVarsMgr_ptr vars_mgr, const int k)
{
/**The variables are only allocated if the il-optimisation is active*/
  if(k==0) {
    return Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  return get_il_at_time(vars_mgr,k,k);  
}


/**Function********************************************************************

  Synopsis           [Returns an expression which initialises f(k+1) for 
                      an F or an GF formula when we use the il-optimisation.]

  Description        [Creates the expression f(k+1):=\vee_{i=1}^k il(i)\wedge f1(i)]

  SideEffects        []

  SeeAlso            []

******************************************************************************/


static be_ptr
bmc_tableauGetEventuallyIL_opt(const BmcVarsMgr_ptr vars_mgr,
				   const node_ptr ltl_wff,
				   const int k, 
				   const int l, const unsigned pastdepth, hashPtr table)
{
  nusmv_assert(pastdepth==bmc_past_depth(ltl_wff));
  be_ptr tableau=Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  int i;
  
  if(Bmc_Utils_IsAllLoopbacks(l)) {
    /**set f(k):= \vee_{i=1}^k il(i) \wedge f1(i) */
    for (i=1; i<=k; i++) {	
      tableau=Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		    Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
			   get_il_at_time(vars_mgr,i,k),			
			   (node_get_type(ltl_wff) == OP_GLOBAL) ?
			   /**we're dealing with GF p, skip to f_p(i)*/
			   get_f_at_time(vars_mgr, car(car(ltl_wff)), table, i, k, l, pastdepth) :
			   /**normal case: F p*/
			   get_f_at_time(vars_mgr, car(ltl_wff), table, i, k, l, pastdepth)),
		    tableau);
		     
    }
  } 
  else if(Bmc_Utils_IsSingleLoopback(l)) {
    /**Set f(k):=\vee_{i=l}^k f1(i)*/
    for (i=l; i<=k; i++) {
      tableau=Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
		 (node_get_type(ltl_wff) == OP_GLOBAL) ?
		 /**we're dealing with GF p, skip to f_p(i)*/
		 get_f_at_time(vars_mgr, car(car(ltl_wff)), table, i, k, l, pastdepth) :
		 /**normal case: G p*/
		 get_f_at_time(vars_mgr, car(ltl_wff), table, i, k, l, pastdepth),
		 tableau);	
    }
  }
  else { /**No loopback*/
    tableau=Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  return tableau;

}

/**Function********************************************************************

  Synopsis           [Returns an expression which initialises f(k+1) for 
                      a 'globally' or an FG formula when we use the il-optimisation.]

  Description        [Creates the expression f(k+1):=le\wedge \wedge_{i=1}^k \neg il(i)\vee f1(i)]

  SideEffects        []

  SeeAlso            []

******************************************************************************/




static be_ptr
bmc_tableauGetGloballyIL_opt(const BmcVarsMgr_ptr vars_mgr, const node_ptr ltl_wff,
			      const int k, const int l, const unsigned pastdepth, hashPtr table)
{
  nusmv_assert(pastdepth==bmc_past_depth(ltl_wff));
  be_ptr tableau=Be_Truth(BmcVarsMgr_GetBeMgr(vars_mgr));
  int i;
  
  if(Bmc_Utils_IsAllLoopbacks(l)) {
    /**set f(k):= le\wedge \wedge_{i=1}^k \neg il(i) \vee f1(i) */
    for (i=1; i<=k; i++) {	
      tableau=Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  Be_Or(BmcVarsMgr_GetBeMgr(vars_mgr),
			Be_Not(BmcVarsMgr_GetBeMgr(vars_mgr), 
			       get_il_at_time(vars_mgr,i,k)),
			(node_get_type(ltl_wff) == OP_FUTURE) ?
			/**we're dealing with FG p, skip to f_p(i)*/
			get_f_at_time(vars_mgr, car(car(ltl_wff)), table, i, k, l, pastdepth) :
			/**normal case G p*/
			get_f_at_time(vars_mgr, car(ltl_wff), table, i, k, l, pastdepth)),
		  tableau);
      
    }
    tableau=Be_And(BmcVarsMgr_GetBeMgr(vars_mgr), tableau,
		   get_loop_exists(vars_mgr,k));
  } 
  else if(Bmc_Utils_IsSingleLoopback(l)) {
    /**Set f(k):=\wedge_{i=l}^k f1(i)*/
    for (i=l; i<=k; i++) {
      tableau=Be_And(BmcVarsMgr_GetBeMgr(vars_mgr),
		  (node_get_type(ltl_wff) == OP_FUTURE) ?
		  /**we're dealing with GF p, skip to f_p(i)*/
		  get_f_at_time(vars_mgr, car(car(ltl_wff)), table, i, k, l,pastdepth) :
		  /**normal case: F p*/
		  get_f_at_time(vars_mgr, car(ltl_wff), table, i, k, l,pastdepth),
		  tableau);	
    }
  }
  else { /**No loopback*/
    tableau=Be_Falsity(BmcVarsMgr_GetBeMgr(vars_mgr));
  }
  return tableau;
}

/**Function********************************************************************

  Synopsis           [Initialises the chache used to store f_i(time) and g_(time) 
                      values.]

  Description        [The function allocates an array of size (k+1)*count for 
                      both f and g. The array is used to cache values of f_i(time)
		      and g_i(time). Only values for temporal formulas are stored.]

  SideEffects        []

  SeeAlso            [bmc_delete_cache]

******************************************************************************/


static void bmc_cache_init(const int count, const int k, const unsigned pastdepth)
{
  nusmv_assert(count>=0 && k>=0);
  int i;
  bmc_cache_f=(be_ptr *) malloc(sizeof(be_ptr)* count * (k+1) * (pastdepth+1));
  bmc_cache_g=(be_ptr *) malloc(sizeof(be_ptr)* count * (k+1) * (pastdepth+1));
  
  for(i=(k+1)*count*(pastdepth+1); i--; ) {
    bmc_cache_f[i]=0;
    bmc_cache_g[i]=0;
  }

  bmc_cache_il=(be_ptr *) malloc(sizeof(be_ptr)* k);
  for(i=k; i--; ) {
    bmc_cache_il[i]=0;
  }

}

/**Function********************************************************************

  Synopsis           [Frees the arrays used by the cache]

  Description        []

  SideEffects        []

  SeeAlso            [bmc_init_cache]

******************************************************************************/

static void bmc_cache_delete()
{
  free(bmc_cache_f);
  free(bmc_cache_g);
  free(bmc_cache_il);
}

static be_ptr bmc_cache_fetch_f(const node_ptr ltl_wff, const int time, const int k,const unsigned pastdepth,
				hashPtr table)

{
  nusmv_assert(time<k+1 && time>=0 && pastdepth <= bmc_tab_past_depth);
  int data=Bmc_Hash_find(table,ltl_wff);
  /**It is an error if the formula is not in the table or ltl_wff is not a temporal operator*/
  nusmv_assert(data!=-1 || data!=-2);
    
  if(bmc_cache_f!=0)    
    return bmc_cache_f[(pastdepth+(bmc_tab_past_depth+1)*data)*(k+1)+time];
  return 0;
}

static be_ptr bmc_cache_fetch_g(const node_ptr ltl_wff, const int time, const int k, 
				const unsigned pastdepth,hashPtr table) 

{
  nusmv_assert(time<k+1 && time>=0 && pastdepth <= bmc_tab_past_depth);
  int data=Bmc_Hash_find(table,ltl_wff);
  /**It is an error if the formula is not in the table or ltl_wff is not a temporal operator*/
  nusmv_assert(data!=-1 || data!=-2);

  if(bmc_cache_g!=0)
    return bmc_cache_g[(pastdepth+(bmc_tab_past_depth+1)*data)*(k+1)+time];
  return 0;

}

static be_ptr bmc_cache_insert_f(const node_ptr ltl_wff, const int time, const int k, const unsigned pastdepth,
				 hashPtr table, be_ptr result)

{
  nusmv_assert(time<k+1 && time>=0 && pastdepth <= bmc_tab_past_depth);
  int data=Bmc_Hash_find(table,ltl_wff);
  /**It is an error if the formula is not in the table or ltl_wff is not a temporal operator*/
  nusmv_assert(data!=-1 || data!=-2);

  if(bmc_cache_f!=0) { 
    bmc_cache_f[(pastdepth+(bmc_tab_past_depth+1)*data)*(k+1)+time]=result;
  }
  return result;
}

static be_ptr bmc_cache_insert_g(const node_ptr ltl_wff, const int time, const int k, const unsigned pastdepth,
				 hashPtr table, be_ptr result)
			    
{
  nusmv_assert(time<k+1 && time>=0 && pastdepth <= bmc_tab_past_depth);
  int data=Bmc_Hash_find(table,ltl_wff);
  /**It is an error if the formula is not in the table or ltl_wff is not a temporal operator*/
  nusmv_assert(data!=-1 || data!=-2);

  if(bmc_cache_g!=0) { 
    bmc_cache_g[(pastdepth+(bmc_tab_past_depth+1)*data)*(k+1)+time]=result;
  }
  return result;
}

static be_ptr 
bmc_cache_insert_il(const int time, const int k, be_ptr result)
{
  nusmv_assert(time>=1 && time<=k);

  if(bmc_cache_il!=0) {
    bmc_cache_il[time-1]=result;
  }
  return result;
}

static be_ptr 
bmc_cache_fetch_il(const int time, const int k)
{
  nusmv_assert(time>0 && time<=k);
  
  if(bmc_cache_il!=0) {
    return bmc_cache_il[time-1];
  }
  return 0;
}


/**Function********************************************************************

  Synopsis           [Computes the maximum nesting depth of past operators in PLTL formula]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/

static unsigned
bmc_past_depth(const node_ptr ltl_wff)
{
  switch(node_get_type(ltl_wff)) {
  case TRUEEXP:
  case FALSEEXP:
  case BIT: 
  case ARRAY:
  case DOT:
    return 0;
  case NOT:
    nusmv_assert( node_get_type(car(ltl_wff)) == DOT  ||
		  node_get_type(car(ltl_wff)) == BIT || 
		  node_get_type(car(ltl_wff)) == ARRAY );
    return 0;
  case AND:
  case OR:
  case IFF:
  case UNTIL:
  case RELEASES: {
    unsigned left=bmc_past_depth(car(ltl_wff));
    unsigned right=bmc_past_depth(cdr(ltl_wff));
    return (left>right)? left : right;
  }
    
  case SINCE:
  case TRIGGERED: {
    unsigned left=bmc_past_depth(car(ltl_wff));
    unsigned right=bmc_past_depth(cdr(ltl_wff));
    return (left>right)? left+1 : right+1;
  } 
    
  case OP_NEXT:
  case OP_GLOBAL:
  case OP_FUTURE: 
    return bmc_past_depth(car(ltl_wff));
    
  case OP_PREC:       
  case OP_NOTPRECNOT: 
  case OP_ONCE:       
  case OP_HISTORICAL: 
    return bmc_past_depth(car(ltl_wff)) + 1;
    
  default:
    nusmv_assert(false);
  }  
  return 0;
}
