/*2004 Timo Latvala <timo.latvala@hut.fi>. See the file LGPL-2.1 for details*/
/**A hashtable for (node, unsigned) pairs*/

#include <stdlib.h>
#include "bmcHash.h"
#include "utils/utils.h"

/**Constant declarations*/

#define HASH_TABLE_SIZE 127
#define HASH_MAX_DENSITY 0.5


/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int table_hash_fun ARGS((node_ptr, int));
static int node_eq_fun ARGS((node_ptr, node_ptr));
static int node_neq_fun ARGS((node_ptr, node_ptr));
static int find ARGS((hashPtr table, node_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Create a new hash_table*/

hashPtr Bmc_Hash_new_htable()
{
  int i;
  hashPtr hash_table = (hashPtr) malloc(sizeof(struct htable));
  hash_table->alloc=HASH_TABLE_SIZE;
  hash_table->occupied=0;
  hash_table->table = (struct table_pair *) malloc(sizeof(struct table_pair) * hash_table->alloc);
  if (hash_table->table == NULL) {
    fprintf(stderr, "new_assoc: Out of Memory\n");
    exit(1);
  }
  for(i=hash_table->alloc; i--; ) {
    hash_table->table[i].key=0;
  }
  return hash_table;
}

/**Find a node in the table
 * @return BMC_HASH_NOTFOUND if the node is not present 
 */

int Bmc_Hash_find(hashPtr table, node_ptr node) 
{ 
  int i=find(table, node);
  if(table->table[i].key) 
    return table->table[i].data;
  return BMC_HASH_NOTFOUND;
}

/**@return the number of occupied slots*/
unsigned Bmc_Hash_size(hashPtr hash) 
{
  return hash->occupied;
}

/**insert a new element in the table*/
void Bmc_Hash_insert (hashPtr table, node_ptr key, int data)
{
  int i=find(table, key);
  if(table->table[i].key)
    return; /**The node already is in the table*/
  if( (table->occupied+1)/table->alloc > 0.5 ) { /*a rehash is needed*/    
    unsigned j;
    struct table_pair *temp= table->table;
    table->alloc=(table->alloc)*2;
    table->table= (struct table_pair *) 
      malloc(sizeof(struct table_pair) * table->alloc);
    nusmv_assert(table->table != NULL);
    
    /**reset new table*/
    for(j=table->alloc; j--; ) {
      (table->table)[j].key = 0;
    }
    
    /**copy the old table to the one*/
    for( j=(table->alloc)/2; j--; ) {
      if(temp[j].key == 0) /**Empty slot*/
	continue;
      int index=find(table, temp[j].key);
      nusmv_assert( (table->table)[index].key == 0 );
      (table->table)[index].data=temp[j].data;
      (table->table)[index].key=temp[j].key;
    }
    free(temp);
    Bmc_Hash_insert(table, key, data);
    return;
  }
  nusmv_assert((table->table)[i].key == 0);
  (table->table)[i].key=key;
  (table->table)[i].data=data;
  (table->occupied)++;
  return;
}

/**delete the table*/

void Bmc_Hash_delete_table(hashPtr hash)
{
  free(hash->table);
  free(hash);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

static int table_hash_fun(node_ptr key, int size)
{ return((int)(key) % size); }

static int node_eq_fun(node_ptr a1, node_ptr a2)
{ return((a1) == (a2)); }

static int node_neq_fun(node_ptr a1, node_ptr a2)
{ return((a1) != (a2)); }


/**@return index of node, a free index if the node is not in the table*/
static int find(hashPtr table, node_ptr node)
{
  int hash=table_hash_fun(node, table->alloc);
  int i;
  for(i=hash; ; ) {
    if((table->table)[i].key) {
      if(node_eq_fun(node, (table->table)[i].key))
	return i;
    }
    else {
      break;
    }
    i=(i+1)%(table->alloc);
  }
  return i;
}
