#include "bmcTJ.h"

#include "bmcBmc.h"
#include "bmcInt.h"
#include "bmcGen.h"
#include "bmcTableau.h"
#include "bmcConv.h"
#include "bmcDump.h"
#include "bmcModel.h"
#include "bmcVarsMgr.h"
#include "bmcWff.h"
#include "bmcUtils.h"
#include "bmcIncrSat.h"

#include "dag/dag.h"

#include "node/node.h"
#include "be/be.h"
#include "mc/mc.h" /* for print_spec */

#include "sat/sat.h" /* for solver and result */
#include "sat/SatSolver.h"
#include "sat/SatIncSolver.h"

#include "bmcSatTrace.h"

#include "enc/enc.h"
#include "parser/symbols.h" /* for ATOM */
#include "utils/ustring.h"
EXTERN node_ptr boolean_range;

static const char* _SNH_text = "%s:%d: Should not happen";
static const char* _SNYI_text = "%s:%d: Something not yet implemented\n";

#ifdef BENCHMARKING
#include <unistd.h>
#include <sys/times.h>
struct tms time_bmc_start;
struct tms time_gen_start;
struct tms time_solver_start;
struct tms time_temp;
static double time_diff(struct tms *start, struct tms *stop)
{
  return ((((double)stop->tms_utime + (double)stop->tms_stime) -
	   ((double)start->tms_utime + (double)start->tms_stime)) /
	  (double)CLK_TCK);
}
#endif


/*
 * Predefinition for a modified cex printer
 */
static Trace_ptr _generate_and_print_cntexample(BmcVarsMgr_ptr vars_mgr,
						Bmc_IncrSat_ptr solver, 
						BddEnc_ptr bdd_enc, 
						const int k, 
						const char* trace_name,
						node_ptr l_var_node);




typedef struct HJL_state_vars_struct_TAG
{
  /* lists of node_ptr */

  /* State variables of the system */
  lsList   original_state_vars;

  /* System state variables occurring in transition relation */
  lsList   transition_state_vars;

  /* System input variables occurring in transition relation */
  lsList   transition_input_vars;

  lsList   translation_vars_all;  /* State variable available for transl. */
  lsList   translation_vars_pd0;  /* [[f]]_i^0 vars */
  lsList   translation_vars_pdx;  /* [[f]]_i^d, d > 0 vars */
  lsList   translation_vars_aux;  /* <<f>>_i vars */
  lsList   translation_vars_unused; /* State variables not used by transl. */
  node_ptr l_var;
  node_ptr LoopExists_var;
  
  /* System state variables occurring in formula */
  lsList   formula_state_vars;

  /* System input variables occurring in formula */
  lsList   formula_input_vars;

  /* System variables for the simple path constraint.
   * Union of transition_state_vars, formula_state_vars, formula_input_vars */
  lsList   simple_path_system_vars;
} HJL_state_vars_struct;

static HJL_state_vars_struct HJL_state_vars;

/*
 *
 * Auxiliary data structure code for a set of node_ptr
 *
 */
static hash_ptr _set_create();
static void     _set_destroy(hash_ptr hash);
static void     _set_insert(hash_ptr hash, node_ptr bexp);
static int      _set_is_in(hash_ptr hash, node_ptr bexp);

static hash_ptr _set_create()
{
  hash_ptr hash = new_assoc();
  nusmv_assert(hash != (hash_ptr) NULL);
}

static void _set_destroy(hash_ptr hash)
{
  if(hash != (hash_ptr) NULL)
    free_assoc(hash);
}

static void _set_insert(hash_ptr hash, node_ptr bexp)
{
  nusmv_assert(hash != (hash_ptr) NULL);
  nusmv_assert((node_ptr)1 != (node_ptr)NULL);
  insert_assoc(hash, bexp, (node_ptr)1);
}

static int _set_is_in(hash_ptr hash, node_ptr bexp)
{
  node_ptr result;
  nusmv_assert(hash != (hash_ptr) NULL);
  result = find_assoc(hash, bexp);
  if(result == (node_ptr)NULL)
    return 0;
  return 1;
}



/*
 *
 * The structure maintaining state variable information
 *
 */

static void _init_HJL_state_vars(BmcVarsMgr_ptr vars_mgr)
{
  int i;

  HJL_state_vars.original_state_vars = lsCreate();
  HJL_state_vars.transition_state_vars = lsCreate();
  HJL_state_vars.transition_input_vars = lsCreate();
  HJL_state_vars.l_var = NULL;
  HJL_state_vars.LoopExists_var = NULL;
  HJL_state_vars.translation_vars_all = lsCreate();
  HJL_state_vars.translation_vars_pd0 = lsCreate();
  HJL_state_vars.translation_vars_pdx = lsCreate();
  HJL_state_vars.translation_vars_aux = lsCreate();
  HJL_state_vars.translation_vars_unused = lsCreate();
  HJL_state_vars.formula_state_vars = lsCreate();
  HJL_state_vars.formula_input_vars = lsCreate();
  HJL_state_vars.simple_path_system_vars = lsCreate();

  for(i = 0; i < BmcVarsMgr_GetStateVarsNum(vars_mgr); i++)
    {
      /* Get i:th state variable */
      node_ptr node_sv = BmcVarsMgr_VarIndex2Name(vars_mgr, i);

      /* Is sv a real, original state variable? */
      if(node_get_type(node_sv) == DOT && car(node_sv) == Nil && 
	 cdr(node_sv) && node_get_type(cdr(node_sv)) == ATOM &&
	 car(cdr(node_sv)) &&
	 strncmp(((string_ptr)car(cdr(node_sv)))->text, "#TJ", 3) == 0)
	{
	  /* An auxiliary state variable */
	  if(strcmp(((string_ptr)car(cdr(node_sv)))->text, "#TJ_l") == 0)
	    {
	      HJL_state_vars.l_var = node_sv;
	    }
	  else if(strcmp(((string_ptr)car(cdr(node_sv)))->text,
			 "#TJ_LoopExists") == 0)
	    {
	      HJL_state_vars.LoopExists_var = node_sv;
	    }
	  else if(strncmp(((string_ptr)car(cdr(node_sv)))->text, "#TJ_LTL", 6) == 0)
	    {
	      lsNewEnd(HJL_state_vars.translation_vars_all, (lsGeneric)node_sv,
		       LS_NH);
	    }
	  else
	    internal_error("%s:%d: TJ: Unknown auxialiary state variable",
			   __FILE__, __LINE__);
	}
      else
	{
	  /* An original state variable */
	  lsNewEnd(HJL_state_vars.original_state_vars,
		   (lsGeneric)node_sv,
		   LS_NH);
	}
    }
}


static void _print_node_list(FILE *out, lsList l)
{
  char *sep = "";
  lsGen  iterator; 
  node_ptr node;

  lsForEachItem(l, iterator, node)
    {
      fprintf(out, sep); sep = ",";
      print_node(out, node);
    }
}


static void _print_HJL_state_vars(FILE* out)
{
  lsGen  iterator; 
  node_ptr node;
  char *sep;

  fprintf(out, "The %d original state variables are: ",
	  lsLength(HJL_state_vars.original_state_vars));
  _print_node_list(out, HJL_state_vars.original_state_vars);
  fprintf(out, "\n");

  fprintf(out, "The %d state variables in transition relation are: ",
	  lsLength(HJL_state_vars.transition_state_vars));
  _print_node_list(out, HJL_state_vars.transition_state_vars);
  fprintf(out, "\n");

  fprintf(out, "The %d input variables in transition relation are: ",
	  lsLength(HJL_state_vars.transition_input_vars));
  _print_node_list(out, HJL_state_vars.transition_input_vars);
  fprintf(out, "\n");

  fprintf(out, "The l variable is: ");
  print_node(out, HJL_state_vars.l_var);
  fprintf(out, "\n");

  fprintf(out, "The LoopExists variable is: ");
  print_node(out, HJL_state_vars.LoopExists_var);
  fprintf(out, "\n");

  fprintf(out, "Formula state variables are: ");
  _print_node_list(out, HJL_state_vars.translation_vars_all);
  fprintf(out, "\n");

  fprintf(out, "Formula state variables at depth 0 are: ");
  _print_node_list(out, HJL_state_vars.translation_vars_pd0);
  fprintf(out, "\n");

  fprintf(out, "Formula state variables at depth d>0 are: ");
  _print_node_list(out, HJL_state_vars.translation_vars_pdx);
  fprintf(out, "\n");

  fprintf(out, "Formula state variables fox aux translation are: ");
  _print_node_list(out, HJL_state_vars.translation_vars_aux);
  fprintf(out, "\n");

  fprintf(out, "Unused formula state variables are: "); sep = "";
  _print_node_list(out, HJL_state_vars.translation_vars_unused);
  fprintf(out, "\n");

  fprintf(out, "State variable in formula are: ");
  _print_node_list(out, HJL_state_vars.formula_state_vars);
  fprintf(out, "\n");

  fprintf(out, "Input variables in formula are: ");
  _print_node_list(out, HJL_state_vars.formula_input_vars);
  fprintf(out, "\n");

  fprintf(out, "The %d system variables in the simple path constraint are: ",
	  lsLength(HJL_state_vars.simple_path_system_vars));
  _print_node_list(out, HJL_state_vars.simple_path_system_vars);
  fprintf(out, "\n");
}





node_ptr bmc_TJ_add_state_var(const char *name)
{
  node_ptr result;
  
  result = find_node(DOT, Nil,
		     find_node(ATOM,(node_ptr)find_string(name),Nil));
  Encoding_declare_state_var(Enc_get_symb_encoding(), result, boolean_range);
  return result;
}


node_ptr bmc_TJ_find_state_var(const char *name)
{
  node_ptr result;
  
  result = find_node(DOT, Nil,
		     find_node(ATOM,(node_ptr)find_string(name),Nil));
  nusmv_assert(result != (node_ptr)NULL);
  return result;
}
  






/*
 * Make the formula "\land_{v \in V} s_i = s_j"
 * using the state variables V in the list state_vars
 */
static be_ptr _equal_vectors_formula(BmcVarsMgr_ptr vars_mgr,
				     lsList state_vars,
				     const unsigned int i,
				     const unsigned int j)
{
  lsGen    iterator;
  node_ptr node_sv;

  be_ptr be_constraint;

  Be_Manager_ptr be_mgr;

  /* Get be manager */
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  assert(be_mgr);

  /* Initialize constraint */
  be_constraint = Be_Truth(be_mgr);

  lsForEachItem(state_vars, iterator, node_sv)
    {
      /* Get state variable sv at time i */
      be_ptr be_sv_i = BmcVarsMgr_Name2Timed(vars_mgr, node_sv, i, i+1);
      /* Get state variable sv at time j */
      be_ptr be_sv_j = BmcVarsMgr_Name2Timed(vars_mgr, node_sv, j, j+1);

      be_constraint = Be_And(be_mgr,
			     be_constraint,
			     Be_Iff(be_mgr, be_sv_i, be_sv_j));
    }

  return be_constraint;
}






/*
 *
 * Heljanko, Junttila & Latvala PLTL translations
 *
 */

/* State 0 is the L state and 1 is the E state */
static int _L_state() {return 0; }
static int _E_state() {return 1; }
static int _real_k(int k) {return k+2; }
static unsigned int _model_k(int k) {assert(k >= 2); return k-2; }
static void _real_k_print(FILE *out, int k)
{
  if(k == _L_state()) fprintf(out, "L");
  else if(k == _E_state()) fprintf(out, "E");
  else fprintf(out, "%u", _model_k(k));
}    
static char* _real_k_string(const unsigned int k_real)
{
  char *str = ALLOC(char, 32);
  if(k_real == _L_state()) sprintf(str, "L");
  else if(k_real == _E_state()) sprintf(str, "E");
  else sprintf(str, "%u", _model_k(k_real));
  return str;
}


/*
 * Get the initial state and invariant constraint at time i.
 * Similar to Bmc_Model_GetInit0(be_fsm) except for the i argument.
 */
static be_ptr Bmc_TJ_GetInitI(const Bmc_Fsm_ptr be_fsm, const int i)
{
  BmcVarsMgr_ptr vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  Be_Manager_ptr be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  be_ptr init = BmcVarsMgr_ShiftCurrNext2Time(vars_mgr,
					      Be_And(be_mgr,
						     Bmc_Fsm_GetInit(be_fsm),
						     Bmc_Fsm_GetInvar(be_fsm)),
					      i);
  return init;
}


typedef struct _HJL_node_info_struct
{
  unsigned int past_depth;

  array_t*     main_nodes;   /* nodeptr[past_depth] of node_ptr */
  array_t*     main_trans;   /* array_t [time] array_t [past_depth] be_ptr */

  node_ptr     aux_F_node;   /* non-NULL if << F \phi >> is needed */
  array_t*     aux_F_trans;  /* be_ptr aux_F_trans[time] */

  node_ptr     aux_G_node;   /* non-NULL if << G \phi >> is needed */
  array_t*     aux_G_trans;  /* be_ptr aux_G_trans[time] */
} _HJL_node_info;


static enum st_retval _HJL_node_info_free_func(char *k, char *r, char *a)
{
  int i;
  if(r) 
    {
      _HJL_node_info* info = (_HJL_node_info*)r;
      if(info->main_nodes)
	array_free(info->main_nodes), info->main_nodes = 0;
      if(info->main_trans) {
	for(i = 0; i < array_n(info->main_trans); i++) {
	  array_t* a = array_fetch(array_t*, info->main_trans, i);
	  if(a)
	    array_free(a);
	}
	array_free(info->main_trans); info->main_trans = 0;
      }
      if(info->aux_F_trans)
	array_free(info->aux_F_trans), info->aux_F_trans = 0;
      if(info->aux_G_trans)
	array_free(info->aux_G_trans), info->aux_G_trans = 0;
      FREE(info);
    }
  return ST_DELETE;
}


/*
 * ltlspec must be in NNF
 */
static void _HJL_free_main_trans(node_ptr ltlspec, hash_ptr info_map)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  int i;

  /* Debug output */
  if (opt_verbose_level_gt(options, 4))
    {
      fprintf(nusmv_stderr,
	      "Freeing main trans of \n");
      print_node(nusmv_stderr, ltlspec);
      fprintf(nusmv_stderr, "\n");
      fflush(nusmv_stderr);
    }

  visit_cache = _set_create();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      _HJL_node_info* info = 0;
      int has_unprocessed_children;
      
      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get info */
      info = (_HJL_node_info*)find_assoc(info_map, node);
      if(!info)
	internal_error(_SNH_text,__FILE__,__LINE__);

      if(_set_is_in(visit_cache, node))
	{
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}
      
      /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	case TRUEEXP:
	case FALSEEXP:
	  if(info->main_trans) {
	    for(i = 0; i < array_n(info->main_trans); i++) {
	      array_t* a = array_fetch(array_t*, info->main_trans, i);
	      if(a) {
		array_free(a);
		array_insert(array_t*, info->main_trans, i, 0);
	      }
	    }
	  }
	  break;

	case AND:
	case OR:
	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	case UNTIL:
	case RELEASES:
	case TRIGGERED:
	case SINCE:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!_set_is_in(visit_cache, rsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  if(info->main_trans) {
	    for(i = 0; i < array_n(info->main_trans); i++) {
	      array_t* a = array_fetch(array_t*, info->main_trans, i);
	      if(a) {
		array_free(a);
		array_insert(array_t*, info->main_trans, i, 0);
	      }
	    }
	  }
	  break;
	  
	case NOT:
	case OP_GLOBAL:
	case OP_FUTURE:
	case OP_NEXT:
	case OP_HISTORICAL:
	case OP_ONCE:
	case OP_PREC:
	case OP_NOTPRECNOT:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  if(info->main_trans) {
	    for(i = 0; i < array_n(info->main_trans); i++) {
	      array_t* a = array_fetch(array_t*, info->main_trans, i);
	      if(a) {
		array_free(a);
		array_insert(array_t*, info->main_trans, i, 0);
	      }
	    }
	  }
	  break;
	  
	default:
	  print_node(stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      _set_insert(visit_cache, node);
    }
  
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache);
}



/*
 * Compute the variables that occur in the formula ltlspec.
 * ltlspec must be in NNF.
 */
static lsList _HJL_find_formula_vars(BmcVarsMgr_ptr vars_mgr,
				     node_ptr ltlspec)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  lsList   formula_vars = 0;
  int i;

  /* Debug output */
  if (opt_verbose_level_gt(options, 4))
    {
      fprintf(nusmv_stderr, "Finding vars occurring in ");
      print_node(nusmv_stderr, ltlspec);
      fprintf(nusmv_stderr, "\n");
      fflush(nusmv_stderr);
    }
  
  visit_cache = _set_create();
  formula_vars = lsCreate();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      int has_unprocessed_children;

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);

      /* Already visited? */
      if(_set_is_in(visit_cache, node)) {
	if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	  internal_error(_SNH_text,__FILE__,__LINE__);
	continue;
      }

      /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	  lsNewBegin(formula_vars, (lsGeneric)node, LS_NH);
	  break;
	  
	case TRUEEXP:
	case FALSEEXP:
	  break;

	case AND:
	case OR:
	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	case UNTIL:
	case RELEASES:
	case TRIGGERED:
	case SINCE:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!_set_is_in(visit_cache, rsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  break;
	  
	case NOT:
	case OP_GLOBAL:
	case OP_FUTURE:
	case OP_NEXT:
	case OP_HISTORICAL:
	case OP_ONCE:
	case OP_PREC:
	case OP_NOTPRECNOT:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  break;
	  
	default:
	  print_node(nusmv_stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);

      _set_insert(visit_cache, node);
    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache);

  return formula_vars;
}



/*
 * Compute the input variables that occur in the formula ltlspec.
 * ltlspec must be in NNF.
 */
static lsList _HJL_find_formula_input_vars(BmcVarsMgr_ptr vars_mgr,
					   node_ptr ltlspec)
{
  hash_ptr input_var_set = 0;
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  lsList   formula_input_vars = 0;
  int i;

  /* Debug output */
  if (opt_verbose_level_gt(options, 4))
    {
      fprintf(nusmv_stderr, "Finding input vars occurring in ");
      print_node(nusmv_stderr, ltlspec);
      fprintf(nusmv_stderr, "\n");
      fflush(nusmv_stderr);
    }
  
  /* Compute the input variable set */
  input_var_set = _set_create();
  for(i = BmcVarsMgr_GetStateVarsNum(vars_mgr);
      i < (BmcVarsMgr_GetStateVarsNum(vars_mgr) +
	   BmcVarsMgr_GetInputVarsNum(vars_mgr));
      i++)
    {
      /* Get i:th input variable */
      _set_insert(input_var_set, BmcVarsMgr_VarIndex2Name(vars_mgr, i));
    }

  visit_cache = _set_create();
  formula_input_vars = lsCreate();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      int has_unprocessed_children;

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);

      /* Already visited? */
      if(_set_is_in(visit_cache, node)) {
	if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	  internal_error(_SNH_text,__FILE__,__LINE__);
	continue;
      }

      /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	  if(_set_is_in(input_var_set, node))
	    lsNewBegin(formula_input_vars, (lsGeneric)node, LS_NH);
	  break;
	  
	case TRUEEXP:
	case FALSEEXP:
	  break;

	case AND:
	case OR:
	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	case UNTIL:
	case RELEASES:
	case TRIGGERED:
	case SINCE:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!_set_is_in(visit_cache, rsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  break;
	  
	case NOT:
	case OP_GLOBAL:
	case OP_FUTURE:
	case OP_NEXT:
	case OP_HISTORICAL:
	case OP_ONCE:
	case OP_PREC:
	case OP_NOTPRECNOT:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  break;
	  
	default:
	  print_node(nusmv_stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);

      _set_insert(visit_cache, node);
    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(input_var_set);
  _set_destroy(visit_cache);

  return formula_input_vars;
}



/*
 * ltlspec must be in NNF
 */
unsigned int bmc_HJL_count_aux_state_vars(node_ptr ltlspec)
{
  hash_ptr visit_cache = 0;
  hash_ptr past_depth_map = 0;
  lsList   unprocessed_nodes = 0;
  unsigned int nof_aux_state_vars = 0;

  /* Debug output */
  if (opt_verbose_level_gt(options, 4))
    {
      fprintf(nusmv_stderr,
	      "Counting needed aux state vars for HJL translation of ");
      print_node(nusmv_stderr, ltlspec);
      fprintf(nusmv_stderr, "\n");
      fflush(nusmv_stderr);
    }

  visit_cache = _set_create();
  past_depth_map = new_assoc();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      unsigned int past_depth, lsf_past_depth, rsf_past_depth;
      int has_unprocessed_children;

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);

      if(_set_is_in(visit_cache, node))
	{
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}

     /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	case TRUEEXP:
	case FALSEEXP:
	  past_depth = 0;
	  insert_assoc(past_depth_map, node, (node_ptr)0);
	  break;

	case AND:
	case OR:
	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	case UNTIL:
	case RELEASES:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!_set_is_in(visit_cache, rsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  lsf_past_depth = (unsigned int)find_assoc(past_depth_map, lsf);
	  rsf_past_depth = (unsigned int)find_assoc(past_depth_map, rsf);
	  past_depth = max(lsf_past_depth, rsf_past_depth);
	  insert_assoc(past_depth_map, node, (node_ptr)past_depth);
	  break;
	  
	case NOT:
	case OP_GLOBAL:
	case OP_FUTURE:
	case OP_NEXT:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  lsf_past_depth = (unsigned int)find_assoc(past_depth_map, lsf);
	  past_depth = lsf_past_depth;
	  insert_assoc(past_depth_map, node, (node_ptr)past_depth);
	  break;

	case OP_HISTORICAL:
	case OP_ONCE:
	case OP_PREC:
	case OP_NOTPRECNOT:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  lsf_past_depth = (unsigned int)find_assoc(past_depth_map, lsf);
	  past_depth = lsf_past_depth + 1;
	  insert_assoc(past_depth_map, node, (node_ptr)past_depth);
	  break;

	case TRIGGERED:
	case SINCE:
	  if(!_set_is_in(visit_cache, lsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!_set_is_in(visit_cache, rsf)) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;
	  lsf_past_depth = (unsigned int)find_assoc(past_depth_map, lsf);
	  rsf_past_depth = (unsigned int)find_assoc(past_depth_map, rsf);
	  past_depth = max(lsf_past_depth, rsf_past_depth) + 1;
	  insert_assoc(past_depth_map, node, (node_ptr)past_depth);
	  break;
	  
	default:
	  past_depth = 0;
	  insert_assoc(past_depth_map, node, (node_ptr)past_depth);
	  break;
#if 0
	default:
	  print_node(stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
#endif
	}
      if(has_unprocessed_children)
	continue;
      
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);

      _set_insert(visit_cache, node);

      /* Debug output */
      if (opt_verbose_level_gt(options, 5)) {
	fprintf(nusmv_stderr, " The past depth of ");
	print_node(nusmv_stderr, node);
	fprintf(nusmv_stderr, " is %u\n", past_depth);
	fflush(nusmv_stderr);
      }

      nof_aux_state_vars += (past_depth + 2);
    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache);
  free_assoc(past_depth_map);

  return nof_aux_state_vars * 2;
}



/*
 * Returns a fresh state variable taken from the list fresh_state_vars.
 */
static node_ptr _1_fresh_state_var(lsList fresh_state_vars)
{
  node_ptr node;
  if(lsDelBegin(fresh_state_vars, (lsGeneric*)&node) != LS_OK)
    internal_error(_SNH_text,__FILE__,__LINE__);
  return node;
}


/*
 * Returns an array of n fresh state variables
 * taken from the list fresh_state_vars.
 */
static array_t* _n_fresh_state_vars(const unsigned int n,
				    lsList fresh_state_vars)
{
  array_t* array;
  unsigned int i;

  assert(n > 0);
  array = array_alloc(node_ptr, n);
  assert(array);

  for(i = 0; i < n; i++)
    {
      array_insert(node_ptr, array, i, _1_fresh_state_var(fresh_state_vars));
    }
  return array;
}



/*
 * Prints the formula state variable association
 */
static void _HJL_print_varmap(FILE *out, node_ptr node, _HJL_node_info *info)
{
  assert(node);
  assert(info);
  if(opt_verbose_level_gt(options, 1))
    {
      unsigned int d;
      for(d = 0; d <= info->past_depth; d++)
	{
	  fprintf(out, "[[");
	  print_node(out, node);
	  fprintf(out, "]]^%u = ", d);
	  print_node(out, array_fetch(node_ptr, info->main_nodes, d));
	  fprintf(out, ";\n");
	}
    }
}
static void _HJL_print_Gvarmap(FILE *out, node_ptr var, node_ptr formula)
{
  if(opt_verbose_level_gt(options, 1))
    {
      fprintf(nusmv_stderr, "<<G ");
      print_node(nusmv_stderr, formula);
      fprintf(nusmv_stderr, ">> = ");
      print_node(nusmv_stderr, var);
      fprintf(nusmv_stderr, ";\n");
    }
}
static void _HJL_print_Fvarmap(FILE *out, node_ptr var, node_ptr formula)
{
  if(opt_verbose_level_gt(options, 1))
    {
      fprintf(nusmv_stderr, "<<F ");
      print_node(nusmv_stderr, formula);
      fprintf(nusmv_stderr, ">> = ");
      print_node(nusmv_stderr, var);
      fprintf(nusmv_stderr, ";\n");
    }
}


/*
 * Returns a hash from node_ptr to _HJL_LTL_info*.
 * The used state vars are removed from free_state_vars and
 *  inserted in state_vars_formula_??? appropriately.
 */
static hash_ptr _HJL_init_LTL_info(node_ptr ltlspec,
				   lsList free_state_vars,
				   lsList state_vars_formula_pd0,
				   lsList state_vars_formula_pdx,
				   lsList state_vars_formula_aux,
				   const int opt_force_state_vars,
				   const int opt_do_virtual_unrolling)
{
  hash_ptr info_map = 0;
  lsList   unprocessed_nodes = 0;
  unsigned int d;

  assert(free_state_vars);
  assert(state_vars_formula_pd0);
  assert(state_vars_formula_pdx);
  assert(state_vars_formula_aux);

  /* Debug output */
  if (opt_verbose_level_gt(options, 4))
    {
      fprintf(nusmv_stderr, "Computing the past depth information and"
	      " allocating variables for ");
      print_node(nusmv_stderr, ltlspec);
      fprintf(nusmv_stderr, "\n");
      fflush(nusmv_stderr);
    }

  info_map = new_assoc();
  nusmv_assert(info_map != (hash_ptr)NULL);

  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node = 0, lsf = 0, rsf = 0;
      _HJL_node_info *info = 0, *lsf_info = 0, *rsf_info = 0;
      int has_unprocessed_children;

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);

      /* Is info already computed? */
      if(find_assoc(info_map, node) != 0)
	{
	  /* Remove from unprocessed stack */
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}
      
      /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	case TRUEEXP:
	case FALSEEXP:
	  info = ALLOC(_HJL_node_info, 1);
	  info->past_depth = 0;
	  if(opt_force_state_vars)
	    {
	      /* Formula variable translation */
	      info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						     free_state_vars);
	      lsNewBegin(state_vars_formula_pd0,
			 (lsGeneric)array_fetch(node_ptr, info->main_nodes, 0),
			 LS_NH);
	      _HJL_print_varmap(nusmv_stderr, node, info);
	    }
	  else
	    {
	      /* Definitional translation */
	      info->main_nodes = 0;
	    }
	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);
	  break;


	case AND:
	case OR:
	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!rsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);
	  if(opt_do_virtual_unrolling)
	    info->past_depth = max(lsf_info->past_depth, rsf_info->past_depth);
	  else
	    info->past_depth = 0;
	  if(opt_force_state_vars)
	    {
	      /* Formula variable translation */
	      info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						     free_state_vars);
	      lsNewBegin(state_vars_formula_pd0,
			 (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
			 LS_NH);
	      for(d = 1; d <= info->past_depth; d++)
		lsNewBegin(state_vars_formula_pdx,
			   (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
			   LS_NH);
	      _HJL_print_varmap(nusmv_stderr, node, info);
	    }
	  else
	    {
	      /* Definitional translation */
	      info->main_nodes = 0;
	    }
	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);
	  break;

	  
	case NOT:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = lsf_info->past_depth;
	  else
	    info->past_depth = 0;
	  assert(info->past_depth == 0);

	  if(opt_force_state_vars)
	    {
	      /* Formula variable translation */
	      info->main_nodes =  _n_fresh_state_vars(info->past_depth + 1,
						      free_state_vars);
	      lsNewBegin(state_vars_formula_pd0,
			 (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
			 LS_NH);
	      for(d = 1; d <= info->past_depth; d++)
		lsNewBegin(state_vars_formula_pdx,
			   (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
			   LS_NH);
	      _HJL_print_varmap(nusmv_stderr, node, info);
	    }
	  else
	    {
	      /* Definitional translation */
	      info->main_nodes = 0;
	    }
	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);
	  break;

	  
	case OP_FUTURE:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = lsf_info->past_depth;
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);

	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);

	  /* Aux_F_node for the subformula */
	  if(lsf_info->aux_F_node == 0) {
	    lsf_info->aux_F_node = _1_fresh_state_var(free_state_vars);
	    lsNewBegin(state_vars_formula_aux,
		       (lsGeneric)lsf_info->aux_F_node, LS_NH);
	    _HJL_print_Fvarmap(nusmv_stderr, lsf_info->aux_F_node, lsf);
	  }
	  break;


	case OP_GLOBAL:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = lsf_info->past_depth;
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);

	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);

	  /* Aux_G_node for the subformula */
	  if(lsf_info->aux_G_node == 0) {
	    lsf_info->aux_G_node = _1_fresh_state_var(free_state_vars);
	    lsNewBegin(state_vars_formula_aux,
		       (lsGeneric)lsf_info->aux_G_node, LS_NH);
	    _HJL_print_Gvarmap(nusmv_stderr, lsf_info->aux_G_node, lsf);
	  }
	  break;


	case OP_NEXT:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = lsf_info->past_depth;
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);
	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);

	  /* Subformula referred in the L and k+1 state,
	     make sure it can be free */
	  if(lsf_info->main_nodes == 0)
	    {
	      lsf_info->main_nodes =
		_n_fresh_state_vars(lsf_info->past_depth + 1,
				    free_state_vars);
	      lsNewBegin(state_vars_formula_pd0,
			 (lsGeneric)array_fetch(node_ptr,
						lsf_info->main_nodes,0),
			 LS_NH);
	      for(d = 1; d <= lsf_info->past_depth; d++)
		lsNewBegin(state_vars_formula_pdx,
			   (lsGeneric)array_fetch(node_ptr,
						  lsf_info->main_nodes,d),
			   LS_NH);
	      _HJL_print_varmap(nusmv_stderr, lsf, lsf_info);
	    }
	  break;


	case OP_HISTORICAL:
	case OP_ONCE:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = lsf_info->past_depth + 1;
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);

	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);
	  break;



	case OP_PREC:
	case OP_NOTPRECNOT:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = lsf_info->past_depth + 1;
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);
	  
	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);

	  /* Subformula referred in the E state, make sure it can be free */
	  if(lsf_info->main_nodes == 0)
	    {
	      lsf_info->main_nodes =
		_n_fresh_state_vars(lsf_info->past_depth + 1,
				    free_state_vars);
	      lsNewBegin(state_vars_formula_pd0,
			 (lsGeneric)array_fetch(node_ptr,
						lsf_info->main_nodes,0),
			 LS_NH);
	      for(d = 1; d <= lsf_info->past_depth; d++)
		lsNewBegin(state_vars_formula_pdx,
			   (lsGeneric)array_fetch(node_ptr,
						  lsf_info->main_nodes,d),
			   LS_NH);
	      _HJL_print_varmap(nusmv_stderr, lsf, lsf_info);
	    }
	  break;


	case UNTIL:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!rsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = max(lsf_info->past_depth, rsf_info->past_depth);
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);

	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);

	  /* Aux_F_node for the right subformula */
	  if(rsf_info->aux_F_node == 0) {
	    rsf_info->aux_F_node = _1_fresh_state_var(free_state_vars);
	    lsNewBegin(state_vars_formula_aux,
		       (lsGeneric)rsf_info->aux_F_node, LS_NH);
	    _HJL_print_Fvarmap(nusmv_stderr, rsf_info->aux_F_node, rsf);
	  }
	  break;


	case RELEASES:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!rsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = max(lsf_info->past_depth, rsf_info->past_depth);
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);

	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);

	  /* Aux_G_node for the right subformula */
	  if(rsf_info->aux_G_node == 0) {
	    rsf_info->aux_G_node = _1_fresh_state_var(free_state_vars);
	    lsNewBegin(state_vars_formula_aux,
		       (lsGeneric)rsf_info->aux_G_node, LS_NH);
	    _HJL_print_Gvarmap(nusmv_stderr, rsf_info->aux_G_node, rsf);
	  }
	  break;


	case TRIGGERED:
	case SINCE:
	  lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	  rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	  if(!lsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(!rsf_info) {
	    lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	    has_unprocessed_children = 1;
	  }
	  if(has_unprocessed_children)
	    break;

	  info = ALLOC(_HJL_node_info, 1);

	  if(opt_do_virtual_unrolling)
	    info->past_depth = max(lsf_info->past_depth, rsf_info->past_depth)+1;
	  else
	    info->past_depth = 0;

	  info->main_nodes = _n_fresh_state_vars(info->past_depth + 1,
						 free_state_vars);
	  lsNewBegin(state_vars_formula_pd0,
		     (lsGeneric)array_fetch(node_ptr,info->main_nodes,0),
		     LS_NH);
	  for(d = 1; d <= info->past_depth; d++)
	    lsNewBegin(state_vars_formula_pdx,
		       (lsGeneric)array_fetch(node_ptr,info->main_nodes,d),
		       LS_NH);
	  _HJL_print_varmap(nusmv_stderr, node, info);

	  info->main_trans = array_alloc(array_t*, 1);
	  info->aux_F_node = 0;
	  info->aux_F_trans = 0;
	  info->aux_G_node = 0;
	  info->aux_G_trans = 0;
	  insert_assoc(info_map, node, (node_ptr)info);
	  break;

	  
	default:
	  print_node(stderr, node);
	  fprintf(stderr, "\n%d\n", node_get_type(node));
	  internal_error(_SNYI_text, __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      assert(find_assoc(info_map, node) != 0);

      /* Remove from unprocessed stack */
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);

      /* Debug output */
      if (opt_verbose_level_gt(options, 5))
	{
	  fprintf(nusmv_stderr, " The past depth of ");
	  print_node(nusmv_stderr, node);
	  fprintf(nusmv_stderr, " is %u\n", info->past_depth);
	  fflush(nusmv_stderr);
	}
    }
      
  lsDestroy(unprocessed_nodes, 0);

  return info_map;
}


#define DEFDONE 1


/*
 * Initialize main_trans[i][d], 0<=d<=pd, to
 *   (i) 0 for definitionally translated subformulae,
 *   (ii) [[f]]_i^d variable for variable translated subformulae
 */
static void _HJL_init_state_vector(const BmcVarsMgr_ptr vars_mgr,
				   const node_ptr ltlspec,
				   const hash_ptr info_map,
				   const unsigned int i_real)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  Be_Manager_ptr be_mgr;
  unsigned int d;
  char *i_str = _real_k_string(i_real);

  nusmv_assert(info_map != (hash_ptr)NULL);


  /* Get be manager */
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);


  /* Debug output */
  if (opt_verbose_level_gt(options, 1))
    {
      fprintf(nusmv_stderr, "Initializing state vector at time %s\n", i_str);
      fflush(nusmv_stderr);
    }
  
  
  visit_cache = _set_create();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      _HJL_node_info *info, *lsf_info, *rsf_info;
      array_t *past_array, *lsf_past_array, *rsf_past_array;
      int has_unprocessed_children;
      

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);

      
      /* Get info */
      info = (_HJL_node_info*)find_assoc(info_map, node);
      if(!info)
	internal_error(_SNH_text,__FILE__,__LINE__);

      
      /* Get past_array */
      assert(info->main_trans);
      while(array_n(info->main_trans) <= i_real)
	array_insert(array_t*, info->main_trans, array_n(info->main_trans), 0);
      /*
	if(array_n(info->main_trans) <= i)
	array_insert(array_t*, info->main_trans, i, 0);
      */
      past_array = array_fetch(array_t*, info->main_trans, i_real);


      /* Already visited? */
      if(_set_is_in(visit_cache, node))
	{
	  assert(past_array);
	  /* Remove node from unprocessed stack */
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}
      

      /* Initialize past_array[i] if not already done */
      if(!past_array)
	{
	  past_array = array_alloc(be_ptr, info->past_depth+1);
	  for(d = 0; d <= info->past_depth; d++)
	    array_insert(be_ptr, past_array, d, 0);
	  array_insert(array_t*, info->main_trans, i_real, past_array);
	}

      for(d = 0; d <= info->past_depth; d++)
	assert(array_fetch(be_ptr, past_array, d) == 0);      

      /* Traverse children and build past_array */
      lsf = car(node);
      rsf = cdr(node);
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	  
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	  {
	    be_ptr be_result = 0;
	    
	    if(opt_verbose_level_gt(options, 5))
	      {
		fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
		fprintf(nusmv_stderr, "\n");
	      }
	    
	    assert(info->past_depth == 0);

	    if(info->main_nodes != 0)
	      {
		/* main_array[i][0] = [[p]]_i */
		node_ptr node_p = array_fetch(node_ptr, info->main_nodes, 0);
		assert(node_p);
		be_result = BmcVarsMgr_Name2Timed(vars_mgr,node_p,
						  i_real,i_real+1);
		assert(be_result);
		array_insert(be_ptr, past_array, 0, be_result);
		    
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  at depth 0: ");
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }
	    else
	      {
#if DEFDONE
		/* Definitional translation */
		/* main_array[i][0] = p_i */
		be_result =
		  BmcVarsMgr_Name2Timed(vars_mgr, node, i_real, i_real+1);
		assert(be_result);
		array_insert(be_ptr, past_array, 0, be_result);
		
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[p]]_%s^0 := p_%s: ",i_str,i_str);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
#endif
	      }
	    break;
	  }

	case TRUEEXP:
	  {
	    be_ptr be_result = 0;
	    
	    if(opt_verbose_level_gt(options, 5))
	      {
		fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
		fprintf(nusmv_stderr, "\n");
	      }
	    
	    assert(info->past_depth == 0);

	    if(info->main_nodes != 0)
	      {
		/* main_array[i][0] = [[TRUE]]_i */
		node_ptr node_aux = array_fetch(node_ptr, info->main_nodes, 0);
		assert(node_aux);
		be_result = BmcVarsMgr_Name2Timed(vars_mgr,node_aux,
						  i_real,i_real+1);
		assert(be_result);
		array_insert(be_ptr, past_array, 0, be_result);
		
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  at depth %u: ", d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }
	    else
	      {
#if DEFDONE
		/* Definitional translation */
		/* main_array[i][0] = TRUE */
		be_result = Be_Truth(be_mgr);
		array_insert(be_ptr, past_array, 0, be_result);
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[TRUE]]_%s^0 := TRUE: ", i_str);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
#endif
	      }
	    
	    break;
	  }

	case FALSEEXP:
	  {
	    be_ptr be_result = 0;
	    
	    if(opt_verbose_level_gt(options, 5))
	      {
		fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
		fprintf(nusmv_stderr, "\n");
	      }
	    
	    assert(info->past_depth == 0);

	    if(info->main_nodes != 0)
	      {
		/* Formula variable translation */
		/* main_array[i][0] = [[FALSE]]_i */
		node_ptr node_aux = array_fetch(node_ptr, info->main_nodes, 0);
		assert(node_aux);
		be_result = BmcVarsMgr_Name2Timed(vars_mgr,node_aux,
						  i_real,i_real+1);
		assert(be_result);
		array_insert(be_ptr, past_array, 0, be_result);
		    
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  at depth 0: ");
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }
	    else
	      {
#if DEFDONE
		/* Definitional translation */
		/* main_array[i][0] = FALSE */
		be_result = Be_Falsity(be_mgr);
		array_insert(be_ptr, past_array, 0, be_result);
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[FALSE]]_%s^0 := FALSE: ", i_str);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
#endif
	      }
	    break;
	  }


	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	  {
	    internal_error("%s:%d: Formula not in NNF\n", __FILE__,__LINE__);
	    break;
	  }
	  
	case OR:
	  {
	    unsigned int d;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }
	    
	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(array_t*,rsf_info->main_trans,i_real);
	    assert(rsf_past_array);
	    
	    if(info->main_nodes != 0)
	      {
		/* Formula variable translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* main_array[i][d] = [[f | g]]_i^d */
		    node_ptr node_aux;
		    be_ptr   be_result;
		    node_aux = array_fetch(node_ptr, info->main_nodes, d);
		    assert(node_aux);
		    be_result = BmcVarsMgr_Name2Timed(vars_mgr,node_aux,
						      i_real, i_real);
		    assert(be_result);
		    array_insert(be_ptr, past_array, d, be_result);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  at depth %u: ", d);
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
			fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    else
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Definitional translation */
		    /* [[f | g]]_i^d := [[f]]_i^d | [[g]]_i^d */
		    be_ptr be_lsf, be_rsf, be_result;
		    be_lsf = array_fetch(be_ptr, lsf_past_array,
					 min(d, lsf_info->past_depth));
		    assert(be_lsf);
		    be_rsf = array_fetch(be_ptr, rsf_past_array,
					 min(d, rsf_info->past_depth));
		    assert(be_rsf);
		    be_result = Be_Or(be_mgr, be_lsf, be_rsf);
		    assert(array_fetch(be_ptr, past_array, d) == 0);
		    array_insert(be_ptr, past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[f | g]]_%s^%d := [[f]]_%s^%d | [[g]]_%s^%d: ",
			      i_str, d,
			      i_str, min(d, lsf_info->past_depth),
			      i_str, min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");	
		    }
		  }
#endif
	      }

	    break;
	  }


	case AND:
	  {
	    unsigned int d;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }
	    
	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(array_t*,rsf_info->main_trans,i_real);
	    assert(rsf_past_array);
	    
	    if(info->main_nodes != 0)
	      {
		/* Formula variable translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* main_array[i][d] = [[f & g]]_i^d */
		    node_ptr node_aux;
		    be_ptr   be_result;
		    node_aux = array_fetch(node_ptr, info->main_nodes, d);
		    assert(node_aux);
		    be_result = BmcVarsMgr_Name2Timed(vars_mgr,node_aux,
						      i_real, i_real);
		    assert(be_result);
		    array_insert(be_ptr, past_array, d, be_result);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  at depth %u: ", d);
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
			fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    else
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Definitional translation */
		    /* [[f | g]]_i^d := [[f]]_i^d & [[g]]_i^d */
		    be_ptr be_lsf, be_rsf, be_result;
		    be_lsf = array_fetch(be_ptr, lsf_past_array,
					 min(d, lsf_info->past_depth));
		    assert(be_lsf);
		    be_rsf = array_fetch(be_ptr, rsf_past_array,
					 min(d, rsf_info->past_depth));
		    assert(be_rsf);
		    be_result = Be_And(be_mgr, be_lsf, be_rsf);
		    assert(array_fetch(be_ptr, past_array, d) == 0);
		    array_insert(be_ptr, past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[f & g]]_%s^%d := [[f]]_%s^%d & [[g]]_%s^%d: ",
			      i_str, d,
			      i_str, min(d, lsf_info->past_depth),
			      i_str, min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
#endif
	      }
	    break;
	  }



	case NOT:
	  {
	    unsigned int d;
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->past_depth == 0);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    if(info->main_nodes != 0)
	      {
		/* Formula variable translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* main_array[i][d] = [[!f]]_i^d */
		    node_ptr node_aux;
		    be_ptr   be_result;
		    node_aux = array_fetch(node_ptr, info->main_nodes, d);
		    assert(node_aux);
		    be_result = BmcVarsMgr_Name2Timed(vars_mgr,node_aux,
						      i_real, i_real);
		    assert(be_result);
		    array_insert(be_ptr, past_array, d, be_result);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  at depth %u: ", d);
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
			fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    else
	      {
#if DEFDONE
		/* Definitional translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[!f]]_i^d := ![[f]]_i^d */
		    be_ptr be_lsf, be_result;
		    be_lsf = array_fetch(be_ptr, lsf_past_array, d);
		    assert(be_lsf);
		    be_result = Be_Not(be_mgr, be_lsf);
		    assert(array_fetch(be_ptr, past_array, d) == 0);
		    array_insert(be_ptr, past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  [!f]]_%s^%d := ![[f]]_%s^%d: ",
			      i_str, d, i_str, d),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
#endif
	      }
	    break;
	  }


	case UNTIL:
	case RELEASES:
	case TRIGGERED:
	case SINCE:
	  {
	    unsigned int d;
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_nodes);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* main_array[i][d] = [[f OP g]]_i^d */
		node_ptr node_aux;
		be_ptr   be_result;
		node_aux = array_fetch(node_ptr, info->main_nodes, d);
		assert(node_aux);
		be_result = BmcVarsMgr_Name2Timed(vars_mgr, node_aux,
						  i_real, i_real);
		assert(be_result);
		array_insert(be_ptr, past_array, d, be_result);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  at depth %u: ", d);
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		    fprintf(nusmv_stderr, "\n");
		}
	      }
	    break;
	  }

	case OP_FUTURE:
	case OP_GLOBAL:
	case OP_NEXT:
	case OP_HISTORICAL:
	case OP_ONCE:
	case OP_PREC:
	case OP_NOTPRECNOT:
	  {
	    unsigned int d;
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_nodes);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* main_array[i][d] = [[OP f]]_i^d */
		node_ptr node_aux;
		be_ptr   be_result;
		node_aux = array_fetch(node_ptr, info->main_nodes, d);
		assert(node_aux);
		be_result = BmcVarsMgr_Name2Timed(vars_mgr, node_aux,
						  i_real, i_real);
		assert(be_result);
		array_insert(be_ptr, past_array, d, be_result);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  at depth %u: ", d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }
	    break;
	  }
	  
	default:
	  print_node(stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      /* Remove node from unprocessed stack */
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Mark visited */
      _set_insert(visit_cache, node);
    }

      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache); visit_cache = 0;
  FREE(i_str); i_str = 0;
}




/*
 * Create the base constraints
 * Return a list of be_ptrs for the created constraints
 */
static lsList _HJL_unroll_base(BmcVarsMgr_ptr vars_mgr,
			       node_ptr ltlspec,
			       hash_ptr info_map,
			       be_ptr be_LoopExists,
			       const int do_optimization)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  lsList   created_constraints = 0;
  Be_Manager_ptr be_mgr;

  nusmv_assert(info_map != (hash_ptr)NULL);

  /* Get be manager */
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);

  /* Create list for the constraints */
  created_constraints = lsCreate();

  /* Debug output */
  if(opt_verbose_level_gt(options, 1))
    {
      fprintf(nusmv_stderr, "Creating the base constraints\n");
      fflush(nusmv_stderr);
    }
  
  visit_cache = _set_create();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);

  /*internal_error("%s:%d: Not yet implemented\n");*/
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      _HJL_node_info *info, *lsf_info, *rsf_info;
      array_t *E_past_array = 0, *L_past_array = 0;
      int has_unprocessed_children;

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get info */
      info = (_HJL_node_info*)find_assoc(info_map, node);
      if(!info)
	internal_error(_SNH_text,__FILE__,__LINE__);

      /* Get past_array */
      /* _HJL_init_state_vector should have allocated main_trans[i] before */
      if(info->main_trans)
	{
	  assert(array_n(info->main_trans) >= 2);
	  L_past_array = array_fetch(array_t*, info->main_trans, _L_state());
	  assert(L_past_array);
	  E_past_array = array_fetch(array_t*, info->main_trans, _E_state());
	  assert(E_past_array);
	}

      
      /* Already build? */
      if(_set_is_in(visit_cache, node))
	{
	  /* Remove node from unprocessed stack */
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}


     /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{

	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	  {
	    if(do_optimization)
	      {
		/* Optimization:
		   Add [[p]]_E^d <=> [[p]]_E */
		be_ptr be_result = 0;
		assert(info->past_depth == 0);
		if(info->main_nodes == 0)
		  {
#if DEFDONE
		    assert(array_fetch(be_ptr, E_past_array, 0) != 0);
#else
		    /* Definitional translation */
		    /* [[p]]_i^0 := p_i */
		    assert(array_fetch(be_ptr, E_past_array, 0) == 0);
		    be_result = BmcVarsMgr_Name2Timed(vars_mgr, node,
						      _E_state(),_E_state()+1);
		    assert(be_result);
		    array_insert(be_ptr, E_past_array, 0, be_result);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[p]]_E^0 := p_E: ");
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
#endif
		  }
		else
		  {
		    /* State variable translation
		     * Build [[p]]_E^0 <=> p_E
		     */
		    be_ptr be_p_E_0, be_realp_E;
		    be_p_E_0 = array_fetch(be_ptr, E_past_array, 0);
		    assert(be_p_E_0);
		    be_realp_E = BmcVarsMgr_Name2Timed(vars_mgr, node,
						       _E_state(),
						       _E_state()+1);
		    assert(be_realp_E);
		    
		    be_result = Be_Iff(be_mgr, be_p_E_0, be_realp_E);
		    assert(be_result);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[p]]_E^0 <=> p_E: ");
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }

	case TRUEEXP:
	  {
	    if(do_optimization)
	      {
		be_ptr be_result;
		
		assert(info->past_depth == 0);
		
		if(info->main_nodes == 0)
		  {
#if DEFDONE
		    assert(array_fetch(be_ptr, E_past_array, 0) != 0);
#else
		    /* Definitional translation */
		    /* [[TRUE]]_E^0 := TRUE */
		    assert(array_fetch(be_ptr, E_past_array, 0) == 0);
		    be_result = Be_Truth(be_mgr);
		    array_insert(be_ptr, E_past_array, 0, be_result);
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[TRUE]]_E^0 := TRUE: ");
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
#endif
		  }
		else
		  {
		    /* Formula variable translation */
		    /* Build [[TRUE]]_E^0 <=> TRUE */
		    be_ptr be_TRUE_E_0 = array_fetch(be_ptr, E_past_array, 0);
		    assert(be_TRUE_E_0);
		    
		    be_result = Be_Iff(be_mgr, be_TRUE_E_0, Be_Truth(be_mgr));
		    assert(be_result);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[TRUE]]_E^0 <=> TRUE: ");
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }

	case FALSEEXP:
	  {
	    if(do_optimization)
	      {
		be_ptr be_result;
		
		assert(info->past_depth == 0);
		
		if(info->main_nodes == 0)
		  {
#if DEFDONE
		    assert(array_fetch(be_ptr, E_past_array, 0) != 0);
#else
		    /* Definitional translation */
		    /* [[FALSE]]_E^0 := FALSE */
		    assert(array_fetch(be_ptr, E_past_array, 0) == 0);
		    be_result = Be_Falsity(be_mgr);
		    array_insert(be_ptr, E_past_array, 0, be_result);
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[FALSE]]_E^0 := FALSE: ");
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
#endif
		  }
		else
		  {
		    /* Formula variable translation */
		    /* Build [[FALSE]]_E^0 <=> FALSE */
		    be_ptr be_FALSE_E_0 = array_fetch(be_ptr, E_past_array, 0);
		    assert(be_FALSE_E_0);
		    
		    be_result = Be_Iff(be_mgr,be_FALSE_E_0,Be_Falsity(be_mgr));
		    assert(be_result);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[FALSE]]_E^0 <=> FALSE: ");
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }

	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	  {
	    internal_error("%s:%d: Formula not in NNF\n",__FILE__,__LINE__);
	    break;
	  }

	case OR:
	  {
	    unsigned int d;
	    array_t *past_array_f_E, *past_array_g_E;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    past_array_f_E = array_fetch(array_t*, lsf_info->main_trans,
					 _E_state());
	    assert(past_array_f_E);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    past_array_g_E = array_fetch(array_t*, rsf_info->main_trans,
					 _E_state());
	    assert(past_array_g_E);
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  assert(array_fetch(be_ptr, E_past_array, d) != 0);
#else
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Definitional translation */
		    /* [[f | g]]_E^d := [[f]]_E^d | [[g]]_E^d */
		    be_ptr be_lsf, be_rsf, be_result;
		    be_lsf = array_fetch(be_ptr, past_array_f_E,
					 min(d, lsf_info->past_depth));
		    assert(be_lsf);
		    be_rsf = array_fetch(be_ptr, past_array_g_E,
					 min(d, rsf_info->past_depth));
		    assert(be_rsf);
		    be_result = Be_Or(be_mgr, be_lsf, be_rsf);
		    assert(array_fetch(be_ptr, E_past_array, d) == 0);
		    array_insert(be_ptr, E_past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr,
			      "  [[f | g]]_E^%d := [[f]]_E^%d | [[g]]_E^%d: ",
			      d,
			      min(d, lsf_info->past_depth),
			      min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");	
		    }
		  }
#endif
	      }
	    else
	      {
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[f | g]]_E^d <=> [[f]]_E^d | [[g]]_E^d */
		    be_ptr be_fORg_E_d, be_f_E_d, be_g_E_d, be_result;
		    be_fORg_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_fORg_E_d);
		    be_f_E_d = array_fetch(be_ptr, past_array_f_E,
					   min(d, lsf_info->past_depth));
		    assert(be_f_E_d);
		    be_g_E_d = array_fetch(be_ptr, past_array_g_E,
					   min(d, rsf_info->past_depth));
		    assert(be_g_E_d);
		    be_result = Be_Iff(be_mgr, be_fORg_E_d,
				       Be_Or(be_mgr, be_f_E_d, be_g_E_d));
		    assert(be_result);

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr,
			      "  [[f | g]]_E^%d <=> [[f]]_E^%d | [[g]]_E^%d: ",
			      d,
			      min(d, lsf_info->past_depth),
			      min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }


	case AND:
	  {
	    unsigned int d;
	    array_t *past_array_f_E, *past_array_g_E;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    past_array_f_E = array_fetch(array_t*, lsf_info->main_trans,
					 _E_state());
	    assert(past_array_f_E);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    past_array_g_E = array_fetch(array_t*, rsf_info->main_trans,
					 _E_state());
	    assert(past_array_g_E);
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  assert(array_fetch(be_ptr, E_past_array, d) != 0);
#else
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Definitional translation */
		    /* [[f & g]]_E^d := [[f]]_E^d & [[g]]_E^d */
		    be_ptr be_lsf, be_rsf, be_result;
		    be_lsf = array_fetch(be_ptr, past_array_f_E,
					 min(d, lsf_info->past_depth));
		    assert(be_lsf);
		    be_rsf = array_fetch(be_ptr, past_array_g_E,
					 min(d, rsf_info->past_depth));
		    assert(be_rsf);
		    be_result = Be_And(be_mgr, be_lsf, be_rsf);
		    assert(array_fetch(be_ptr, E_past_array, d) == 0);
		    array_insert(be_ptr, E_past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr,
			      "  [[f | g]]_E^%d := [[f]]_E^%d & [[g]]_E^%d: ",
			      d,
			      min(d, lsf_info->past_depth),
			      min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");	
		    }
		  }
#endif
	      }
	    else
	      {
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[f & g]]_E^d <=> [[f]]_E^d & [[g]]_E^d */
		    be_ptr be_fANDg_E_d, be_f_E_d, be_g_E_d, be_result;
		    be_fANDg_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_fANDg_E_d);
		    be_f_E_d = array_fetch(be_ptr, past_array_f_E,
					   min(d, lsf_info->past_depth));
		    assert(be_f_E_d);
		    be_g_E_d = array_fetch(be_ptr, past_array_g_E,
					   min(d, rsf_info->past_depth));
		    assert(be_g_E_d);
		    be_result = Be_Iff(be_mgr, be_fANDg_E_d,
				       Be_And(be_mgr, be_f_E_d, be_g_E_d));
		    assert(be_result);

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr,
			      "  [[f & g]]_E^%d <=> [[f]]_E^%d & [[g]]_E^%d: ",
			      d,
			      min(d, lsf_info->past_depth),
			      min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }

	case NOT:
	  {
	    unsigned int d;
	    array_t* past_array_f_E;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->past_depth == 0);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    past_array_f_E = array_fetch(array_t*, lsf_info->main_trans,
					 _E_state());
	    assert(past_array_f_E);
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  assert(array_fetch(be_ptr, E_past_array, d) != 0);
#else
		/* Definitional translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[!f]]_E^d := ![[f]]_E^d */
		    be_ptr be_f_E_d, be_result;
		    be_f_E_d = array_fetch(be_ptr, past_array_f_E, d);
		    assert(be_f_E_d);
		    be_result = Be_Not(be_mgr, be_f_E_d);
		    assert(array_fetch(be_ptr, E_past_array, d) == 0);
		    array_insert(be_ptr, E_past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  [!f]]_E^%d := ![[f]]_E^%d: ",
			      d, d),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
#endif
	      }
	    else
	      {
		/* State variable translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Add [[!f]]_E^d <=> ![[f]]_E^d */
		    be_ptr be_notf_E_d, be_f_E_d, be_result;
		    be_notf_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_notf_E_d);
		    be_f_E_d = array_fetch(be_ptr, past_array_f_E, d);
		    assert(be_f_E_d);
		    be_result = Be_Iff(be_mgr, be_notf_E_d,
				       Be_Not(be_mgr, be_f_E_d));
		    assert(be_result);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  [!f]]_E^%d <=> ![[f]]_E^%d: ",
			      d, d);
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
	      }
	    break;
	  }


	case OP_FUTURE:
	  {
	    unsigned int  d;
	    array_t*      lsf_E_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    assert(lsf_info->past_depth == info->past_depth);
	    lsf_E_past_array = array_fetch(array_t*,
					   lsf_info->main_trans, _E_state());
	    assert(lsf_E_past_array);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    {
	      /* Add LoopExists => ([[Ff]]_E^pd(Gf) => <<Ff>>_E */
	      be_ptr be_Ff_E_pd, be_auxFf_E, be_result;
	      be_Ff_E_pd = array_fetch(be_ptr, E_past_array, info->past_depth);
	      assert(be_Ff_E_pd);
	      assert(lsf_info->aux_F_node);
	      be_auxFf_E = BmcVarsMgr_Name2Timed(vars_mgr,
						 lsf_info->aux_F_node,
						 _E_state(), _E_state()+1);
	      assert(be_auxFf_E);
	      be_result = Be_Implies(be_mgr,
				     be_LoopExists,
				     Be_Implies(be_mgr,
						be_Ff_E_pd,
						be_auxFf_E));
	      
	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr,
			"  LoopExists => ([[Ff]]_E^%d => <<Ff>>_E): ",
			info->past_depth);
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }


	    if(do_optimization)
	      {
		/* Optimization.
		 * Add [[Ff]]_E^d <=> [[f]]_E^d | [[Ff]]_L^min(d+1,pd(f)) */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    be_ptr be_aux, be_lsf, be_aux_next, be_result;
		    be_aux = array_fetch(be_ptr, E_past_array, d);
		    assert(be_aux);
		    be_lsf = array_fetch(be_ptr, lsf_E_past_array, d);
		    assert(be_lsf);
		    be_aux_next = array_fetch(be_ptr, L_past_array,
					      min(d+1, info->past_depth));
		    assert(be_aux_next);
		    be_result = Be_Iff(be_mgr, be_aux,
				       Be_Or(be_mgr, be_lsf, be_aux_next));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[Ff]]_E^%d <=> [[f]]_E^%d | [[Ff]]_L^min(%d,%d): ", d, d, d+1, info->past_depth);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
		
	    if(do_optimization)
	      {
		/* Optimization:
		   Add [[Ff]]_E^d+1 => [[Ff]]_E^d */
		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Ff_E_d, be_Ff_E_dP1, be_result;
		    be_Ff_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_Ff_E_d);
		    be_Ff_E_dP1 = array_fetch(be_ptr, E_past_array, d+1);
		    assert(be_Ff_E_dP1);
		    be_result = Be_Implies(be_mgr, be_Ff_E_dP1, be_Ff_E_d);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, " [[Ff]]_E^%u => [[Ff]]_E^%u: ",
			      d+1, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    if(do_optimization)
	      {
		/* Optimization: Add <<Ff>>_E => [[Ff]]_E^pd(Gf) */
		be_ptr be_Ff_E_pd, be_auxFf_E, be_result;
		be_Ff_E_pd = array_fetch(be_ptr,E_past_array,info->past_depth);
		assert(be_Ff_E_pd);
		assert(lsf_info->aux_F_node);
		be_auxFf_E = BmcVarsMgr_Name2Timed(vars_mgr,
						   lsf_info->aux_F_node,
						   _E_state(), _E_state()+1);
		assert(be_auxFf_E);
		be_result = Be_Implies(be_mgr, be_auxFf_E, be_Ff_E_pd);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
		
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr,
			"   <<Gf>>_E => [[Gf]]_E^%d: ",
			info->past_depth);
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }	      
	    }


	    break;
	  }


	case OP_GLOBAL:
	  {
	    unsigned int  d;
	    array_t*      lsf_E_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    assert(lsf_info->past_depth == info->past_depth);
	    lsf_E_past_array = array_fetch(array_t*,
					   lsf_info->main_trans, _E_state());
	    assert(lsf_E_past_array);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    {
	      /* Add LoopExists => ([[Gf]]_E^pd(Gf) <= <<Gf>>_E^pd(Gf)) */
	      be_ptr be_Gf_E_pd, be_auxGf_E, be_result;
	      be_Gf_E_pd = array_fetch(be_ptr, E_past_array, info->past_depth);
	      assert(be_Gf_E_pd);
	      assert(lsf_info->aux_G_node);
	      be_auxGf_E = BmcVarsMgr_Name2Timed(vars_mgr,
						 lsf_info->aux_G_node,
						 _E_state(), _E_state()+1);
	      assert(be_auxGf_E);
	      be_result = Be_Implies(be_mgr,
				     be_LoopExists,
				     Be_Implies(be_mgr,
						be_auxGf_E,
						be_Gf_E_pd));
	      
	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr,
			"  LoopExists => ([[Gf]]_E^%d <= <<Gf>>_E^%d): ",
			info->past_depth, info->past_depth);
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }	      
	    }
	    
	    if(do_optimization)
	      {
		/* Optimization:
		   Add [[Gf]]_E^d <=> [[f]]_E^d & [[Gf]]_L^min(d+1,pd(Gf)) */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    be_ptr be_aux, be_lsf, be_aux_next, be_result;
		    be_aux = array_fetch(be_ptr, E_past_array, d);
		    assert(be_aux);
		    be_lsf = array_fetch(be_ptr, lsf_E_past_array, d);
		    assert(be_lsf);
		    be_aux_next = array_fetch(be_ptr, L_past_array,
					      min(d+1, info->past_depth));
		    assert(be_aux_next);
		    be_result = Be_Iff(be_mgr, be_aux,
				       Be_And(be_mgr, be_lsf, be_aux_next));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[Gf]]_E^%u <=> [[f]]_E^%u & [[Gf]]_L^min(%u+1,%u))", d, d, d, info->past_depth);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    if(do_optimization)
	      {
		/* Optimization:
		   Add [[Gf]]_E^d => [[Gf]]_E^d+1 */
		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Gf_E_d, be_Gf_E_dP1, be_result;
		    be_Gf_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_Gf_E_d);
		    be_Gf_E_dP1 = array_fetch(be_ptr, E_past_array, d+1);
		    assert(be_Gf_E_dP1);
		    be_result = Be_Implies(be_mgr, be_Gf_E_d, be_Gf_E_dP1);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, " [[Gf]]_E^%u => [[Gf]]_E^%u: ",
			      d, d+1);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }


	    if(do_optimization)
	      {
		/* Optimization: Add [[Gf]]_E^pd(Gf) => <<Gf>>_E */
		be_ptr be_Gf_E_pd, be_auxGf_E, be_result;
		be_Gf_E_pd = array_fetch(be_ptr,E_past_array,info->past_depth);
		assert(be_Gf_E_pd);
		assert(lsf_info->aux_G_node);
		be_auxGf_E = BmcVarsMgr_Name2Timed(vars_mgr,
						   lsf_info->aux_G_node,
						   _E_state(), _E_state()+1);
		assert(be_auxGf_E);
		be_result = Be_Implies(be_mgr, be_Gf_E_pd, be_auxGf_E);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
		
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr,
			"  [[Gf]]_E^%d => <<Gf>>_E): ",
			info->past_depth);
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }	      
	    }

	    break;
	  }


	case UNTIL:
	  {
	    unsigned int  d;
	    array_t*      past_array_f_E;
	    array_t*      past_array_g_E;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    past_array_f_E = array_fetch(array_t*,
					 lsf_info->main_trans, _E_state());
	    assert(past_array_f_E);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);	    
	    past_array_g_E = array_fetch(array_t*,
					 rsf_info->main_trans, _E_state());
	    assert(past_array_g_E);


	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    {
	      /* Add LoopExists => ([[fUg]]_E^pd(Gf) => <<Fg>>_E) */
	      be_ptr be_fUg_E_pd, be_auxFg_E, be_result;
	      be_fUg_E_pd = array_fetch(be_ptr,E_past_array,info->past_depth);
	      assert(be_fUg_E_pd);
	      assert(rsf_info->aux_F_node);
	      be_auxFg_E = BmcVarsMgr_Name2Timed(vars_mgr,
						 rsf_info->aux_F_node,
						 _E_state(), _E_state()+1);
	      assert(be_auxFg_E);
	      be_result = Be_Implies(be_mgr,
				     be_LoopExists,
				     Be_Implies(be_mgr,
						be_fUg_E_pd,
						be_auxFg_E));
	      
	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr,
			"  LoopExists => ([[fUg]]_E^%u => <<Fg>>_E): ",
			info->past_depth);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
	      }
	    }


	    if(do_optimization)
	      {
		/* Optimization: [[fUg]]_E^d <=>
		 * [[g]]_E^d | ([[f]]_E^d  & [[fUg]]_L^min(d+1,pd(fUg))) */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    be_ptr be_fUg_E_d, be_f_E_d, be_g_E_d, be_fUg_L_dP1;
		    be_ptr be_result;

		    be_fUg_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_fUg_E_d);

		    be_f_E_d = array_fetch(be_ptr, past_array_f_E,
					   min(d, lsf_info->past_depth));
		    assert(be_f_E_d);

		    be_g_E_d = array_fetch(be_ptr, past_array_g_E,
					   min(d, rsf_info->past_depth));
		    assert(be_g_E_d);

		    be_fUg_L_dP1 = array_fetch(be_ptr, L_past_array,
					       min(d+1, info->past_depth));
		    assert(be_fUg_L_dP1);

		    be_result = Be_Iff(be_mgr,
				       be_fUg_E_d,
				       Be_Or(be_mgr,
					     be_g_E_d,
					     Be_And(be_mgr,
						    be_f_E_d,
						    be_fUg_L_dP1)));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[fUg]]_E^%u <=> [[g]]_E^%u | ([[f]]_E^%u & [[fUg]]_L^min(%u,%u)): ",
			      d, min(d, rsf_info->past_depth),
			      min(d, lsf_info->past_depth),
			      d+1, info->past_depth);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }


	case RELEASES:
	  {
	    unsigned int  d;
	    array_t*      past_array_f_E;
	    array_t*      past_array_g_E;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    past_array_f_E = array_fetch(array_t*,
					 lsf_info->main_trans, _E_state());
	    assert(past_array_f_E);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);	    
	    past_array_g_E = array_fetch(array_t*,
					 rsf_info->main_trans, _E_state());
	    assert(past_array_g_E);


	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }


	    {
	      /* Add LoopExists => ([[fRg]]_E^pd(Gf) <= <<Gg>>_E) */
	      be_ptr be_fRg_E_pd, be_auxGg_E, be_result;
	      be_fRg_E_pd = array_fetch(be_ptr,E_past_array,info->past_depth);
	      assert(be_fRg_E_pd);
	      assert(rsf_info->aux_G_node);
	      be_auxGg_E = BmcVarsMgr_Name2Timed(vars_mgr,
						 rsf_info->aux_G_node,
						 _E_state(), _E_state()+1);
	      assert(be_auxGg_E);
	      be_result = Be_Implies(be_mgr,
				     be_LoopExists,
				     Be_Implies(be_mgr,
						be_auxGg_E,
						be_fRg_E_pd));
	      
	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr,
			"  LoopExists => ([[fRg]]_E^%u <= <<Gg>>_E): ",
			info->past_depth);
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }


	    if(do_optimization)
	      {
		/* Optimization: [[fRg]]_E^d <=>
		 * [[g]]_E^d & ([[f]]_E^d  | [[fRg]]_L^min(d+1,pd(fUg))) */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    be_ptr be_fRg_E_d, be_f_E_d, be_g_E_d, be_fRg_L_dP1;
		    be_ptr be_result;

		    be_fRg_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_fRg_E_d);

		    be_f_E_d = array_fetch(be_ptr, past_array_f_E,
					   min(d, lsf_info->past_depth));
		    assert(be_f_E_d);

		    be_g_E_d = array_fetch(be_ptr, past_array_g_E,
					   min(d, rsf_info->past_depth));
		    assert(be_g_E_d);

		    be_fRg_L_dP1 = array_fetch(be_ptr, L_past_array,
					       min(d+1, info->past_depth));
		    assert(be_fRg_L_dP1);

		    be_result = Be_Iff(be_mgr,
				       be_fRg_E_d,
				       Be_And(be_mgr,
					      be_g_E_d,
					      Be_Or(be_mgr,
						    be_f_E_d,
						    be_fRg_L_dP1)));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[fRg]]_E^%u <=> [[g]]_E^%u & ([[f]]_E^%u | [[fRg]]_L^min(%u,%u)): ",
			      d, min(d, rsf_info->past_depth),
			      min(d, lsf_info->past_depth),
			      d+1, info->past_depth);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }


	case OP_NEXT:
	  {
	    unsigned int  d;
	    array_t*      lsf_L_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    assert(lsf_info->past_depth == info->past_depth);
	    lsf_L_past_array = array_fetch(array_t*,
					   lsf_info->main_trans, _L_state());
	    assert(lsf_L_past_array);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    if(do_optimization)
	      {
		/* Add [[Xf]]_E^d <=> [[f]]_L^min(d+1,pd(f)) */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    be_ptr be_aux, be_lsf_next, be_result;
		    be_aux = array_fetch(be_ptr, E_past_array, d);
		    assert(be_aux);
		    be_lsf_next = array_fetch(be_ptr, lsf_L_past_array,
					      min(d+1, info->past_depth));
		    assert(be_lsf_next);
		    be_result = Be_Iff(be_mgr, be_aux, be_lsf_next);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr,
			      " [[Xf]]_E^%u <=> [[f]]_L^min(%u+1,%u): ",
			      d, d, lsf_info->past_depth);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    break;
	  }


	case OP_HISTORICAL:
	  {
	    unsigned int  d;
	    
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    if(do_optimization)
	      {
		/* Optimization:
		   Add [[Hf]]_E^d+1 => [[Hf]]_E^d */
		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Hf_E_d, be_Hf_E_dP1, be_result;
		    be_Hf_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_Hf_E_d);
		    be_Hf_E_dP1 = array_fetch(be_ptr, E_past_array, d+1);
		    assert(be_Hf_E_dP1);
		    be_result = Be_Implies(be_mgr, be_Hf_E_dP1, be_Hf_E_d);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, " [[Hf]]_E^%u => [[Hf]]_E^%u: ",
			      d+1, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }

	case OP_ONCE:
	  {
	    unsigned int  d;
	    
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);

	    if(do_optimization)
	      {
		/* Optimization:
		   Add [[Of]]_E^d => [[Of]]_E^d+1 */
		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Of_E_d, be_Of_E_dP1, be_result;
		    be_Of_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_Of_E_d);
		    be_Of_E_dP1 = array_fetch(be_ptr, E_past_array, d+1);
		    assert(be_Of_E_dP1);
		    be_result = Be_Implies(be_mgr, be_Of_E_d, be_Of_E_dP1);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, " [[Of]]_E^%u => [[Of]]_E^%u: ",
			      d, d+1);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }


	case OP_PREC:
	case OP_NOTPRECNOT:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);
	    break;
	  }

	case TRIGGERED:
	case SINCE:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->main_trans);
	    assert(L_past_array);
	    assert(E_past_array);
	    break;
	  }

	default:
	  print_node(stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      /* Remove node from unprocessed stack */
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Mark visited */
      _set_insert(visit_cache, node);


      if(info->main_nodes)
	{
	  /***
	   * Formula variable translation
	   * Add !LoopExists => ([[f]]_L^d == FALSE)
	   */
	  int d;
	  for(d = 0; d <= info->past_depth; d++)
	    {
	      be_ptr be_f_L_d, be_result;
	      be_f_L_d = array_fetch(be_ptr, L_past_array, d);
	      assert(be_f_L_d);
	      be_result = Be_Implies(be_mgr,
				     Be_Not(be_mgr, be_LoopExists),
				     Be_Not(be_mgr, be_f_L_d));
	  
	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, " f: ");
		print_node(nusmv_stderr, node);
		fprintf(nusmv_stderr, "\n");
		fprintf(nusmv_stderr,
			"  !LoopExists => ([[f]]_L^%d <=> FALSE): ", d);
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}

    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache); visit_cache = 0;

  return created_constraints;
}





/*
 * Create the k-invariant constraints at time i
 * Return a list of be_ptrs for the created constraints
 */
static lsList _HJL_unroll_invariant_f(BmcVarsMgr_ptr vars_mgr,
				      node_ptr ltlspec,
				      hash_ptr info_map,
				      const unsigned int i_model,
				      be_ptr be_InLoop_i,
				      be_ptr be_l_i,
				      const int do_optimization)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  lsList   created_constraints = 0;
  Be_Manager_ptr be_mgr;
  const unsigned int i_real = _real_k(i_model);
  unsigned int d;

  /* Get be manager */
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);

  /* Create list for the constraints */
  created_constraints = lsCreate();

  /* Debug output */
  if(opt_verbose_level_gt(options, 1))
    {
      fprintf(nusmv_stderr, "Unrolling k-invariant future stuff at time %u\n",
	      i_model);
      fflush(nusmv_stderr);
    }
  

  visit_cache = _set_create();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      _HJL_node_info *info, *lsf_info, *rsf_info;
      array_t *past_array, *lsf_past_array, *rsf_past_array;
      int has_unprocessed_children;
      
      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get info */
      info = (_HJL_node_info*)find_assoc(info_map, node);
      if(!info)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get past_array */
      /* _HJL_init_state_vector should have built main_trans[i] before */
      assert(info->main_trans);
      assert(array_n(info->main_trans) >= i_real+1);
      past_array = array_fetch(array_t*, info->main_trans, i_real);
      assert(past_array);


      /* Already build? */
      if(_set_is_in(visit_cache, node))
	{
	  for(d = 0; d <= info->past_depth; d++)
	    assert(array_fetch(be_ptr, past_array, d) != 0);
	  /* Remove node from unprocessed stack */
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}


      /* Traverse children and make k-invariant constraints */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{

	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	  {
	    be_ptr be_result;

	    assert(info->past_depth == 0);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		assert(array_fetch(be_ptr, past_array, 0) != 0);
#else
		/* Definitional translation */
		/* [[p]]_i^0 := p_i */
		assert(array_fetch(be_ptr, past_array, 0) == 0);
		be_result =
		  BmcVarsMgr_Name2Timed(vars_mgr, node, i_real, i_real + 1);
		assert(be_result);
		array_insert(be_ptr, past_array, 0, be_result);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[p]]_%d^0 := p_%d: ",
			  i_model, i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
#endif
		break;
	      }
	    else
	      {
		/* State variable translation
		 * Build [[p]]_i^0 <=> p_i
		 */
		be_ptr be_p_i_0, be_realp_i;
		be_p_i_0 = array_fetch(be_ptr, past_array, 0);
		assert(be_p_i_0);
		be_realp_i = BmcVarsMgr_Name2Timed(vars_mgr, node,
						   i_real, i_real + 1);
		assert(be_realp_i);
		
		be_result = Be_Iff(be_mgr, be_p_i_0, be_realp_i);
		assert(be_result);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[p]]_%d^0 <=> p_%d: ",
			  i_model, i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }	    
	    break;
	  }


	case TRUEEXP:
	  {
	    be_ptr be_result;

	    assert(info->past_depth == 0);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		assert(array_fetch(be_ptr, past_array, 0) != 0);
#else
		/* Definitional translation */
		/* [[TRUE]]_i^0 := TRUE */
		assert(array_fetch(be_ptr, past_array, 0) == 0);
		be_result = Be_Truth(be_mgr);
		array_insert(be_ptr, past_array, 0, be_result);
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[TRUE]]_%d^0 :=> TRUE: ", i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
#endif
	      }
	    else
	      {
		/*
		 * Build [[TRUE]]_i^0 <=> TRUE
		 */
		be_ptr be_p_i_0, be_true;
		be_p_i_0 = array_fetch(be_ptr, past_array, 0);
		assert(be_p_i_0);
		be_true = Be_Truth(be_mgr);
		assert(be_true);
		
		be_result = Be_Iff(be_mgr, be_p_i_0, be_true);
		assert(be_result);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[TRUE]]_%d^0 <=> TRUE: ", i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }
	    break;
	  }


	case FALSEEXP:
	  {
	    be_ptr be_result;

	    assert(info->past_depth == 0);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		assert(array_fetch(be_ptr, past_array, 0) != 0);
#else
		/* Definitional translation */
		/* [[FALSE]]_i^0 := FALSE */
		assert(array_fetch(be_ptr, past_array, 0) == 0);
		be_result = Be_Falsity(be_mgr);
		array_insert(be_ptr, past_array, 0, be_result);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[FALSE]]_%d^0 := FALSE: ",i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
#endif
	      }
	    else
	      {
		/*
		 * Build [[FALSE]]_i^0 <=> FALSE
		 */
		be_ptr be_p_i_0, be_false;
		be_p_i_0 = array_fetch(be_ptr, past_array, 0);
		assert(be_p_i_0);
		be_false = Be_Falsity(be_mgr);
		assert(be_false);
		
		be_result = Be_Iff(be_mgr, be_p_i_0, be_false);
		assert(be_result);

		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
		
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr,"  [[FALSE]]_%d^0 <=> FALSE: ",i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }
	    break;
	  }


	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	  {
	    internal_error("%s:%d: Formula not in NNF\n",__FILE__,__LINE__);
	    break;
	  }



	case OR:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(array_t*,rsf_info->main_trans,i_real);
	    assert(rsf_past_array);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  assert(array_fetch(be_ptr, past_array, d) != 0);
#else
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Definitional translation */
		    /* [[f | g]]_i^d := [[f]]_i^d | [[g]]_i^d */
		    be_ptr be_lsf, be_rsf, be_result;
		    be_lsf = array_fetch(be_ptr, lsf_past_array,
					 min(d, lsf_info->past_depth));
		    assert(be_lsf);
		    be_rsf = array_fetch(be_ptr, rsf_past_array,
					 min(d, rsf_info->past_depth));
		    assert(be_rsf);
		    be_result = Be_Or(be_mgr, be_lsf, be_rsf);
		    assert(array_fetch(be_ptr, past_array, d) == 0);
		    array_insert(be_ptr, past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[f | g]]_%d^%d := [[f]]_%d^%d | [[g]]_%d^%d: ",
			      i_model, d,
			      i_model, min(d, lsf_info->past_depth),
			      i_model, min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");	
		    }
		  }
#endif
	      }
	    else
	      {
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[f | g]]_i^d <=> [[f]]_i^d | [[g]]_i^d */
		    be_ptr be_fORg_i_d, be_f_i_d, be_g_i_d, be_result;
		    be_fORg_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_fORg_i_d);
		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array,
					   min(d, rsf_info->past_depth));
		    assert(be_g_i_d);
		    be_result = Be_Iff(be_mgr, be_fORg_i_d,
				       Be_Or(be_mgr, be_f_i_d, be_g_i_d));
		    assert(be_result);

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[f | g]]_%d^%d <=> [[f]]_%d^%d | [[g]]_%d^%d: ",
			      i_model, d,
			      i_model, min(d, lsf_info->past_depth),
			      i_model, min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    break;
	  }


	case AND:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(array_t*,rsf_info->main_trans,i_real);
	    assert(rsf_past_array);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  assert(array_fetch(be_ptr, past_array, d) != 0);
#else
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Definitional translation */
		    /* [[f & g]]_i^d := [[f]]_i^d & [[g]]_i^d */
		    be_ptr be_lsf, be_rsf, be_result;
		    be_lsf = array_fetch(be_ptr, lsf_past_array,
					 min(d, lsf_info->past_depth));
		    assert(be_lsf);
		    be_rsf = array_fetch(be_ptr, rsf_past_array,
					 min(d, rsf_info->past_depth));
		    assert(be_rsf);
		    be_result = Be_And(be_mgr, be_lsf, be_rsf);
		    assert(array_fetch(be_ptr, past_array, d) == 0);
		    array_insert(be_ptr, past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[f & g]]_%d^%d := [[f]]_%d^%d & [[g]]_%d^%d: ",
			      i_model, d,
			      i_model, min(d, lsf_info->past_depth),
			      i_model, min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");		 
		    }
		  }
#endif
	      }
	    else
	      {
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[f & g]]_i^d <=> [[f]]_i^d & [[g]]_i^d */
		    be_ptr be_fANDg_i_d, be_f_i_d, be_g_i_d, be_result;
		    be_fANDg_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_fANDg_i_d);
		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array,
					   min(d, rsf_info->past_depth));
		    assert(be_g_i_d);
		    be_result = Be_Iff(be_mgr,
				       be_fANDg_i_d,
				       Be_And(be_mgr, be_f_i_d, be_g_i_d));
		    assert(be_result);

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[f & g]]_%d^%d <=> [[f]]_%d^%d & [[g]]_%d^%d: ",
			      i_model, d,
			      i_model, min(d, lsf_info->past_depth),
			      i_model, min(d, rsf_info->past_depth));
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    break;
	  }



	case NOT:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    assert(info->past_depth == 0);

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");
	    
	    if(info->main_nodes == 0)
	      {
#if DEFDONE
		for(d = 0; d <= info->past_depth; d++)
		  assert(array_fetch(be_ptr, past_array, d) != 0);
#else
		/* Definitional translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* [[!f]]_i^d := ![[f]]_i^d */
		    be_ptr be_lsf, be_result;
		    be_lsf = array_fetch(be_ptr, lsf_past_array, d);
		    assert(be_lsf);
		    be_result = Be_Not(be_mgr, be_lsf);
		    assert(array_fetch(be_ptr, past_array, d) == 0);
		    array_insert(be_ptr, past_array, d, be_result);

		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  [!f]]_%d^%d := ![[f]]_%d^%d: ",
			      i_model, d, i_model, d),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
#endif
	      }
	    else
	      {
		/* State variable translation */
		for(d = 0; d <= info->past_depth; d++)
		  {
		    /* Add [[!f]]_i^d <=> ![[f]]_i^d */
		    be_ptr be_notf_i_d, be_f_i_d, be_result;
		    be_notf_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_notf_i_d);
		    be_f_i_d = array_fetch(be_ptr, lsf_past_array, d);
		    assert(be_f_i_d);
		    be_result = Be_Iff(be_mgr, be_notf_i_d,
				       Be_Not(be_mgr, be_f_i_d));
		    assert(be_result);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  [!f]]_%d^%d <=> ![[f]]_%d^%d: ",
			      i_model, d, i_model, d),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
	      }

	    break;
	  }


	case OP_NEXT:
	  {
	    array_t*      lsf_next_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_next_past_array = array_fetch(array_t*, lsf_info->main_trans,
					      i_real + 1);
	    assert(lsf_next_past_array);
	    
	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* Add [[Xf]]_i^d <=> [[f]]_{i+1}^d */
		be_ptr be_aux, be_lsf_next, be_result;
		be_aux = array_fetch(be_ptr, past_array, d);
		assert(be_aux);
		be_lsf_next = array_fetch(be_ptr, lsf_next_past_array, d);
		assert(be_lsf_next);
		be_result = Be_Iff(be_mgr, be_aux, be_lsf_next);
		assert(be_result);

		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr, "  [Xf]]_%d^%d <=> [[f]]_%d^%d: ",
			  i_model, d, i_model+1, d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		    fprintf(nusmv_stderr, "\n");
	      }
	    break;
	  }


	case OP_FUTURE:
	  {
	    array_t*      next_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;
	    
	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);
	    
	    next_past_array = array_fetch(array_t*, info->main_trans,
					  i_real+1);
	    assert(next_past_array);
	    
	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");
	    
	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* Add [[F f]]_i^d <=> [[f]]_i^d | [[F f]]_{i+1}^d */
		be_ptr be_aux, be_lsf, be_aux_next, be_result;
		be_aux = array_fetch(be_ptr, past_array, d);
		assert(be_aux);
		be_lsf = array_fetch(be_ptr, lsf_past_array, d);
		assert(be_lsf);
		be_aux_next = array_fetch(be_ptr, next_past_array, d);
		assert(be_aux_next);
		be_result = Be_Iff(be_mgr, be_aux,
				   Be_Or(be_mgr, be_lsf, be_aux_next));

		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr,
			  " [[F f]]_%d^%d <=> [[f]]_%d^%d | [[F f]]_%d^%d: ",
			  i_model, d, i_model, d, i_model+1, d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		    fprintf(nusmv_stderr, "\n");
	      }


	    if(do_optimization)
	      {
		/* Optimization: <<Ff>>_i => [[Ff]]_i^pd(Ff) */
		be_ptr be_Ff_i_pd, be_auxFf_i, be_result;

		assert(lsf_info->aux_F_node);
		be_auxFf_i = BmcVarsMgr_Name2Timed(vars_mgr,
						   lsf_info->aux_F_node,
						   i_real, i_real);
		assert(be_auxFf_i);

		be_Ff_i_pd = array_fetch(be_ptr, past_array, info->past_depth);
		assert(be_Ff_i_pd);
		
		be_result = Be_Implies(be_mgr, be_auxFf_i, be_Ff_i_pd);
		    
		/* Save the created constraint */
		lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  <<Ff>>_%u => [[Ff]]_%u^%u: ",
			  i_model, i_model, info->past_depth);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }


	    if(do_optimization)
	      {
		/* Optimization: InLoop_i => ([[Ff]]_i^d+1 => [[Ff]]_i^d) */
		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Ff_i_dP1, be_Ff_i_d, be_result;
		    
		    be_Ff_i_dP1 = array_fetch(be_ptr, past_array, d+1);
		    assert(be_Ff_i_dP1);
		    
		    be_Ff_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_Ff_i_d);
		    
		    be_result = Be_Implies(be_mgr,
					   be_InLoop_i,
					   Be_Implies(be_mgr,
						      be_Ff_i_dP1,
						      be_Ff_i_d));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  InLoop_%u => ([[Ff]]_%u^%d => [[Ff]]_%u^%d): ",
			      i_model, i_model, d+1, i_model, d),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
	      }
	    

	    if(do_optimization)
	      {
		/* Optimization: [[Ff]]_E^d => [[Ff]]_i^d */
		array_t* past_array_Ff_E =
		  array_fetch(array_t*, info->main_trans, _E_state());
		assert(past_array_Ff_E);

		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Ff_i_d, be_Ff_E_d, be_result;
		    
		    be_Ff_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_Ff_i_d);
		    
		    be_Ff_E_d = array_fetch(be_ptr, past_array_Ff_E, d);
		    assert(be_Ff_E_d);
		    
		    be_result = Be_Implies(be_mgr, be_Ff_E_d, be_Ff_i_d);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[Ff]]_E^%u => [[Ff]]_%u^%u: ",
			      d, i_model, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }


	case OP_GLOBAL:
	  {
	    array_t*      next_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(array_t*,lsf_info->main_trans,i_real);
	    assert(lsf_past_array);

	    next_past_array = array_fetch(array_t*,info->main_trans,i_real+1);
	    assert(next_past_array);

	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* Add [[Gf]]_i^d <=> [[f]]_i^d & [[Gf]]_{i+1}^d */
		be_ptr be_Gf_i_d, be_f_i_d, be_Gf_ip1_d, be_result;
		be_Gf_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_Gf_i_d);
		be_f_i_d = array_fetch(be_ptr, lsf_past_array, d);
		assert(be_f_i_d);
		be_Gf_ip1_d = array_fetch(be_ptr, next_past_array, d);
		assert(be_Gf_ip1_d);
		be_result = Be_Iff(be_mgr, be_Gf_i_d,
				   Be_And(be_mgr, be_f_i_d, be_Gf_ip1_d));

		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr,
			  "  [[Gf]]_%u^%d <=> [[f]]_%u^%d & [[Gf]]_%u^%d: ",
			  i_model, d, i_model, d, i_model+1, d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		    fprintf(nusmv_stderr, "\n");
	      }


	    if(do_optimization)
	      {
		/* Optimization: [[Gf]]_i^pd(Ff) => <<Gf>>_i */
		be_ptr be_Gf_i_pd, be_auxGf_i, be_result;

		assert(lsf_info->aux_G_node);
		be_auxGf_i = BmcVarsMgr_Name2Timed(vars_mgr,
						   lsf_info->aux_G_node,
						   i_real, i_real);
		assert(be_auxGf_i);

		be_Gf_i_pd = array_fetch(be_ptr, past_array, info->past_depth);
		assert(be_Gf_i_pd);
		
		be_result = Be_Implies(be_mgr, be_Gf_i_pd, be_auxGf_i);
		    
		/* Save the created constraint */
		lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		
		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  [[Gf]]_%u^%u => <<Gf>>_%u: ",
			  i_model, info->past_depth, i_model);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}
	      }


	    if(do_optimization)
	      {
		/* Optimization: InLoop_i => ([[Gf]]_i^d => [[Gf]]_i^d+1) */
		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Gf_i_d, be_Gf_i_dP1, be_result;
		    
		    be_Gf_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_Gf_i_d);
		    
		    be_Gf_i_dP1 = array_fetch(be_ptr, past_array, d+1);
		    assert(be_Gf_i_dP1);
		    
		    be_result = Be_Implies(be_mgr,
					   be_InLoop_i,
					   Be_Implies(be_mgr,
						      be_Gf_i_d,
						      be_Gf_i_dP1));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  InLoop_%u => ([[Gf]]_%u^%d => [[Gf]]_%u^%d): ",
			      i_model, i_model, d, i_model, d+1),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");
		  }
	      }


	    if(do_optimization)
	      {
		/* Optimization: [[Gf]]_i^d => [[Gf]]_E^d */
		array_t* past_array_Gf_E =
		  array_fetch(array_t*, info->main_trans, _E_state());
		assert(past_array_Gf_E);

		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Gf_i_d, be_Gf_E_d, be_result;
		    
		    be_Gf_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_Gf_i_d);
		    
		    be_Gf_E_d = array_fetch(be_ptr, past_array_Gf_E, d);
		    assert(be_Gf_E_d);
		    
		    be_result = Be_Implies(be_mgr, be_Gf_i_d, be_Gf_E_d);
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  [[Gf]]_%u^%u => [[Gf]]_E^%u: ",
			      i_model, d, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }
	    break;
	  }


	case UNTIL:
	  {
	    array_t*      next_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;
	    
	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(be_ptr, rsf_info->main_trans, i_real);
	    assert(rsf_past_array);
	    
	    next_past_array = array_fetch(array_t*, info->main_trans, i_real+1);
	    assert(next_past_array);
	    
	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* Add [[f U g]]_i^d <=>
		 *     [[g]]_i^d | ([[f]]_i^d & [[f U g]]_{i+1}^d)
		 */
		be_ptr be_fUg_i_d, be_f_i_d, be_g_i_d, be_fUg_iP1_d, be_result;

		be_fUg_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_fUg_i_d);
		be_f_i_d = array_fetch(be_ptr, lsf_past_array, d);
		assert(be_f_i_d);
		be_g_i_d = array_fetch(be_ptr, rsf_past_array, d);
		assert(be_g_i_d);
		be_fUg_iP1_d = array_fetch(be_ptr, next_past_array, d);
		assert(be_fUg_iP1_d);
		be_result = Be_Iff(be_mgr,
				   be_fUg_i_d,
				   Be_Or(be_mgr,
					 be_g_i_d,
					 Be_And(be_mgr,
						be_f_i_d,
						be_fUg_iP1_d)));
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr,
			  " [[f U g]]_%u^%d <=>"
			  " [[g]]_%u^%d | ([[f]]_%u^%d & [[f U g]]_%u^%d): ",
			  i_model, d, i_model, d, i_model, d, i_model+1, d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		    fprintf(nusmv_stderr, "\n");
	      }
	    
	    break;
	  }


	case RELEASES:
	  {
	    array_t*      next_past_array;

	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;
	    
	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);
	    
	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(be_ptr, rsf_info->main_trans, i_real);
	    assert(rsf_past_array);
	    
	    next_past_array = array_fetch(array_t*,info->main_trans,i_real+1);
	    assert(next_past_array);
	    
	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		/* Add [[f R g]]_i^d <=>
		 *     [[g]]_i^d & ([[f]]_i^d | [[f R g]]_{i+1}^d)
		 */
		be_ptr be_fRg_i_d, be_f_i_d, be_g_i_d, be_fRg_iP1_d, be_result;

		be_fRg_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_fRg_i_d);
		be_f_i_d = array_fetch(be_ptr, lsf_past_array, d);
		assert(be_f_i_d);
		be_g_i_d = array_fetch(be_ptr, rsf_past_array, d);
		assert(be_g_i_d);
		be_fRg_iP1_d = array_fetch(be_ptr, next_past_array, d);
		assert(be_fRg_iP1_d);
		be_result = Be_Iff(be_mgr,
				   be_fRg_i_d,
				   Be_And(be_mgr,
					  be_g_i_d,
					  Be_Or(be_mgr,
						be_f_i_d,
						be_fRg_iP1_d)));
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr,
			  " [[f R g]]_%u^%d <=>"
			  " [[g]]_%u^%d & ([[f]]_%u^%d | [[f R g]]_%u^%d): ",
			  i_model, d, i_model, d, i_model, d, i_model+1, d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		    fprintf(nusmv_stderr, "\n");
	      }

	    break;
	  }


	case OP_ONCE:
	case OP_HISTORICAL:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    for(d = 0; d <= info->past_depth; d++)
	      assert(array_fetch(be_ptr, past_array, d) != 0);

	    break;
	  }


	case SINCE:
	case TRIGGERED:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    for(d = 0; d <= info->past_depth; d++)
	      assert(array_fetch(be_ptr, past_array, d) != 0);

	    break;
	  }


	case OP_PREC:
	case OP_NOTPRECNOT:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    for(d = 0; d <= info->past_depth; d++)
	      assert(array_fetch(be_ptr, past_array, d) != 0);

	    break;
	  }


	default:
	  print_node(stderr, node);
	  internal_error(_SNYI_text,__FILE__,__LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;

      /* Do the auxiliary translations if necessary */
      if(info->aux_F_node)
	{
	  be_ptr be_aux_i, be_aux_i_minus_1, be_f_i_pd;
	  be_ptr be_result;

	  be_aux_i = BmcVarsMgr_Name2Timed(vars_mgr, info->aux_F_node,
					   i_real, i_real);
	  assert(be_aux_i);

	  if(opt_verbose_level_gt(options, 5))
	    fprintf(nusmv_stderr, " f: "),
	      print_node(nusmv_stderr, node),
	      fprintf(nusmv_stderr, "\n");

	  if(i_model == 0)
	    {
	      /* <<F f>>_0 <=> FALSE */
	      be_result = Be_Iff(be_mgr, be_aux_i, Be_Falsity(be_mgr));

	      if(opt_verbose_level_gt(options, 5))
		fprintf(nusmv_stderr, "  <<F f>>_0 <=> FALSE: "),
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		  fprintf(nusmv_stderr, "\n");
	    }
	  else
	    {	    
	      /* <<F f>>_i <=> <<F f>>_{i-1} | (InLoop_i & [[f]]_i^PD(f)) */
	      be_aux_i_minus_1 = BmcVarsMgr_Name2Timed(vars_mgr,
						       info->aux_F_node,
						       i_real-1, i_real-1);
	      assert(be_aux_i_minus_1);
	      be_f_i_pd = array_fetch(be_ptr, past_array, info->past_depth);
	      assert(be_f_i_pd);
	      be_result = Be_Iff(be_mgr,
				 be_aux_i,
				 Be_Or(be_mgr,
				       be_aux_i_minus_1,
				       Be_And(be_mgr,
					      be_InLoop_i,
					      be_f_i_pd)));

	      if(opt_verbose_level_gt(options, 5))
		fprintf(nusmv_stderr,
			"  <<F f>>_%u <=> <<F f>>_%u | (InLoop_%u & [[f]]_%u^%d): ", i_model, i_model-1, i_model, i_model, info->past_depth),
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		  fprintf(nusmv_stderr, "\n");
	    }

	  /* Save the created constraint */
	  lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	}
      if(info->aux_F_node && do_optimization)
	{
	  /* Optimization: add <<Ff>>_i => <<Ff>>_E */
	  be_ptr be_aux_i, be_aux_E;
	  be_ptr be_result;

	  be_aux_i = BmcVarsMgr_Name2Timed(vars_mgr, info->aux_F_node,
					   i_real, i_real);
	  assert(be_aux_i);

	  be_aux_E = BmcVarsMgr_Name2Timed(vars_mgr, info->aux_F_node,
					   _E_state(), _E_state());
	  assert(be_aux_E);

	  be_result = Be_Implies(be_mgr, be_aux_i, be_aux_E);

	  /* Save the created constraint */
	  lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	  
	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "  <<F f>>_%u <=> <<F f>>_E: ", i_model);
	    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}

      if(info->aux_G_node)
	{
	  be_ptr be_aux_i, be_aux_i_minus_1, be_f_i_pd;
	  be_ptr be_result;

	  be_aux_i = BmcVarsMgr_Name2Timed(vars_mgr, info->aux_G_node, i_real, i_real);
	  assert(be_aux_i);

	  if(opt_verbose_level_gt(options, 5))
	    fprintf(nusmv_stderr, " f: "),
	      print_node(nusmv_stderr, node),
	      fprintf(nusmv_stderr, "\n");

	  if(i_model == 0)
	    {
	      /* <<G f>>_0 <=> TRUE */
	      be_result = Be_Iff(be_mgr, be_aux_i, Be_Truth(be_mgr));

	      if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr, "  <<G f>>_0 <=> TRUE: "),
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		  fprintf(nusmv_stderr, "\n");
	    }
	  else
	    {	    
	      /* <<G f>>_i <=> <<G f>>_{i-1} & (!InLoop_i | [[f]]_i^PD(f)) */
	      be_aux_i_minus_1 = BmcVarsMgr_Name2Timed(vars_mgr,
						       info->aux_G_node,
						       i_real-1, i_real-1);
	      assert(be_aux_i_minus_1);
	      be_f_i_pd = array_fetch(be_ptr, past_array, info->past_depth);
	      assert(be_f_i_pd);
	      be_result = Be_Iff(be_mgr,
				 be_aux_i,
				 Be_And(be_mgr,
					be_aux_i_minus_1,
					Be_Or(be_mgr,
					      Be_Not(be_mgr, be_InLoop_i),
					      be_f_i_pd)));

	      if(opt_verbose_level_gt(options, 5))
		fprintf(nusmv_stderr, "  <<G f>>_%u <=> <<G f>>_%u & (!InLoop_%u | [[f]]_%u^%d): ", i_model, i_model-1, i_model, i_model, info->past_depth),
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		  fprintf(nusmv_stderr, "\n");
	    }
	  /* Save the created constraint */
	  lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	}
      if(info->aux_G_node && do_optimization)
	{
	  /* Optimization: add <<Gf>>_E => <<Gf>>_i */
	  be_ptr be_aux_i, be_aux_E;
	  be_ptr be_result;

	  be_aux_i = BmcVarsMgr_Name2Timed(vars_mgr, info->aux_G_node,
					   i_real, i_real);
	  assert(be_aux_i);

	  be_aux_E = BmcVarsMgr_Name2Timed(vars_mgr, info->aux_G_node,
					   _E_state(), _E_state());
	  assert(be_aux_E);
	  
	  be_result = Be_Implies(be_mgr, be_aux_E, be_aux_i);

	  /* Save the created constraint */
	  lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	  
	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "  <<F f>>_E <=> <<F f>>_%u: ", i_model);
	    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}

      
      if(info->main_nodes)
	{
	  /*
	   * Add (l_i => ([[f]]_L^d <=> [[f]]_i^d)) for
	   * formula variable translated subformulae
	   */
	  unsigned int d;
	  array_t *f_L_past_array;

	  f_L_past_array =
	    array_fetch(array_t*, info->main_trans, _L_state());
	  assert(f_L_past_array);
	  
	  if(opt_verbose_level_gt(options, 5))
	    fprintf(nusmv_stderr, " f: "),
	      print_node(nusmv_stderr, node),
	      fprintf(nusmv_stderr, "\n");
	  
	  for(d = 0; d <= info->past_depth; d++)
	    {
	      be_ptr be_f_L_d, be_f_i_d, be_result;

	      be_f_i_d = array_fetch(be_ptr, past_array, d);
	      assert(be_f_i_d);

	      be_f_L_d = array_fetch(be_ptr, f_L_past_array, d);
	      assert(be_f_L_d);

	      be_result = Be_Implies(be_mgr,
				     be_l_i,
				     Be_Iff(be_mgr, be_f_L_d, be_f_i_d));

	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

	      if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr,
			  "  l_%u => ([[f]]_L_%d <=> [[f]]_%u^%d): ",
			  i_model, d, i_model, d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
		    fprintf(nusmv_stderr, "\n");
	    }
	}

      
      /* Remove node from unprocessed stack */
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Mark visited */
      _set_insert(visit_cache, node);
    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache); visit_cache = 0;

  return created_constraints;
}

#if 0
static unsigned int _get_past_depth(const hash_ptr info_map,
				    const node_ptr node)
{
  _HJL_node_info* info = (_HJL_node_info*)find_assoc(info_map, node);
  assert(info);
  return info->past_depth;
}


static be_ptr _get_timed_be(const hash_ptr info_map,
			    const node_ptr node,
			    const unsigned int i_real,
			    const unsigned int depth)
{
  _HJL_node_info* info = 0;
  array_t* past_array = 0;
  be_ptr result = 0;

  info = (_HJL_node_info*)find_assoc(info_map, node);
  assert(info);
  assert(depth <= info->past_depth);

  assert(info->main_trans);
  assert(array_n(info->main_trans) >= i_real+1);
  past_array = array_fetch(array_t*, info->main_trans, i_real);
  assert(past_array);

  result = array_fetch(be_ptr, past_array, depth);
  assert(result);

  return result;
}
#endif

/*
 * Create the k-invariant constraints at time i
 * Return a list of be_ptrs for the created constraints
 */
static lsList _HJL_unroll_invariant_p(BmcVarsMgr_ptr vars_mgr,
				      node_ptr ltlspec,
				      hash_ptr info_map,
				      const unsigned int i_model,
				      be_ptr be_InLoop_i,
				      be_ptr be_l_i,
				      const int do_optimization)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  lsList   created_constraints = 0;
  Be_Manager_ptr be_mgr;
  const unsigned int i_real = _real_k(i_model);
  unsigned int d;

  /* Get be manager */
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);

  /* Create list for the constraints */
  created_constraints = lsCreate();

  /* Debug output */
  if(opt_verbose_level_gt(options, 1))
    {
      fprintf(nusmv_stderr, "Unrolling k-invariant past stuff at time %u\n",
	      i_model);
      fflush(nusmv_stderr);
    }
  

  visit_cache = _set_create();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);
  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      _HJL_node_info *info, *lsf_info, *rsf_info;
      array_t *past_array, *lsf_past_array, *rsf_past_array;
      int has_unprocessed_children;
      
      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get info */
      info = (_HJL_node_info*)find_assoc(info_map, node);
      if(!info)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get past_array */
      /* _HJL_init_state_vector should have built main_trans[i] before */
      assert(info->main_trans);
      assert(array_n(info->main_trans) >= i_real+1);
      past_array = array_fetch(array_t*, info->main_trans, i_real);
      assert(past_array);


      /* Already build? */
      if(_set_is_in(visit_cache, node))
	{
	  for(d = 0; d <= info->past_depth; d++)
	    assert(array_fetch(be_ptr, past_array, d) != 0);
	  /* Remove node from unprocessed stack */
	  if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	    internal_error(_SNH_text,__FILE__,__LINE__);
	  continue;
	}


      /* Traverse children and make k-invariant constraints */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{
	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	  {
	    assert(info->past_depth == 0);
	    break;
	  }


	case TRUEEXP:
	case FALSEEXP:
	  {
	    assert(info->past_depth == 0);
	    break;
	  }


	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	  {
	    internal_error("%s:%d: Formula not in NNF\n",__FILE__,__LINE__);
	    break;
	  }


	case OR:
	case AND:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    break;
	  }


	case NOT:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    break;
	  }


	case OP_NEXT:
	case OP_FUTURE:
	case OP_GLOBAL:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    break;
	  }


	case UNTIL:
	case RELEASES:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;
	    
	    break;
	  }


	case OP_ONCE:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);

	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node),
		fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		be_ptr be_Of_i_d, be_result;
		const unsigned int d_lsf = min(d, lsf_info->past_depth);

		be_Of_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_Of_i_d);

		if(i_model == 0)
		  {
		    /* Add [[O f]]_0^d <=> [[f]]_0^0 */
		    be_ptr be_f_0_0 = array_fetch(be_ptr, lsf_past_array, 0);
		    assert(be_f_0_0);
		    be_result = Be_Iff(be_mgr, be_Of_i_d, be_f_0_0);
		  }
		else if(i_model >= 1 && d == 0)
		  {
		    /* Add [[O f]]_i^0 <=> [[f]]_i^0 | [[O f]]_{i-1}^0*/
		    be_ptr be_f_i_d, be_Of_iM1_d;
		    array_t* past_array_Of_iM1;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array, d_lsf);
		    assert(be_f_i_d);

		    past_array_Of_iM1 = array_fetch(array_t*, info->main_trans,
						    i_real - 1);
		    assert(past_array_Of_iM1);
		    be_Of_iM1_d = array_fetch(be_ptr, past_array_Of_iM1, d);
		    assert(be_Of_iM1_d);

		    be_result = Be_Iff(be_mgr,
				       be_Of_i_d,
				       Be_Or(be_mgr, be_f_i_d, be_Of_iM1_d));
		  }
		else if(i_model >= 1 && d > 0)
		  {
		    /*
		     * Add [[O f]]_i^d <=>
		     * [[f]]_i^d | ITE(l_i,[[O f]]_E^{d-1},[[O f]]_{i-1}^d)
		     */
		    be_ptr be_prev, be_E, be_f_i_d;
		    array_t* prev_past_array;
		    array_t* E_past_array;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array, d_lsf);
		    assert(be_f_i_d);

		    prev_past_array =
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(prev_past_array);
		    be_prev = array_fetch(be_ptr, prev_past_array, d);
		    assert(be_prev);
		    
		    E_past_array =
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(E_past_array);
		    be_E = array_fetch(be_ptr, E_past_array, d-1);
		    assert(be_E);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Of_i_d,
				       Be_Or(be_mgr,
					     be_f_i_d,
					     Be_Ite(be_mgr,
						    be_l_i, be_E, be_prev)));
		  }
		else
		  internal_error(_SNH_text,__FILE__,__LINE__);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}

		/* The additional constraints ensuring that the past formulae
		   have stabilized at the last unrolling level */
		if(i_model >= 1 && d == info->past_depth)
		  {
		    /*
		     * Add [[O f]]_i^d <=>
		     * [[f]]_i^d | ITE(l_i,[[O f]]_E^d,[[O f]]_{i-1}^d)
		     */
		    be_ptr be_Of_iM1_d = 0;
		    be_ptr be_Of_E_d = 0;
		    be_ptr be_f_i_d = 0;
		    array_t* prev_past_array = 0;
		    array_t* E_past_array = 0;

		    /*be_f_i_d = _get_timed_be(info_map, lsf, i_real, d_lsf);*/
		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);

		    prev_past_array =
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(prev_past_array);
		    be_Of_iM1_d = array_fetch(be_ptr, prev_past_array, d);
		    assert(be_Of_iM1_d);
		    
		    E_past_array =
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(E_past_array);
		    be_Of_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_Of_E_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Of_i_d,
				       Be_Or(be_mgr,
					     be_f_i_d,
					     Be_Ite(be_mgr,
						    be_l_i,
						    be_Of_E_d,
						    be_Of_iM1_d)));
		
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }

	      } /*for(d = 0; d <= info->past_depth; d++)*/

	    if(do_optimization)
	      {
		/* Optimization: InLoop_i => ([[Of]]_i^d => [[Of]]_i^d+1) */
		/* Optimization: InLoop_i => ([[Of]]_i^d => [[Of]]_E^d) */
		array_t* past_array_Of_E =
		  array_fetch(array_t*, info->main_trans, _E_state());
		assert(past_array_Of_E);

		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Of_i_d, be_Of_i_dP1, be_Of_E_d, be_result;
		    
		    be_Of_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_Of_i_d);

		    be_Of_i_dP1 = array_fetch(be_ptr, past_array, d+1);
		    assert(be_Of_i_dP1);
		    
		    be_result = Be_Implies(be_mgr,
					   be_InLoop_i,
					   Be_Implies(be_mgr,
						      be_Of_i_d,
						      be_Of_i_dP1));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5))
		      fprintf(nusmv_stderr, "  InLoop_%u => ([[Of]]_%d^%d => [[Of]]_%d^%d): ",
			      i_model, i_model, d, i_model, d+1),
			Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr),
			fprintf(nusmv_stderr, "\n");


		    be_Of_E_d = array_fetch(be_ptr, past_array_Of_E, d);
		    assert(be_Of_E_d);
		    be_result = Be_Implies(be_mgr,
					   be_InLoop_i,
					   Be_Implies(be_mgr,
						      be_Of_i_d,
						      be_Of_E_d));

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  InLoop_%u => ([[Of]]_%d^%d => [[Of]]_E^%d): ",
			      i_model, i_model, d, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    break;
	  }


	case OP_HISTORICAL:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    for(d = 0; d <= info->past_depth; d++)
	      {
		be_ptr be_Hf_i_d, be_lsf, be_result;
		const unsigned int d_lsf = min(d, lsf_info->past_depth);

		be_Hf_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_Hf_i_d);

		be_lsf = array_fetch(be_ptr, lsf_past_array, d_lsf);
		assert(be_lsf);

		if(i_model == 0)
		  {
		    /* Add [[H f]]_0^0 <=> [[f]]_0^0 */
		    be_ptr be_f_0_0 = array_fetch(be_ptr, lsf_past_array, 0);
		    assert(be_f_0_0);
		    be_result = Be_Iff(be_mgr, be_Hf_i_d, be_f_0_0);
		  }
		else if(i_model >= 1 && d == 0)
		  {
		    /* Add [[H f]]_i^0 <=> [[f]]_i^0 & [[H f]]_{i-1}^0*/
		    be_ptr be_prev;
		    array_t* prev_past_array =
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(prev_past_array);
		    be_prev = array_fetch(be_ptr, prev_past_array, d);
		    assert(be_prev);
		    be_result = Be_Iff(be_mgr,
				       be_Hf_i_d,
				       Be_And(be_mgr, be_lsf, be_prev));
		  }
		else if(i_model >= 1 && d > 0)
		  {
		    /*
		     * Add [[H f]]_i^d <=>
		     * [[f]]_i^d & ITE(l_i,[[H f]]_E^{d-1},[[H f]]_{i-1}^d)
		     */
		    be_ptr be_prev, be_E;
		    array_t* prev_past_array;
		    array_t* E_past_array;

		    prev_past_array =
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(prev_past_array);
		    be_prev = array_fetch(be_ptr, prev_past_array, d);
		    assert(be_prev);
		    
		    E_past_array =
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(E_past_array);
		    be_E = array_fetch(be_ptr, E_past_array, d-1);
		    assert(be_E);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Hf_i_d,
				       Be_And(be_mgr,
					      be_lsf,
					      Be_Ite(be_mgr,
						     be_l_i, be_E, be_prev)));
		  }
		else
		  internal_error(_SNH_text,__FILE__,__LINE__);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}

		/* The additional constraints ensuring that the past formulae
		   have stabilized at the last unrolling level */
		if(i_model >= 1 && d == info->past_depth)
		  {
		    /*
		     * Add [[H f]]_i^d <=>
		     * [[f]]_i^d & ITE(l_i,[[H f]]_E^d,[[H f]]_{i-1}^d)
		     */
		    be_ptr be_Hf_iM1_d = 0;
		    be_ptr be_Hf_E_d = 0;
		    array_t* prev_past_array;
		    array_t* E_past_array;

		    prev_past_array =
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(prev_past_array);
		    be_Hf_iM1_d = array_fetch(be_ptr, prev_past_array, d);
		    assert(be_Hf_iM1_d);
		    
		    E_past_array =
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(E_past_array);
		    be_Hf_E_d = array_fetch(be_ptr, E_past_array, d);
		    assert(be_Hf_E_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Hf_i_d,
				       Be_And(be_mgr,
					      be_lsf,
					      Be_Ite(be_mgr,
						     be_l_i,
						     be_Hf_E_d,
						     be_Hf_iM1_d)));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }

	      } /*for(d = 0; d <= info->past_depth; d++)*/

	    if(do_optimization)
	      {
		/* Optimization: InLoop_i => ([[Hf]]_i^d+1 => [[Hf]]_i^d) */
		/* Optimization: InLoop_i => ([[Hf]]_E^d => [[Hf]]_i^d) */
		array_t* past_array_Hf_E =
		  array_fetch(array_t*, info->main_trans, _E_state());
		assert(past_array_Hf_E);

		for(d = 0; d < info->past_depth; d++)
		  {
		    be_ptr be_Hf_i_d, be_Hf_i_dP1, be_Hf_E_d, be_result;
		    
		    be_Hf_i_d = array_fetch(be_ptr, past_array, d);
		    assert(be_Hf_i_d);
		    
		    be_Hf_i_dP1 = array_fetch(be_ptr, past_array, d+1);
		    assert(be_Hf_i_dP1);
		    
		    be_result = Be_Implies(be_mgr,
					   be_InLoop_i,
					   Be_Implies(be_mgr,
						      be_Hf_i_dP1,
						      be_Hf_i_d));
		    
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  InLoop_%u => ([[Hf]]_%d^%d => [[Hf]]_%d^%d): ",
			      i_model, i_model, d+1, i_model, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }


		    be_Hf_E_d = array_fetch(be_ptr, past_array_Hf_E, d);
		    assert(be_Hf_E_d);

		    be_result = Be_Implies(be_mgr,
					   be_InLoop_i,
					   Be_Implies(be_mgr,
						      be_Hf_E_d,
						      be_Hf_i_d));

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  InLoop_%u => ([[Hf]]_E^%d => [[Hf]]_%d^%d): ",
			      i_model, d, i_model, d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }
	      }

	    break;
	  }


	case SINCE:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);

	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(be_ptr, rsf_info->main_trans, i_real);
	    assert(rsf_past_array);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    for(d = 0; d <= info->past_depth; d++)
	      {
		be_ptr be_fSg_i_d = 0;
		be_ptr be_result = 0;

		be_fSg_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_fSg_i_d);
		
		if(i_model == 0)
		  {
		    /* Add [[f S g]]_0^d <=> [[g]]_0^0 */
		    be_ptr be_g_0_0 = array_fetch(be_ptr, rsf_past_array, 0);
		    assert(be_g_0_0);
		    be_result = Be_Iff(be_mgr, be_fSg_i_d, be_g_0_0);
		  }
		else if(i_model >= 1 && d == 0)
		  {
		    /* Add [[f S g]]_i^0 <=>
		       [[g]]_i^0 | ([[f]]_i^0 & [[f S g]]_{i-1}^0)*/
		    be_ptr be_f_i_d, be_g_i_d, be_fSg_iM1_d;
		    array_t* past_array_fSg_iM1;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array, d);
		    assert(be_f_i_d);
		    
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array, d);
		    assert(be_g_i_d);

		    past_array_fSg_iM1 = 
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(past_array_fSg_iM1);
		    be_fSg_iM1_d = array_fetch(be_ptr, past_array_fSg_iM1, d);
		    assert(be_fSg_iM1_d);

		    be_result = Be_Iff(be_mgr,
				       be_fSg_i_d,
				       Be_Or(be_mgr,
					     be_g_i_d,
					     Be_And(be_mgr,
						    be_f_i_d,
						    be_fSg_iM1_d)));
		  }
		else if(i_model >= 1 && d > 0)
		  {
		    /*
		     * Add [[f S g]]_i^d <=>
		     * [[g]]_i^d | ([[f]]_i_d &
		     *              ITE(l_i,[[fSg]]_E^{d-1},[[fSg]]_{i-1}^d))
		     */
		    be_ptr be_f_i_d, be_g_i_d, be_fSg_E_dM1, be_fSg_iM1_d;
		    array_t* past_array_fSg_iM1;
		    array_t* past_array_fSg_E;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);
		    
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array,
					   min(d, rsf_info->past_depth));
		    assert(be_g_i_d);

		    past_array_fSg_E = 
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(past_array_fSg_E);
		    be_fSg_E_dM1 = array_fetch(be_ptr, past_array_fSg_E, d-1);
		    assert(be_fSg_E_dM1);

		    past_array_fSg_iM1 = 
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(past_array_fSg_iM1);
		    be_fSg_iM1_d = array_fetch(be_ptr, past_array_fSg_iM1, d);
		    assert(be_fSg_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_fSg_i_d,
				       Be_Or(be_mgr,
					     be_g_i_d,
					     Be_And(be_mgr,
						    be_f_i_d,
						    Be_Ite(be_mgr,
							   be_l_i,
							   be_fSg_E_dM1,
							   be_fSg_iM1_d))));
		  }
		else
		  internal_error(_SNH_text,__FILE__,__LINE__);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}

		/* The additional constraints ensuring that the past formulae
		   have stabilized at the last unrolling level */
		if(i_model >= 1 && d == info->past_depth)
		  {
		    /*
		     * Add [[f S g]]_i^d <=>
		     * [[g]]_i^d | ([[f]]_i_d &
		     *              ITE(l_i,[[fSg]]_E^d,[[fSg]]_{i-1}^d))
		     */
		    be_ptr be_f_i_d = 0;
		    be_ptr be_g_i_d = 0;
		    be_ptr be_fSg_E_d = 0;
		    be_ptr be_fSg_iM1_d = 0;
		    array_t* past_array_fSg_iM1 = 0;
		    array_t* past_array_fSg_E = 0;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);
		    
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array,
					   min(d, rsf_info->past_depth));
		    assert(be_g_i_d);

		    past_array_fSg_E = 
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(past_array_fSg_E);
		    be_fSg_E_d = array_fetch(be_ptr, past_array_fSg_E, d);
		    assert(be_fSg_E_d);

		    past_array_fSg_iM1 = 
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(past_array_fSg_iM1);
		    be_fSg_iM1_d = array_fetch(be_ptr, past_array_fSg_iM1, d);
		    assert(be_fSg_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_fSg_i_d,
				       Be_Or(be_mgr,
					     be_g_i_d,
					     Be_And(be_mgr,
						    be_f_i_d,
						    Be_Ite(be_mgr,
							   be_l_i,
							   be_fSg_E_d,
							   be_fSg_iM1_d))));
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }

	      } /* for(d = 0; d <= info->past_depth; d++) */


	    break;
	  }


	case TRIGGERED:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);

	    rsf_info = (_HJL_node_info*)find_assoc(info_map, rsf);
	    assert(rsf_info);
	    rsf_past_array = array_fetch(be_ptr, rsf_info->main_trans, i_real);
	    assert(rsf_past_array);

	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " "); print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	    }

	    for(d = 0; d <= info->past_depth; d++)
	      {
		be_ptr be_fTg_i_d, be_result;

		be_fTg_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_fTg_i_d);
		
		if(i_model == 0)
		  {
		    /* Add [[f T g]]_0^d <=> [[g]]_0^0 */
		    be_ptr be_g_0_0 = array_fetch(be_ptr, rsf_past_array, 0);
		    assert(be_g_0_0);
		    be_result = Be_Iff(be_mgr, be_fTg_i_d, be_g_0_0);
		  }
		else if(i_model >= 1 && d == 0)
		  {
		    /* Add [[f T g]]_i^0 <=>
		       [[g]]_i^0 & ([[f]]_i^0 | [[f T g]]_{i-1}^0)*/
		    be_ptr be_f_i_d, be_g_i_d, be_fTg_iM1_d;
		    array_t* past_array_fTg_iM1;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array, d);
		    assert(be_f_i_d);
		    
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array, d);
		    assert(be_g_i_d);

		    past_array_fTg_iM1 = 
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(past_array_fTg_iM1);
		    be_fTg_iM1_d = array_fetch(be_ptr, past_array_fTg_iM1, d);
		    assert(be_fTg_iM1_d);

		    be_result = Be_Iff(be_mgr,
				       be_fTg_i_d,
				       Be_And(be_mgr,
					      be_g_i_d,
					      Be_Or(be_mgr,
						    be_f_i_d,
						    be_fTg_iM1_d)));
		  }
		else if(i_model >= 1 && d > 0)
		  {
		    /*
		     * Add [[f T g]]_i^d <=>
		     * [[g]]_i^d & ([[f]]_i_d |
		     *              ITE(l_i,[[fTg]]_E^{d-1},[[fTg]]_{i-1}^d))
		     */
		    be_ptr be_f_i_d, be_g_i_d, be_fTg_E_dM1, be_fTg_iM1_d;
		    array_t* past_array_fTg_iM1;
		    array_t* past_array_fTg_E;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);
		    
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array,
					   min(d, rsf_info->past_depth));
		    assert(be_g_i_d);

		    past_array_fTg_E = 
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(past_array_fTg_E);
		    be_fTg_E_dM1 = array_fetch(be_ptr, past_array_fTg_E, d-1);
		    assert(be_fTg_E_dM1);

		    past_array_fTg_iM1 = 
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(past_array_fTg_iM1);
		    be_fTg_iM1_d = array_fetch(be_ptr, past_array_fTg_iM1, d);
		    assert(be_fTg_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_fTg_i_d,
				       Be_And(be_mgr,
					      be_g_i_d,
					      Be_Or(be_mgr,
						    be_f_i_d,
						    Be_Ite(be_mgr,
							   be_l_i,
							   be_fTg_E_dM1,
							   be_fTg_iM1_d))));
		  }
		else
		  internal_error(_SNH_text,__FILE__,__LINE__);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}


		/* The additional constraints ensuring that the past formulae
		   have stabilized at the last unrolling level */
		if(i_model >= 1 && d == info->past_depth)
		  {
		    /*
		     * Add [[f T g]]_i^d <=>
		     * [[g]]_i^d & ([[f]]_i_d |
		     *              ITE(l_i,[[fTg]]_E^{d},[[fTg]]_{i-1}^d))
		     */
		    be_ptr be_f_i_d = 0;
		    be_ptr be_g_i_d = 0;
		    be_ptr be_fTg_E_d = 0;
		    be_ptr be_fTg_iM1_d = 0;
		    array_t* past_array_fTg_iM1 = 0;
		    array_t* past_array_fTg_E = 0;

		    be_f_i_d = array_fetch(be_ptr, lsf_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_i_d);
		    
		    be_g_i_d = array_fetch(be_ptr, rsf_past_array,
					   min(d, rsf_info->past_depth));
		    assert(be_g_i_d);

		    past_array_fTg_E = 
		      array_fetch(array_t*, info->main_trans, _E_state());
		    assert(past_array_fTg_E);
		    be_fTg_E_d = array_fetch(be_ptr, past_array_fTg_E, d);
		    assert(be_fTg_E_d);

		    past_array_fTg_iM1 = 
		      array_fetch(array_t*, info->main_trans, i_real-1);
		    assert(past_array_fTg_iM1);
		    be_fTg_iM1_d = array_fetch(be_ptr, past_array_fTg_iM1, d);
		    assert(be_fTg_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_fTg_i_d,
				       Be_And(be_mgr,
					      be_g_i_d,
					      Be_Or(be_mgr,
						    be_f_i_d,
						    Be_Ite(be_mgr,
							   be_l_i,
							   be_fTg_E_d,
							   be_fTg_iM1_d))));
		
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  Adding constraint at depth %u: ",d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }

	      } /* for(d = 0; d <= info->past_depth; d++) */


	    break;
	  }


	case OP_PREC:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);

	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node), fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		be_ptr be_Yf_i_d, be_result;
		const unsigned int d_lsf = min(d, lsf_info->past_depth);

		be_Yf_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_Yf_i_d);

		if(i_model == 0)
		  {
		    /* Add [[Y f]]_0^d <=> FALSE */
		    be_result = Be_Iff(be_mgr, be_Yf_i_d, Be_Falsity(be_mgr));
		  }
		else if(i_model >= 1 && d == 0)
		  {
		    /* Add [[Y f]]_i^0 <=> [[f]]_{i-1}^0 */
		    be_ptr be_f_iM1_0;
		    array_t* past_array_f_iM1 =
		      array_fetch(array_t*, lsf_info->main_trans, i_real-1);
		    assert(past_array_f_iM1);
		    be_f_iM1_0 = array_fetch(be_ptr, past_array_f_iM1, d);
		    assert(be_f_iM1_0);
		    be_result = Be_Iff(be_mgr, be_Yf_i_d, be_f_iM1_0);
		  }
		else if(i_model >= 1 && d > 0)
		  {
		    /*
		     * Add [[Y f]]_i^d <=>
		     * ITE(l_i,[[f]]_E^{d-1},[[f]]_{i-1}^min(d,pd(f)))
		     */
		    be_ptr be_f_iM1_d, be_f_E_dM1;
		    array_t* past_array_f_iM1;
		    array_t* past_array_f_E;

		    past_array_f_E =
		      array_fetch(array_t*, lsf_info->main_trans, _E_state());
		    assert(past_array_f_E);
		    be_f_E_dM1 = array_fetch(be_ptr, past_array_f_E, d-1);
		    assert(be_f_E_dM1);
		    
		    past_array_f_iM1 =
		      array_fetch(array_t*, lsf_info->main_trans, i_real-1);
		    assert(past_array_f_iM1);
		    be_f_iM1_d = array_fetch(be_ptr, past_array_f_iM1, d_lsf);
		    assert(be_f_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Yf_i_d,
				       Be_Ite(be_mgr,
					      be_l_i,
					      be_f_E_dM1,
					      be_f_iM1_d));
		  }
		else
		  internal_error(_SNH_text,__FILE__,__LINE__);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5)) {
		  fprintf(nusmv_stderr,"  Adding constraint at depth %u: ",d);
		  Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		  fprintf(nusmv_stderr, "\n");
		}

		/* The additional constraints ensuring that the past formulae
		   have stabilized at the last unrolling level */
		if(i_model >= 1 && d == info->past_depth)
		  {
		    /*
		     * Add [[Y f]]_i^d <=>
		     * ITE(l_i,[[f]]_E^{d},[[f]]_{i-1}^min(d,pd(f)))
		     */
		    be_ptr be_f_iM1_d = 0;
		    be_ptr be_f_E_d = 0;
		    array_t* past_array_f_iM1 = 0;
		    array_t* past_array_f_E = 0;

		    past_array_f_E =
		      array_fetch(array_t*, lsf_info->main_trans, _E_state());
		    assert(past_array_f_E);
		    be_f_E_d = array_fetch(be_ptr, past_array_f_E,
					   min(d, lsf_info->past_depth));
		    assert(be_f_E_d);
		    
		    past_array_f_iM1 =
		      array_fetch(array_t*, lsf_info->main_trans, i_real-1);
		    assert(past_array_f_iM1);
		    be_f_iM1_d = array_fetch(be_ptr, past_array_f_iM1,
					     min(d, lsf_info->past_depth));
		    assert(be_f_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Yf_i_d,
				       Be_Ite(be_mgr,
					      be_l_i,
					      be_f_E_d,
					      be_f_iM1_d));

		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);
		    
		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  Adding constraint at depth %u: ", d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }

	      } /*for(d = 0; d <= info->past_depth; d++)*/

	    break;
	  }


	case OP_NOTPRECNOT:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    lsf_info = (_HJL_node_info*)find_assoc(info_map, lsf);
	    assert(lsf_info);
	    lsf_past_array = array_fetch(be_ptr, lsf_info->main_trans, i_real);
	    assert(lsf_past_array);

	    if(opt_verbose_level_gt(options, 5))
	      fprintf(nusmv_stderr, " "), print_node(nusmv_stderr, node), fprintf(nusmv_stderr, "\n");

	    for(d = 0; d <= info->past_depth; d++)
	      {
		be_ptr be_Zf_i_d, be_result;
		const unsigned int d_lsf = min(d, lsf_info->past_depth);

		be_Zf_i_d = array_fetch(be_ptr, past_array, d);
		assert(be_Zf_i_d);

		if(i_model == 0)
		  {
		    /* Add [[Z f]]_0^d <=> TRUE */
		    be_result = Be_Iff(be_mgr, be_Zf_i_d, Be_Truth(be_mgr));
		  }
		else if(i_model >= 1 && d == 0)
		  {
		    /* Add [[Z f]]_i^0 <=> [[f]]_{i-1}^0 */
		    be_ptr be_f_iM1_0;
		    array_t* lsf_prev_past_array =
		      array_fetch(array_t*, lsf_info->main_trans, i_real-1);
		    assert(lsf_prev_past_array);
		    be_f_iM1_0 = array_fetch(be_ptr, lsf_prev_past_array, d);
		    assert(be_f_iM1_0);
		    be_result = Be_Iff(be_mgr, be_Zf_i_d, be_f_iM1_0);
		  }
		else if(i_model >= 1 && d > 0)
		  {
		    /*
		     * Add [[Z f]]_i^d <=>
		     * ITE(l_i,[[f]]_E^{d-1},[[f]]_{i-1}^min(d,pd(f)))
		     */
		    be_ptr be_f_iM1_d, be_f_E_dM1;
		    array_t* lsf_prev_past_array;
		    array_t* lsf_E_past_array;

		    lsf_E_past_array =
		      array_fetch(array_t*, lsf_info->main_trans, _E_state());
		    assert(lsf_E_past_array);
		    be_f_E_dM1 = array_fetch(be_ptr, lsf_E_past_array, d-1);
		    assert(be_f_E_dM1);
		    
		    lsf_prev_past_array =
		      array_fetch(array_t*, lsf_info->main_trans, i_real-1);
		    assert(lsf_prev_past_array);
		    be_f_iM1_d = array_fetch(be_ptr,lsf_prev_past_array,d_lsf);
		    assert(be_f_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Zf_i_d,
				       Be_Ite(be_mgr,
					      be_l_i,
					      be_f_E_dM1,
					      be_f_iM1_d));
		  }
		else
		  internal_error(_SNH_text,__FILE__,__LINE__);
		
		/* Save the created constraint */
		lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

		if(opt_verbose_level_gt(options, 5))
		  fprintf(nusmv_stderr, "  Adding constraint at depth %u: ", d),
		    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr), fprintf(nusmv_stderr, "\n");


		/* The additional constraints ensuring that the past formulae
		   have stabilized at the last unrolling level */
		if(i_model >= 1 && d == info->past_depth)
		  {
		    /*
		     * Add [[Z f]]_i^d <=>
		     * ITE(l_i,[[f]]_E^{d},[[f]]_{i-1}^min(d,pd(f)))
		     */
		    be_ptr be_f_iM1_d = 0;
		    be_ptr be_f_E_d = 0;
		    array_t* lsf_prev_past_array = 0;
		    array_t* lsf_E_past_array = 0;

		    lsf_E_past_array =
		      array_fetch(array_t*, lsf_info->main_trans, _E_state());
		    assert(lsf_E_past_array);
		    be_f_E_d = array_fetch(be_ptr, lsf_E_past_array,
					   min(d, lsf_info->past_depth));
		    assert(be_f_E_d);
		    
		    lsf_prev_past_array =
		      array_fetch(array_t*, lsf_info->main_trans, i_real-1);
		    assert(lsf_prev_past_array);
		    be_f_iM1_d = array_fetch(be_ptr, lsf_prev_past_array,
					     min(d, lsf_info->past_depth));
		    assert(be_f_iM1_d);
		    
		    be_result = Be_Iff(be_mgr,
				       be_Zf_i_d,
				       Be_Ite(be_mgr,
					      be_l_i,
					      be_f_E_d,
					      be_f_iM1_d));
		
		    /* Save the created constraint */
		    lsNewBegin(created_constraints,(lsGeneric)be_result,LS_NH);

		    if(opt_verbose_level_gt(options, 5)) {
		      fprintf(nusmv_stderr, "  Adding constraint at depth %u: ", d);
		      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		      fprintf(nusmv_stderr, "\n");
		    }
		  }

	      } /*for(d = 0; d <= info->past_depth; d++)*/

	    break;
	  }



	default:
	  print_node(stderr, node);
	  internal_error(_SNYI_text,__FILE__,__LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      /* Remove node from unprocessed stack */
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Mark visited */
      _set_insert(visit_cache, node);
    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache); visit_cache = 0;

  return created_constraints;
}





/*
 * Create the formula specific k-dependent constraints
 * Return a list of be_ptrs for the created constraints
 */
static lsList _HJL_formula_dependent(const BmcVarsMgr_ptr vars_mgr,
				    const node_ptr ltlspec,
				    const hash_ptr info_map,
				    const unsigned int k_model)
{
  hash_ptr visit_cache = 0;
  lsList   unprocessed_nodes = 0;
  lsList   created_constraints = 0;
  Be_Manager_ptr be_mgr;

  assert(vars_mgr);
  assert(ltlspec);
  assert(info_map);

  /* Get be manager */
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  assert(be_mgr);

  /* Create list for the created constraints */
  created_constraints = lsCreate();

  /* Debug output */
  if(opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr,
	    "Creating the formula specific k-dependent constraints for k=%d\n",
	    k_model);
    fflush(nusmv_stderr);
  }
  

  visit_cache = _set_create();
  unprocessed_nodes = lsCreate();
  lsNewBegin(unprocessed_nodes, (lsGeneric)ltlspec, LS_NH);

  
  while(lsLength(unprocessed_nodes) > 0)
    {
      node_ptr node, lsf, rsf;
      _HJL_node_info *info, *lsf_info, *rsf_info;
      array_t *f_E_past_array, *f_L_past_array;
      int has_unprocessed_children;

      /* Get node */
      if(lsFirstItem(unprocessed_nodes, (lsGeneric*)&node, LS_NH) != LS_OK ||
	 !node)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Get info */
      info = (_HJL_node_info*)find_assoc(info_map, node);
      if(!info)
	internal_error(_SNH_text,__FILE__,__LINE__);

      /* Get past_array */
      /* _HJL_make_state_vector should have built main_trans[i] before */
      assert(info->main_trans);
      assert(array_n(info->main_trans) >= 2);
      f_L_past_array = array_fetch(array_t*, info->main_trans, _L_state());
      assert(f_L_past_array);
      f_E_past_array = array_fetch(array_t*, info->main_trans, _E_state());
      assert(f_E_past_array);

      /* Traverse children and build info */
      lsf = car(node);
      rsf = cdr(node);      
      has_unprocessed_children = 0;
      switch(node_get_type(node))
	{

	case ATOM:
	case BIT:
	case DOT:
	case ARRAY:
	case TRUEEXP:
	case FALSEEXP:
	  {
	    break;
	  }

	case XOR:
	case XNOR:
	case IMPLIES:
	case IFF:
	  {
	    internal_error("%s:%d: Formula not in NNF\n",__FILE__,__LINE__);
	    break;
	  }

	case OR:
	case AND:
	case TRIGGERED:
	case SINCE:
	case RELEASES:
	case UNTIL:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(!_set_is_in(visit_cache, rsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)rsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    break;
	  }

	case NOT:
	case OP_NEXT:
	case OP_PREC:
	case OP_NOTPRECNOT:
	case OP_ONCE:
	case OP_HISTORICAL:
	case OP_GLOBAL:
	case OP_FUTURE:
	  {
	    if(!_set_is_in(visit_cache, lsf)) {
	      lsNewBegin(unprocessed_nodes, (lsGeneric)lsf, LS_NH);
	      has_unprocessed_children = 1;
	    }
	    if(has_unprocessed_children)
	      break;

	    break;
	  }


	default:
	  print_node(stderr, node);
	  internal_error("%s:%d: TJ: Something not implemented",
			 __FILE__, __LINE__);
	  break;
	}
      if(has_unprocessed_children)
	continue;
      
      /* Remove node from unprocessed stack */
      if(lsDelBegin(unprocessed_nodes, (lsGeneric*)&node) != LS_OK)
	internal_error(_SNH_text,__FILE__,__LINE__);
      
      /* Mark visited */
      _set_insert(visit_cache, node);


      if(info->main_nodes) {
	/*
	 * Add [[f]]_E^d <=> [[f]]_k^d
	 */
	int d;
	array_t *f_k_past_array =
	  array_fetch(array_t*, info->main_trans, _real_k(k_model));
	assert(f_k_past_array);

	for(d = 0; d <= info->past_depth; d++)
	  {
	    be_ptr be_f_E_d, be_f_k_d, be_result;
	    be_f_E_d = array_fetch(be_ptr, f_E_past_array, d);
	    assert(be_f_E_d);
	    be_f_k_d = array_fetch(be_ptr, f_k_past_array, d);
	    assert(be_f_k_d);
	    be_result = Be_Iff(be_mgr, be_f_E_d, be_f_k_d);
	    
	    /* Save the created constraint */
	    lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, " f: ");
	      print_node(nusmv_stderr, node);
	      fprintf(nusmv_stderr, "\n");
	      fprintf(nusmv_stderr, "  ([[f]]_E^%d <=> [[f]]_%u^%d): ",
		      d, k_model, d);
	      Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
	      fprintf(nusmv_stderr, "\n");
	    }
	  }
      }


      if(info->main_nodes)
	{
	  /*
	   * Add [[f]]_{k+1}^d <=> [[f]]_L^min(d+1,pd(f))
	   */
	  int d;
	  array_t *f_kP1_past_array =
	    array_fetch(array_t*, info->main_trans, _real_k(k_model+1));
	  assert(f_kP1_past_array);
	  
	  for(d = 0; d <= info->past_depth; d++)
	    {
	      be_ptr be_f_kP1_d, be_f_L_dP1, be_result;
	      be_f_kP1_d = array_fetch(be_ptr, f_kP1_past_array, d);
	      assert(be_f_kP1_d);
	      be_f_L_dP1 = array_fetch(be_ptr, f_L_past_array,
				       min(d+1,info->past_depth));
	      assert(be_f_L_dP1);
	      be_result = Be_Iff(be_mgr, be_f_kP1_d, be_f_L_dP1);
	    
	      /* Save the created constraint */
	      lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, " f: ");
		print_node(nusmv_stderr, node);
		fprintf(nusmv_stderr, "\n");
		fprintf(nusmv_stderr, "  ([[f]]_%d^%d <=> [[f]]_L^%d): ",
			k_model+1, d, min(d+1,info->past_depth));
		Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}
      

      if(info->aux_F_node)
	{
	  /* Add <<Ff>>_E <=> <<Ff>>_k */
	  be_ptr be_auxFf_E, be_auxFf_k, be_result;

	  be_auxFf_E = BmcVarsMgr_Name2Timed(vars_mgr,
					     info->aux_F_node,
					     _E_state(), _E_state());
	  assert(be_auxFf_E);

	  be_auxFf_k = BmcVarsMgr_Name2Timed(vars_mgr,
					     info->aux_F_node,
					     _real_k(k_model), 
					     _real_k(k_model));
	  assert(be_auxFf_k);

	  be_result = Be_Iff(be_mgr, be_auxFf_E, be_auxFf_k);
	  
	  /* Save the created constraint */
	  lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "  <<Ff>>_E <=> <<Ff>>_%d: ", k_model);
	    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}


      if(info->aux_G_node)
	{
	  /* Add <<Gf>>_E <=> <<Gf>>_k */
	  be_ptr be_auxGf_E, be_auxGf_k, be_result;

	  be_auxGf_E = BmcVarsMgr_Name2Timed(vars_mgr,
					     info->aux_G_node,
					     _E_state(), _E_state());
	  assert(be_auxGf_E);

	  be_auxGf_k = BmcVarsMgr_Name2Timed(vars_mgr,
					     info->aux_G_node,
					     _real_k(k_model), 
					     _real_k(k_model));
	  assert(be_auxGf_k);

	  be_result = Be_Iff(be_mgr, be_auxGf_E, be_auxGf_k);
	  
	  /* Save the created constraint */
	  lsNewBegin(created_constraints, (lsGeneric)be_result, LS_NH);

	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "  <<Gf>>_E <=> <<Gf>>_%d: ", k_model);
	    Be_DumpSexprTJ(vars_mgr, be_result, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}

    }
      
  lsDestroy(unprocessed_nodes, 0);
  _set_destroy(visit_cache); visit_cache = 0;

  return created_constraints;
}


/*
 * Build InLoop_i for previous_k < i <= new_k
 *
 * DEFINE:     InLoop_i := (InLoop_{i-1} || l_i)
 * CONSTRAIN:  InLoop_{i-1} => !l_i
 */
static void _HJL_unroll_InLoop(const int previous_k,
			       const int new_k,
			       array_t *InLoop_array,
			       const BmcVarsMgr_ptr vars_mgr,
			       const Bmc_IncrSat_ptr solver)
{
  int i;
  Be_Manager_ptr be_mgr;

  assert(InLoop_array);
  assert(vars_mgr);
  assert(solver);
  assert(previous_k < new_k);

  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  assert(be_mgr);

  for(i = max(previous_k + 1, 0); i <= new_k; i++)
    {
      be_ptr be_l_i = 0;
      be_ptr be_InLoop_i = 0, be_InLoop_i_minus_1 = 0;
      be_ptr be_constraint = 0;
      
      if(array_n(InLoop_array) <= i)
	array_insert(be_ptr, InLoop_array, i, 0);
      
      be_InLoop_i = array_fetch(be_ptr, InLoop_array, i);
      if(be_InLoop_i)
	/* Already done */
	continue;

      /* Get previous InLoop */
      be_InLoop_i_minus_1 = array_fetch(be_ptr, InLoop_array, i-1);
      assert(be_InLoop_i_minus_1 != 0);
      
      /* Get l_i */
      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
				     HJL_state_vars.l_var,
				     _real_k(i), _real_k(i));
      
      /* Build InLoop_i := (InLoop_{i-1} || l_i) */
      be_InLoop_i = Be_Or(be_mgr, be_InLoop_i_minus_1, be_l_i);
      Bmc_IncrSat_insert(solver, (Rbc_t*)be_InLoop_i);
      array_insert(be_ptr, InLoop_array, i, be_InLoop_i);
      
      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Constructed InLoop_%d := InLoop_%d | l_%d",
		i,i-1,i);
      if(opt_verbose_level_gt(options, 5)) {
	fprintf(nusmv_stderr, ": ");
	Be_DumpSexprTJ(vars_mgr, be_InLoop_i, nusmv_stderr);
      }
      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "\n");
      
      /* Add InLoop_{i-1} => !l_i */
      be_constraint = Be_Implies(be_mgr, be_InLoop_i_minus_1,
				 Be_Not(be_mgr, be_l_i));
      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
      
      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Forced (InLoop_%d => !l_%d)", i-1, i);
      if(opt_verbose_level_gt(options, 5)) {
	fprintf(nusmv_stderr, ": ");
	Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
      }
      if(opt_verbose_level_gt(options, 1)) {
	fprintf(nusmv_stderr, "\n");
	fflush(nusmv_stderr);
      }
    }
}


/*
 * Unroll future fragment from previous_k+1 to new_k
 * Unroll past fragment   from previous_k+1 to new_k
 */
static void _HJL_unroll_invariant(const int previous_k,
				  const int new_k,
				  array_t *InLoop_array,
				  const BmcVarsMgr_ptr vars_mgr,
				  const Bmc_IncrSat_ptr solver,
				  const node_ptr bltlspec,
				  const hash_ptr info_map,
				  const int opt_do_optimization)
{
  int i;

  assert(previous_k < new_k);

  for(i = max(previous_k + 1, 0); i <= new_k; i++)
    {
      lsGen  iterator;
      be_ptr be_constraint;
      be_ptr be_InLoop_i;
      be_ptr be_l_i;

      be_InLoop_i = array_fetch(be_ptr, InLoop_array, i);
      assert(be_InLoop_i);
      
      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
				     HJL_state_vars.l_var,
				     _real_k(i), _real_k(i));
      assert(be_l_i);
      
      {
	/* Future fragment */
	lsList new_constraints = 
	  _HJL_unroll_invariant_f(vars_mgr, bltlspec, info_map,
				  i, be_InLoop_i, be_l_i,
				  opt_do_optimization);
	/* Force constraints to be true in the fixed frame */
	lsForEachItem(new_constraints, iterator, be_constraint) {
	  Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	  Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	}
	lsDestroy(new_constraints, 0);
      }
      {
	/* Past fragment */
	lsList new_constraints = 
	  _HJL_unroll_invariant_p(vars_mgr, bltlspec, info_map,
				  i, be_InLoop_i, be_l_i,
				  opt_do_optimization);
	/* Force constraints to be true in the fixed frame */
	lsForEachItem(new_constraints, iterator, be_constraint) {
	  Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	  Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	}
	lsDestroy(new_constraints, 0);
      }
    }
}


/*
 * Force SimplePath_{i,k} for each 0<=i<k
 */
static void _HJL_SimplePaths(const unsigned int k,
			     const BmcVarsMgr_ptr vars_mgr,
			     const Bmc_IncrSat_ptr solver,
			     array_t *InLoop_array)
{
  Be_Manager_ptr be_mgr;
  unsigned int j;

  assert(vars_mgr);
  assert(solver);
  assert(InLoop_array);
  
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  assert(be_mgr);

  for(j = 0; j < k; j++)
    {
      be_ptr be_equal_pd0_translation_vars;
      be_ptr be_equal_pdx_translation_vars;
      be_ptr be_equal_aux_translation_vars;
      be_ptr be_InLoop_k, be_InLoop_j;
      be_ptr be_equal_InLoops;
      be_ptr be_constraint;
      be_ptr be_c2;
      
      be_ptr be_equal_system_vars =
	_equal_vectors_formula(vars_mgr,
			       HJL_state_vars.simple_path_system_vars,
			       _real_k(j), _real_k(k));

      be_equal_pd0_translation_vars =
	_equal_vectors_formula(vars_mgr,
			       HJL_state_vars.translation_vars_pd0,
			       _real_k(j), _real_k(k));
      
      be_equal_pdx_translation_vars =
	_equal_vectors_formula(vars_mgr,
			       HJL_state_vars.translation_vars_pdx,
			       _real_k(j), _real_k(k));
      
      be_equal_aux_translation_vars =
	_equal_vectors_formula(vars_mgr,
			       HJL_state_vars.translation_vars_aux,
			       _real_k(j), _real_k(k));
      
      be_InLoop_k = array_fetch(be_ptr, InLoop_array, k);
      assert(be_InLoop_k);
      
      be_InLoop_j = array_fetch(be_ptr, InLoop_array, j);
      assert(be_InLoop_j);
      
      be_equal_InLoops = Be_Iff(be_mgr, be_InLoop_j, be_InLoop_k);
      
      /* (s_j != s_k) */
      be_constraint = Be_Not(be_mgr, be_equal_system_vars);
      /* (s_j != s_k) | (InLoop_j != InLoop_k) */
      be_constraint = Be_Or(be_mgr,
			    be_constraint,
			    Be_Not(be_mgr, be_equal_InLoops));
      /* (s_j != s_k) | (InLoop_j != InLoop_k) | ([[f]]_j^0 != [[f]]_k^0) */
      be_constraint = Be_Or(be_mgr,
			    be_constraint,
			    Be_Not(be_mgr,
				   be_equal_pd0_translation_vars));

      be_c2 = Be_And(be_mgr,
		     Be_And(be_mgr, be_InLoop_j, be_InLoop_k),
		     Be_Or(be_mgr,
			   Be_Not(be_mgr,
				  be_equal_pdx_translation_vars),
			   Be_Not(be_mgr,
				  be_equal_aux_translation_vars)
			   )
		     );
      be_constraint = Be_Or(be_mgr, be_constraint, be_c2);
      
      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
      
      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Forced SimplePath_{%d,%d}", j, k);
      if(opt_verbose_level_gt(options, 5)) {
	fprintf(nusmv_stderr, ": ");
	Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
      }
      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "\n"); 
    }
}



/*
 * Add and force the constraint l_{k+1} <=> FALSE
 * Add and force the constraint s_E = s_k
 * Add and force the constraint LoopExists <=> InLoop_k
 * Add and force the formula specific k-dependent constraints
 */
static void _HJL_dependent(const int k,
			   const BmcVarsMgr_ptr vars_mgr,
			   const Bmc_IncrSat_ptr solver,
			   array_t *InLoop_array,
			   const be_ptr be_LoopExists,
			   const node_ptr bltlspec,
			   const hash_ptr info_map)
{
  Be_Manager_ptr be_mgr;

  assert(k >= 0);
  assert(vars_mgr);
  assert(solver);
  assert(InLoop_array);
  assert(bltlspec);
  assert(info_map);

  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  assert(be_mgr);


  /*
   * Add and force the constraint l_{k+1} <=> FALSE
   */
  {
    be_ptr be_l_kP1 = BmcVarsMgr_Name2Timed(vars_mgr,
					    HJL_state_vars.l_var,
					    _real_k(k+1), _real_k(k+1));
    assert(be_l_kP1);
    be_ptr be_constraint = Be_Iff(be_mgr, be_l_kP1, Be_Falsity(be_mgr));
    assert(be_constraint);
    Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
    Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
    
    if(opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "Forced (l_%d <=> FALSE)", k+1);
    if(opt_verbose_level_gt(options, 5)) {
      fprintf(nusmv_stderr, ": ");
      Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
    }
    if(opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "\n");
  }


  /*
   * Add and force the constraint s_E = s_k
   */
  {
    be_ptr be_constraint =
      _equal_vectors_formula(vars_mgr,
			     HJL_state_vars.simple_path_system_vars,
			     _E_state(), _real_k(k));
    
    Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
    Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
    
    if(opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "Forced (s_E = s_%d)", k);
    if(opt_verbose_level_gt(options, 5)) {
      fprintf(nusmv_stderr, ": ");
      Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
    }
    if(opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "\n");
  }


  /*
   * Add and force the constraint LoopExists <=> InLoop_k
   */
  {
    be_ptr be_InLoop_k, be_constraint;
    be_InLoop_k = array_fetch(be_ptr, InLoop_array, k);
    assert(be_InLoop_k);
    be_constraint = Be_Iff(be_mgr, be_LoopExists, be_InLoop_k);
    Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
    Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
    
    if(opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "Forced (LoopExists <=> InLoop_%d)", k);
    if(opt_verbose_level_gt(options, 5)) {
      fprintf(nusmv_stderr, ": ");
      Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
    }
    if(opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "\n");
  }

  
  /*
   * Add and force the formula specific k-dependent constraints
   */
  {
    lsGen  iterator;
    be_ptr be_constraint;
    
    lsList new_constraints = 
      _HJL_formula_dependent(vars_mgr, bltlspec, info_map, k);
    
    /* Force constraints to be true in the fixed frame */
    lsForEachItem(new_constraints, iterator, be_constraint)
      {
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
      }
    
    lsDestroy(new_constraints, 0);
  }
}


/*
 * Return Booleanized, negated and NNFed formula with fairness constraints
 */
static node_ptr _HJL_make_boolean_formula(Prop_ptr ltlprop,
					  const int opt_print_dotty)
{
  node_ptr bltlspec;
  node_ptr fltlspec;

  fltlspec = Compile_FlattenSexpExpandDefine(Enc_get_symb_encoding(), 
					     Prop_get_expr(ltlprop), 
					     Nil);

  if(opt_print_dotty)
    {
      FILE *fp = fopen("debug_fltlspec.dot", "w");
      /*TJ_print_sexp_dotty(fp, fltlspec);*/
      fclose(fp);
    }
    
  fltlspec = Bmc_Wff_MkNot(detexpr2bexpr(fltlspec));
        
  /*
   * Add fairness constraints to the formula
   */
  {
    SexpFsm_ptr sexp_fsm = Prop_get_bool_sexp_fsm(ltlprop);
    assert(sexp_fsm != (SexpFsm_ptr)NULL);
    
    node_ptr justice_list = SexpFsm_get_justice(sexp_fsm);
    while(!is_list_empty(justice_list))
      {
	fltlspec = Bmc_Wff_MkAnd(fltlspec, Bmc_Wff_MkGlobally(Bmc_Wff_MkEventually(car(justice_list))));
	justice_list = cdr(justice_list);
      }
    
    node_ptr compassion_list = SexpFsm_get_compassion(sexp_fsm);
    if(!is_list_empty(compassion_list))
      internal_error("TJ: Compassion not handled. DON'T report to NuSMV team!");
  }
  
  bltlspec = Bmc_Wff_MkNnf(fltlspec);
  
  if(opt_print_dotty) {
    FILE *fp = fopen("debug_bltlspec.dot", "w");
    /*TJ_print_sexp_dotty(fp, bltlspec);*/
    fclose(fp);
  }

  return bltlspec;
}



/*
 * Find state and input variables that occurr in the transition relation
 * and in the formula.
 *
 * HJL_state_vars.transition_state_vars will have the state vars occurring
 *   in the transition relation
 * HJL_state_vars.transition_input_vars will have the input vars occurring
 *   in the transition relation
 * HJL_state_vars.formula_state_vars will have the state vars occurring
 *   in the formula bltlspec
 * HJL_state_vars.formula_input_vars will have the input vars occurring
 *   in the formula bltlspec
 * HJL_state_vars.simple_path_system_vars will be the union of
 *   HJL_state_vars.transition_state_vars,
 *   HJL_state_vars.formula_state_vars, and
 *   HJL_state_vars.formula_input_vars,
 */
static void _HJL_find_relevant_vars(BmcVarsMgr_ptr vars_mgr,
				    Bmc_Fsm_ptr be_fsm,
				    node_ptr bltlspec)
{
  hash_ptr state_var_set = 0;
  hash_ptr input_var_set = 0;
  int i = 0;
  lsGen  iterator;
  node_ptr node;
  

  /* Clean lists if not empty */
  if(lsLength(HJL_state_vars.transition_state_vars) > 0) {
    lsDestroy(HJL_state_vars.transition_state_vars, 0);
    HJL_state_vars.transition_state_vars = lsCreate();
  }
  if(lsLength(HJL_state_vars.transition_input_vars) > 0) {
    lsDestroy(HJL_state_vars.transition_input_vars, 0);
    HJL_state_vars.transition_input_vars = lsCreate();
  }
  if(lsLength(HJL_state_vars.formula_state_vars) > 0) {
    lsDestroy(HJL_state_vars.formula_state_vars, 0);
    HJL_state_vars.formula_state_vars = lsCreate();
  }
  if(lsLength(HJL_state_vars.formula_input_vars) > 0) {
    lsDestroy(HJL_state_vars.formula_input_vars, 0);
    HJL_state_vars.formula_input_vars = lsCreate();
  }
  if(lsLength(HJL_state_vars.simple_path_system_vars) > 0) {
    lsDestroy(HJL_state_vars.simple_path_system_vars, 0);
    HJL_state_vars.simple_path_system_vars = lsCreate();
  }



  /* Compute the state variable set */
  state_var_set = _set_create();
  for(i = 0; i < BmcVarsMgr_GetStateVarsNum(vars_mgr); i++)
    _set_insert(state_var_set, BmcVarsMgr_VarIndex2Name(vars_mgr, i));
    
  /* Compute the input variable set */
  input_var_set = _set_create();
  for(i = BmcVarsMgr_GetStateVarsNum(vars_mgr);
      i < (BmcVarsMgr_GetStateVarsNum(vars_mgr) +
	   BmcVarsMgr_GetInputVarsNum(vars_mgr));
      i++)
    _set_insert(input_var_set, BmcVarsMgr_VarIndex2Name(vars_mgr, i));
  
  

  /* Get all the variables occurring in the transition relation */
  /*lsList tr_vars = Be_TJ_get_vars(vars_mgr, Bmc_Fsm_GetTrans(be_fsm));*/
  lsList tr_vars = Be_TJ_get_vars(vars_mgr,
				  Bmc_Model_GetUnrolling(be_fsm, 0, 1));

  /* Classify the variables to state and input ones */
  lsForEachItem(tr_vars, iterator, node)
    {
      if(_set_is_in(state_var_set, node))
	lsNewEnd(HJL_state_vars.transition_state_vars, (lsGeneric)node, LS_NH);
      else if(_set_is_in(input_var_set, node))
	lsNewEnd(HJL_state_vars.transition_input_vars, (lsGeneric)node, LS_NH);
      else
	internal_error("%s:%d: TJ: Unknown variable type (not state or input)",
		       __FILE__, __LINE__);
    }
  /* Release list */
  lsDestroy(tr_vars, 0); tr_vars = 0;
  
  
  
  /* Get all the variables occurring in the formula */
  lsList f_vars = _HJL_find_formula_vars(vars_mgr, bltlspec);
  
  /* Classify the variables to state and input ones */
  lsForEachItem(f_vars, iterator, node)
    {
      if(_set_is_in(state_var_set, node))
	lsNewEnd(HJL_state_vars.formula_state_vars, (lsGeneric)node, LS_NH);
      else if(_set_is_in(input_var_set, node))
	lsNewEnd(HJL_state_vars.formula_input_vars, (lsGeneric)node, LS_NH);
      else
	internal_error("%s:%d: TJ: Unknown variable type (not state or input)",
		       __FILE__, __LINE__);
    }
  /* Release list */
  lsDestroy(f_vars, 0); f_vars = 0;
  
  
  
  /* Build simple_path_system_vars */
  {
    hash_ptr union_set = _set_create();
    
    lsForEachItem(HJL_state_vars.transition_state_vars, iterator, node)
      {
	if(!_set_is_in(union_set, node))
	  {
	    _set_insert(union_set, node);
	      lsNewEnd(HJL_state_vars.simple_path_system_vars,
		       (lsGeneric)node, LS_NH);
	  }
      }
    lsForEachItem(HJL_state_vars.formula_state_vars, iterator, node)
      {
	if(!_set_is_in(union_set, node))
	  {
	    _set_insert(union_set, node);
	    lsNewEnd(HJL_state_vars.simple_path_system_vars,
		     (lsGeneric)node, LS_NH);
	  }
      }
#ifdef BENCHMARKING
    if (opt_cone_of_influence(options) == true)
      {
	fprintf(nusmv_stderr, "TJ: COI removed %d state variables\n",
		lsLength(HJL_state_vars.original_state_vars) -
		lsLength(HJL_state_vars.simple_path_system_vars));
      }
#endif  
    lsForEachItem(HJL_state_vars.formula_input_vars, iterator, node)
      {
	if(!_set_is_in(union_set, node))
	  {
	    _set_insert(union_set, node);
	    lsNewEnd(HJL_state_vars.simple_path_system_vars,
		     (lsGeneric)node, LS_NH);
	  }
      }
    
    _set_destroy(union_set);
  }
  
  
  _set_destroy(state_var_set);
  _set_destroy(input_var_set);
}




int Bmc_HJL_zigzag_incr(Prop_ptr ltlprop,
			const int max_k,
			const int opt_do_virtual_unrolling,
			const int opt_do_completeness_check,
			const int opt_do_optimization)
{
  node_ptr bltlspec;  /* Its booleanization */
  Bmc_Fsm_ptr be_fsm; /* The corresponding be fsm */
  BmcVarsMgr_ptr vars_mgr = NULL;
  /* sat solver instance */
  Bmc_IncrSat_ptr solver;
  be_ptr beProb; /* A problem in BE format */
  /* ---------------------------------------------------------------------- */
  /* Here a property was selected                                           */
  /* ---------------------------------------------------------------------- */
  int current_k;
  int previous_k = -1; /* used to create Be of execution from time 0 */
  boolean found_solution = 0;
  boolean formula_forced_to_true = 0;
  static const int TJ_print_dotty = 0;
  Be_Manager_ptr be_mgr = NULL;
  
  hash_ptr info_map = 0;
  
  be_ptr be_LoopExists = 0;
  
  /* Array of be_ptr, InLoop_array[i] has the be for InLoop_i
   * (where i is a model index) */
  array_t* InLoop_array;
  

  /* Force each subformula to have its own state variable.
   * Costly, only for debugging/cross validation */
  const int opt_force_state_vars = 0;
  
#ifdef BENCHMARKING
  times(&time_bmc_start);
#endif  


  /* checks that a property was selected: */
  nusmv_assert(ltlprop != PROP(NULL));
  
  /* checks if it has already been checked: */
  if (Prop_get_status(ltlprop) != Prop_Unchecked) {
    /* aborts this check */
    return 0;
  }
  
  /* solver construction: */
  solver = Bmc_IncrSat_init(get_sat_solver(options));
  if (solver == 0) {
    fprintf(nusmv_stderr, "Incremental sat solver '%s' is not available.\n", 
	    get_sat_solver(options));
    return 1;
  }

  
  if (opt_cone_of_influence(options) == true) {
    Prop_apply_coi_for_bmc(ltlprop, global_fsm_builder);
  }
  
  be_fsm = Prop_get_be_fsm(ltlprop);
  if (be_fsm == (Bmc_Fsm_ptr) NULL) {
    Prop_set_fsm_to_master(ltlprop);
    be_fsm = Prop_get_be_fsm(ltlprop);
    nusmv_assert(be_fsm != (Bmc_Fsm_ptr) NULL);
  }
  
  
  /*
   * Get vars managers
   */
  vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  
  
  /*
   * Booleanized, negated and NNFed formula with fairness constraints
   */
  bltlspec = _HJL_make_boolean_formula(ltlprop, TJ_print_dotty);

  /*
   * Initialize the state variable structure
   */
  _init_HJL_state_vars(vars_mgr);

  if (opt_verbose_level_gt(options, 4))
    _print_HJL_state_vars(nusmv_stderr);
  
  
  /*
   * Build info map for the subformulae of the booleanized LTL spec.
   */
  {
    lsList free_state_vars = lsCopy(HJL_state_vars.translation_vars_all, 0);
    info_map = _HJL_init_LTL_info(bltlspec,
				  free_state_vars,
				  HJL_state_vars.translation_vars_pd0,
				  HJL_state_vars.translation_vars_pdx,
				  HJL_state_vars.translation_vars_aux,
				  opt_force_state_vars,
				  opt_do_virtual_unrolling);
    lsJoin(HJL_state_vars.translation_vars_unused, free_state_vars, 0);
    lsDestroy(free_state_vars, 0);
  }


  /*
   * Find state and input variables that occurr in the transition relation
   * and in the formula.
   *
   * HJL_state_vars.transition_state_vars will have the state vars occurring
   *   in the transition relation
   * HJL_state_vars.transition_input_vars will have the input vars occurring
   *   in the transition relation
   * HJL_state_vars.formula_state_vars will have the state vars occurring
   *   in the formula bltlspec
   * HJL_state_vars.formula_input_vars will have the input vars occurring
   *   in the formula bltlspec
   * HJL_state_vars.simple_path_system_vars will be the union of
   *   HJL_state_vars.transition_state_vars,
   *   HJL_state_vars.formula_state_vars, and
   *   HJL_state_vars.formula_input_vars,
   */
  _HJL_find_relevant_vars(vars_mgr, be_fsm, bltlspec);


  if (opt_verbose_level_gt(options, 4))
    _print_HJL_state_vars(nusmv_stderr);



  /*
   * Create the LoopExists variable be
   */
  be_LoopExists = BmcVarsMgr_Name2Timed(vars_mgr,
					HJL_state_vars.LoopExists_var,
					0, 0);
  
  
  /*
   * Initialize the InLoop array
   */
  InLoop_array = array_alloc(be_ptr, 1);
  
  
  /*
   * Initialize state vectors of aux states L and E
   */
  _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _L_state());
  _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _E_state());

  
  /*
   * Initialize state vector of time 0
   */
  _HJL_init_state_vector(vars_mgr, bltlspec, info_map,  _real_k(0));

  
  /*
   * Insert initial conditions into the sat solver permanently
   */
  {
    be_ptr be_init = Bmc_TJ_GetInitI(be_fsm, _real_k(0));
    Bmc_IncrSat_insert(solver, be_init);
    Bmc_IncrSat_force_true(solver, be_init);
    
    if(opt_verbose_level_gt(options, 5)) {
      fprintf(nusmv_stderr, "Initial condition: ");
      Be_DumpSexprTJ(vars_mgr, be_init, nusmv_stderr);
      fprintf(nusmv_stderr, "\n");
    }
  }
  
  
  
  /***
   * Initialize l_0 to false
   */
  {
    be_ptr be_l_0 = BmcVarsMgr_Name2Timed(vars_mgr,
					  HJL_state_vars.l_var,
					  _real_k(0), _real_k(0));
    be_ptr be_constraint = Be_Iff(be_mgr, be_l_0, Be_Falsity(be_mgr));
    Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
    Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);

    if(opt_verbose_level_gt(options, 5)) {
      fprintf(nusmv_stderr, "Forced (l_0 <=> false): ");
      Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
      fprintf(nusmv_stderr, "\n");
    }
  }


  /***
   * Initialize InLoop_0 := false
   */
  {
    be_ptr be_InLoop_0 = Be_Falsity(be_mgr);
    Bmc_IncrSat_insert(solver, (Rbc_t*)be_InLoop_0);
    array_insert(be_ptr, InLoop_array, 0, be_InLoop_0);

    if(opt_verbose_level_gt(options, 5)) {
      fprintf(nusmv_stderr, "Constructed InLoop_0: ");
      Be_DumpSexprTJ(vars_mgr, be_InLoop_0, nusmv_stderr);
      fprintf(nusmv_stderr, "\n");
    }
  }


  /*
   * Make base stuff
   */
  {
    lsGen  iterator;
    be_ptr be_constraint;

    lsList new_constraints = _HJL_unroll_base(vars_mgr, bltlspec,
					      info_map, be_LoopExists,
					      opt_do_optimization);
    
    /* Force constraints to be true in the fixed frame */
    lsForEachItem(new_constraints, iterator, be_constraint) {
      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
    }
    lsDestroy(new_constraints, 0);
  }
  

  /*
   * Start problem generation:
   */
  previous_k = -1000;
  for(current_k = 0;
      current_k <= max_k && !found_solution;
      ++current_k)
    {

      Bmc_IncrSat_result satResult;
      
      if(opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr,
		"\nGenerating problem with bound %d, all possible loopbacks...\n",
                current_k);
      }


      
      /********************************************************************
       *
       * The k-invariant part of the translation
       *
       ********************************************************************/

      
      /*
       * Initialize new state vectors up to k+1
       */
      {
	int i;
	/* State 0 already initialized, start from 1 */
	for(i = max(previous_k+2,1); i <= current_k + 1; i++)
	  {
	    _HJL_init_state_vector(vars_mgr,bltlspec,info_map,_real_k(i));
	  }
      }


      /*
       * Unroll the model transition relation to the fixed frame 0
       */
      {
	int i;
	for(i = max(previous_k,0); i < current_k; i++)
	  {
	    be_ptr be_TR = Bmc_Model_GetUnrolling(be_fsm, 
						  _real_k(i),
						  _real_k(i+1));
	    Bmc_IncrSat_insert(solver, (Rbc_t*)be_TR);
	    Bmc_IncrSat_force_true(solver, (Rbc_t*)be_TR);
	    
	    if(opt_verbose_level_gt(options, 1))
	      fprintf(nusmv_stderr, "Forced T(%d,%d)", i, i+1);
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, ": ");
	      Be_DumpSexprTJ(vars_mgr, be_TR, nusmv_stderr);
	    }
	    if(opt_verbose_level_gt(options, 1))
	      fprintf(nusmv_stderr, "\n");
	  }
      }
      

      /*
       * Unroll the (l_i => (s_{i-1} = s_E)) constraint to the fixed frame 0
       */
      {
	int i;
	for(i = max(previous_k + 1, 0); i <= current_k; i++)
	  {
	    be_ptr be_l_i;
	    be_ptr be_constraint;
	    if(i < 1)
	      continue;
	    
	    be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					   HJL_state_vars.l_var,
					   _real_k(i), _real_k(i));
	    be_constraint =
	      _equal_vectors_formula(vars_mgr,
				     HJL_state_vars.simple_path_system_vars,
				     _real_k(i-1),_E_state());
	    
	    be_constraint =
	      Be_Implies(be_mgr, be_l_i, be_constraint);
	    
	    Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	    Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	    
	    if(opt_verbose_level_gt(options, 5)) {
	      fprintf(nusmv_stderr, "Added constraint l_%d => (s_%d = s_E): ",
		      current_k, current_k-1);
	      Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
	      fprintf(nusmv_stderr, "\n");
	    }
	  }
      }


      if(opt_do_optimization && (previous_k < current_k))
	{
	  /*
	   * Unroll the (l_i => LoopExists) constraint to the fixed frame 0
	   */
	  int i;
	  for(i = max(previous_k + 1, 0); i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_LoopExists);
	      
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => LoopExists: ",
			current_k);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      
      /*
       * Build InLoop_i for previous_k < i <= current_k + 1
       *
       * DEFINE:     InLoop_i := (InLoop_{i-1} || l_i)
       * CONSTRAIN:  InLoop_{i-1} => !l_i
       */
      _HJL_unroll_InLoop(previous_k,
			 current_k + 1,
			 InLoop_array,
			 vars_mgr,
			 solver);      


      /*
       * Unroll future fragment from previous_k+1 to current_k
       * Unroll past fragment   from previous_k+2 to current_k+1
       */
      _HJL_unroll_invariant(previous_k,
			    current_k,
			    InLoop_array,
			    vars_mgr,
			    solver,
			    bltlspec,
			    info_map,
			    opt_do_optimization);

      /*
       * Force negated NNF formula to true if not already done
       */
      if(!formula_forced_to_true)
	{
	  _HJL_node_info* info;
	  array_t*         past_array;
	  be_ptr           be_f_0_0;
	  
	  /* Get info */
	  info = (_HJL_node_info*)find_assoc(info_map, bltlspec);
	  assert(info);
	  
	  /* Get past_array */
	  assert(info->main_trans);
	  assert(array_n(info->main_trans) > 0);
	  past_array = array_fetch(array_t*, info->main_trans, _real_k(0));
	  assert(past_array);
	  
	  be_f_0_0 = array_fetch(be_ptr, past_array, 0);
	  assert(be_f_0_0);
	  
	  Bmc_IncrSat_insert(solver, be_f_0_0);
	  Bmc_IncrSat_force_true(solver, be_f_0_0);
	  
	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "Forced [[f]]_0^0 to true: ");
	    Be_DumpSexprTJ(vars_mgr, be_f_0_0, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	  formula_forced_to_true = true;
	}

      if(opt_do_completeness_check)
	{
	  /*
	   * Unroll the SimplePath constraints
	   */
	  {
	    int i;
	    for(i = max(previous_k+1, 0); i <= current_k; i++)
	      _HJL_SimplePaths(i, vars_mgr, solver, InLoop_array);
	  }


#ifdef BENCHMARKING
	  fprintf(nusmv_stderr, ":START:benchmarking Solving\n");
	  times(&time_solver_start);
#endif  
	  

	  /*
	   * Actually solve the problem
	   */
	  satResult = Bmc_IncrSat_solve(solver);
      
	  
#ifdef BENCHMARKING
	  {
	    times(&time_temp);
	    fprintf(nusmv_stderr, ":UTIME = %.4f secs.\n",
		    time_diff(&time_solver_start, &time_temp));
	    fprintf(nusmv_stderr, ":STOP:benchmarking Solving\n");
	    fprintf(nusmv_stderr, "TJ: completeness, k=%d, solver time=%.4f, "
		    "cumulative time=%.4f.\n",
		    current_k,
		    time_diff(&time_solver_start, &time_temp),
		    time_diff(&time_bmc_start, &time_temp));
	  }
#endif
	  
	  
	  /* Processes the result: */
	  switch (satResult)
	    {
	    case BMC_INCRSAT_SATISFIABLE_PROBLEM:
	      {
		int i;
		
		fprintf(nusmv_stderr,
			"-- could not terminate yet with bound %d",
			current_k);
		
		if (opt_verbose_level_gt(options, 2)) {
		  fprintf(nusmv_stderr, " for "); 		  
		  print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
		}
		fprintf(nusmv_stderr, "\n");
		
		if (opt_verbose_level_gt(options, 9)) {
		  for(i = 0; i <= _real_k(current_k); i++)
		    {
		      lsGen    iterator;
		      node_ptr var;
		      
		      fprintf(nusmv_stderr, "Time ");
		      _real_k_print(nusmv_stderr, i);
		      fprintf(nusmv_stderr, "\n");
		      
		      lsForEachItem(HJL_state_vars.original_state_vars, iterator, var)
			{
			  be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
								_real_k(i),
								_real_k(i));
			  int val = Bmc_IncrSat_get_solution_value(solver, be_var);
			  fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
			  print_node(nusmv_stderr, var);
			  Be_DumpSexprTJ(vars_mgr, be_var, nusmv_stderr);
			  fprintf(nusmv_stderr, " ");
			}
		      
		      lsForEachItem(HJL_state_vars.translation_vars_pd0,
				    iterator, var)
			{
			  be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr,
								var,
								_real_k(i),
								_real_k(i));
			  int val = Bmc_IncrSat_get_solution_value(solver, be_var);
			  fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
			  print_node(nusmv_stderr, var);
			  fprintf(nusmv_stderr, " ");
			}
		      
		      lsForEachItem(HJL_state_vars.translation_vars_pdx,
				    iterator, var)
			{
			  be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr,
								var,
								_real_k(i),
								_real_k(i));
			  int val = Bmc_IncrSat_get_solution_value(solver, be_var);
			  fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
			  print_node(nusmv_stderr, var);
			  fprintf(nusmv_stderr, " ");
			}
		      
		      lsForEachItem(HJL_state_vars.translation_vars_aux,
				    iterator, var)
			{
			  be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr,
								var,
								_real_k(i),
								_real_k(i));
			  int val = Bmc_IncrSat_get_solution_value(solver, be_var);
			  fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
			  print_node(nusmv_stderr, var);
			  fprintf(nusmv_stderr, " ");
			}
		      
		      fprintf(nusmv_stderr, "\n");
		    }
		}
		
		break;
	      }
	    case BMC_INCRSAT_UNSATISFIABLE_PROBLEM:
	      fprintf(nusmv_stderr, "-- ");
	      print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	      fprintf(nusmv_stderr, "  is true\n");
	      Prop_set_status(ltlprop, Prop_True);
	      
	      found_solution = true;
	      break;
	      
	    case BMC_INCRSAT_INTERNAL_ERROR:
	      internal_error("Sorry, solver answered with a fatal Internal " 
			     "Failure during problem solving.\n");
	      break;
	      
	    case BMC_INCRSAT_TIMEOUT:
	    case BMC_INCRSAT_MEMOUT:
	      internal_error("Sorry, solver ran out of resources and aborted "
			     "the execution.\n");
	      break;
	      
	    default:
	      internal_error("Bmc_GenSolveLtlTJ: Unexpected value in satResult");
	      
	    } /* switch */
	  
	  
	  
	  if(found_solution)
	    break;

	} /* if(opt_do_completeness_check) */


      /********************************************************************
       *
       * The k-dependent part of the translation
       *
       ********************************************************************/


      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Adding k-dependent stuff\n");
      

      /*
       * Go to the volatile frame 1
       */
      Bmc_IncrSat_new_frame(solver);


      /*
       * Add constraints
       */
      _HJL_dependent(current_k,
		     vars_mgr,
		     solver,
		     InLoop_array,
		     be_LoopExists,
		     bltlspec,
		     info_map);



#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:benchmarking Solving\n");
      times(&time_solver_start);
#endif  


      /*
       * Actually solve the problem
       */
      satResult = Bmc_IncrSat_solve(solver);
      

#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":UTIME = %.4f secs.\n",
		time_diff(&time_solver_start, &time_temp));
	fprintf(nusmv_stderr, ":STOP:benchmarking Solving\n");
	fprintf(nusmv_stderr, "TJ: counterexample, k=%d, solver time=%.4f, "
		"cumulative time=%.4f.\n",
		current_k,
		time_diff(&time_solver_start, &time_temp),
		time_diff(&time_bmc_start, &time_temp));
      }
#endif


     /* Processes the result: */
      switch (satResult)
	{
	case BMC_INCRSAT_UNSATISFIABLE_PROBLEM:
	  {
	    fprintf(nusmv_stderr,
		    "-- no counterexample found with bound %d",
		  current_k);
	    if (opt_verbose_level_gt(options, 2)) {
	      fprintf(nusmv_stderr, " for "); 		  
	      print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	    }
	    fprintf(nusmv_stderr, "\n");
	    
	    break;
	  }
	case BMC_INCRSAT_SATISFIABLE_PROBLEM:
	  fprintf(nusmv_stderr, "-- ");
	  print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	  fprintf(nusmv_stderr, "  is false\n");
	  Prop_set_status(ltlprop, Prop_False);
	  
	  found_solution = true;
	  
	  if (opt_counter_examples(options)) {
	    Trace_ptr trace = 
	      _generate_and_print_cntexample(vars_mgr,
					     solver,
					     Enc_get_bdd_encoding(), 
					     current_k, 
					     "BMC Counterexample",
					     HJL_state_vars.l_var);
	    Prop_set_trace(ltlprop, Trace_get_id(trace));
	  }
	  
	  break;
	  
	case BMC_INCRSAT_INTERNAL_ERROR:
	  internal_error("Sorry, solver answered with a fatal Internal " 
			 "Failure during problem solving.\n");
	  break;
	  
	case BMC_INCRSAT_TIMEOUT:
	case BMC_INCRSAT_MEMOUT:
	  internal_error("Sorry, solver ran out of resources and aborted "
			 "the execution.\n");
	  break;
	  
	default:
	  internal_error("Bmc_GenSolveLtlTJ: Unexpected value in satResult");
	  
	} /* switch */



      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Deleting k-dependent stuff\n");


      /*
       * Release the volatile frame 1
       */
      Bmc_IncrSat_pop_frame(solver);


      /*
       * Remember the current_k
       */
      previous_k = current_k;
    }


  /*
   * Deallocate info map
   */
  clear_assoc_and_free_entries(info_map, _HJL_node_info_free_func);
  info_map = 0;
  
  
  /*
   * Destroy the sat solver instance
   */
  Bmc_IncrSat_release(solver); solver = 0;
  
  return 0;
}



int Bmc_HJL_nonincr(Prop_ptr ltlprop,
		    const int max_k,
		    const int opt_do_virtual_unrolling,
		    const int opt_do_completeness_check,
		    const int opt_do_optimization)
{
  node_ptr bltlspec;  /* Its booleanization */
  Bmc_Fsm_ptr be_fsm; /* The corresponding be fsm */
  BmcVarsMgr_ptr vars_mgr = NULL;
  be_ptr beProb; /* A problem in BE format */
  /* ---------------------------------------------------------------------- */
  /* Here a property was selected                                           */
  /* ---------------------------------------------------------------------- */
  int current_k;
  int previous_k = -1; /* used to create Be of execution from time 0 */
  boolean found_solution = 0;
  static const int TJ_print_dotty = 0;
  Be_Manager_ptr be_mgr = NULL;
  
  hash_ptr info_map = 0;
  
  be_ptr be_LoopExists = 0;
  
  /* Array of be_ptr, InLoop_array[i] has the be for InLoop_i
   * (where i is a model index) */
  array_t* InLoop_array;
  
  fprintf(stderr, "Not yet updated!");
  exit(-1);
  //#error "CHECK INDICES!"

  /* Force each subformula to have its own state variable.
   * Costly, only for debugging/cross validation */
  const int opt_force_state_vars = 0;
  
#ifdef BENCHMARKING
  times(&time_bmc_start);
#endif  


  /* checks that a property was selected: */
  nusmv_assert(ltlprop != PROP(NULL));
  
  /* checks if it has already been checked: */
  if (Prop_get_status(ltlprop) != Prop_Unchecked) {
    /* aborts this check */
    return 0;
  }
  
  if (opt_cone_of_influence(options) == true) {
    Prop_apply_coi_for_bmc(ltlprop, global_fsm_builder);
  }
  
  be_fsm = Prop_get_be_fsm(ltlprop);
  if (be_fsm == (Bmc_Fsm_ptr) NULL) {
    Prop_set_fsm_to_master(ltlprop);
    be_fsm = Prop_get_be_fsm(ltlprop);
    nusmv_assert(be_fsm != (Bmc_Fsm_ptr) NULL);
  }
  
  
  /*
   * Get vars managers
   */
  vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  
  
  /*
   * Booleanized, negated and NNFed formula with fairness constraints
   */
  bltlspec = _HJL_make_boolean_formula(ltlprop, TJ_print_dotty);


  /*
   * Initialize the state variable structure
   */
  _init_HJL_state_vars(vars_mgr);
  if (opt_verbose_level_gt(options, 4))
    _print_HJL_state_vars(nusmv_stderr);
  
  
  /*
   * Build info map for the subformulae of the booleanized LTL spec.
   */
  {
    lsList free_state_vars = lsCopy(HJL_state_vars.translation_vars_all, 0);
    info_map = _HJL_init_LTL_info(bltlspec,
				  free_state_vars,
				  HJL_state_vars.translation_vars_pd0,
				  HJL_state_vars.translation_vars_pdx,
				  HJL_state_vars.translation_vars_aux,
				  opt_force_state_vars,
				  opt_do_virtual_unrolling);
    lsJoin(HJL_state_vars.translation_vars_unused, free_state_vars, 0);
    lsDestroy(free_state_vars, 0);
  }


  /*
   * Find state and input variables that occurr in the transition relation
   * and in the formula.
   *
   * HJL_state_vars.transition_state_vars will have the state vars occurring
   *   in the transition relation
   * HJL_state_vars.transition_input_vars will have the input vars occurring
   *   in the transition relation
   * HJL_state_vars.formula_state_vars will have the state vars occurring
   *   in the formula bltlspec
   * HJL_state_vars.formula_input_vars will have the input vars occurring
   *   in the formula bltlspec
   * HJL_state_vars.simple_path_system_vars will be the union of
   *   HJL_state_vars.transition_state_vars,
   *   HJL_state_vars.formula_state_vars, and
   *   HJL_state_vars.formula_input_vars,
   */
  _HJL_find_relevant_vars(vars_mgr, be_fsm, bltlspec);


  if (opt_verbose_level_gt(options, 4))
    _print_HJL_state_vars(nusmv_stderr);



  /*
   * Create the LoopExists variable be
   */
  be_LoopExists = BmcVarsMgr_Name2Timed(vars_mgr,
					HJL_state_vars.LoopExists_var,
					0, 0);
  
  


  /*
   * Start problem generation:
   */
  previous_k = -1;
  for(current_k = 1;
      current_k <= max_k && !found_solution;
      ++current_k)
    {
      Bmc_IncrSat_result satResult;
      
      /* sat solver instance */
      Bmc_IncrSat_ptr solver;
      
      /* prints a verbose message: */
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr,
		"\nGenerating problem with bound %d, all possible loopbacks...\n",
                current_k);
      }


      if(!opt_do_completeness_check)
	goto counterexample_part;

      /********************************************************************
       *
       * Completeness part
       *
       ********************************************************************/
      
      /********************************************************************
       *
       * The base and initial parts of the translation
       *
       ********************************************************************/
      
#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:generating problem\n");
      times(&time_gen_start);
#endif  
      
      /* solver construction: */
      solver = Bmc_IncrSat_init(get_sat_solver(options));
      if (solver == 0) {
	fprintf(nusmv_stderr, "Incremental sat solver '%s' is not available.\n", 
		get_sat_solver(options));
	return 1;
      }
      
      /*
       * Initialize the InLoop array
       */
      InLoop_array = array_alloc(be_ptr, 1);
      
      
      /*
       * Initialize state vectors of aux states L and E
       */
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _L_state());
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _E_state());
      
      
      /*
       * Initialize state vector of time 0
       */
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map,  _real_k(0));
      
      
      /*
       * Insert initial conditions into the sat solver permanently
       */
      {
	be_ptr be_init = Bmc_TJ_GetInitI(be_fsm, _real_k(0));
	Bmc_IncrSat_insert(solver, be_init);
	Bmc_IncrSat_force_true(solver, be_init);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Initial condition: ");
	  Be_DumpSexprTJ(vars_mgr, be_init, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
  
  
      /***
       * Initialize l_0 to false
       */
      {
	be_ptr be_l_0 = BmcVarsMgr_Name2Timed(vars_mgr,
					      HJL_state_vars.l_var,
					      _real_k(0), _real_k(0));
	be_ptr be_constraint = Be_Iff(be_mgr, be_l_0, Be_Falsity(be_mgr));
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Forced (l_0 <=> false): ");
	  Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
      
      /***
       * Initialize InLoop_0 := false
       */
      {
	be_ptr be_InLoop_0 = Be_Falsity(be_mgr);
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_InLoop_0);
	array_insert(be_ptr, InLoop_array, 0, be_InLoop_0);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Constructed InLoop_0: ");
	  Be_DumpSexprTJ(vars_mgr, be_InLoop_0, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
      
      /*
       * Make base stuff
       */
      {
	lsGen  iterator;
	be_ptr be_constraint;
	
	lsList new_constraints = _HJL_unroll_base(vars_mgr, bltlspec,
						  info_map, be_LoopExists,
						  opt_do_optimization);
	
	/* Force constraints to be true in the fixed frame */
	lsForEachItem(new_constraints, iterator, be_constraint) {
	  Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	  Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	}
	lsDestroy(new_constraints, 0);
      }
      
      
      
      /********************************************************************
       *
       * The k-invariant part of the translation
       *
       ********************************************************************/

      
      /*
       * Initialize new state vectors up to k+1
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = previous_k + 2; i <= current_k + 1; i++)
	    {
	      _HJL_init_state_vector(vars_mgr,bltlspec,info_map,_real_k(i));
	    }
	}


      /*
       * Unroll the model transition relation to the fixed frame 0
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = max(previous_k,0); i < current_k; i++)
	    {
	      be_ptr be_TR = Bmc_Model_GetUnrolling(be_fsm, 
						    _real_k(i),
						    _real_k(i+1));
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_TR);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_TR);

	      if(opt_verbose_level_gt(options, 1))
		fprintf(nusmv_stderr, "Forced T(%d,%d)", i, i+1);
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, ": ");
		Be_DumpSexprTJ(vars_mgr, be_TR, nusmv_stderr);
	      }
	      if(opt_verbose_level_gt(options, 1))
		fprintf(nusmv_stderr, "\n");
	    }
	}
      

      /*
       * Unroll the (l_i => (s_{i-1} = s_E)) constraint to the fixed frame 0
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = previous_k + 1; i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      if(i < 1)
		continue;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));

	      be_constraint =
		_equal_vectors_formula(vars_mgr,
				       HJL_state_vars.simple_path_system_vars,
				       _real_k(i-1),_E_state());
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_constraint);

	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);

	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => (s_%d = s_E): ",
			current_k, current_k-1);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      if(opt_do_optimization && (previous_k < current_k))
	{
	  /*
	   * Unroll the (l_i => LoopExists) constraint to the fixed frame 0
	   */
	  int i;
	  for(i = previous_k + 1; i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_LoopExists);
	      
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => LoopExists: ",
			current_k);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      
      _HJL_unroll_InLoop(previous_k,
			 current_k + 1,
			 InLoop_array,
			 vars_mgr,
			 solver);
      

      _HJL_unroll_invariant(previous_k,
			    current_k,
			    InLoop_array,
			    vars_mgr,
			    solver,
			    bltlspec,
			    info_map,
			    opt_do_optimization);

      /*
       * Force negated NNF formula to true if not already done
       */
      if(1)
	{
	  _HJL_node_info* info;
	  array_t*         past_array;
	  be_ptr           be_f_0_0;
	  
	  /* Get info */
	  info = (_HJL_node_info*)find_assoc(info_map, bltlspec);
	  assert(info);
	  
	  /* Get past_array */
	  assert(info->main_trans);
	  assert(array_n(info->main_trans) > 0);
	  past_array = array_fetch(array_t*, info->main_trans, _real_k(0));
	  assert(past_array);
	  
	  be_f_0_0 = array_fetch(be_ptr, past_array, 0);
	  assert(be_f_0_0);
	  
	  Bmc_IncrSat_insert(solver, be_f_0_0);
	  Bmc_IncrSat_force_true(solver, be_f_0_0);
	  
	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "Forced [[f]]_0^0 to true: ");
	    Be_DumpSexprTJ(vars_mgr, be_f_0_0, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}


      /*
       * Unroll the SimplePath constraints
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = max(previous_k, 0); i < current_k; i++)
	    _HJL_SimplePaths(i, vars_mgr, solver, InLoop_array);
	}

      
#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":STOP:generating problem\n");
	fprintf(nusmv_stderr, "TJ: generation, k=%d, time=%.4f.\n",
		current_k,
		time_diff(&time_gen_start, &time_temp));
      }
#endif

#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:benchmarking Solving\n");
      times(&time_solver_start);
#endif  
      

      /*
       * Actually solve the problem
       */
      satResult = Bmc_IncrSat_solve(solver);
      

#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":UTIME = %.4f secs.\n",
		time_diff(&time_solver_start, &time_temp));
	fprintf(nusmv_stderr, ":STOP:benchmarking Solving\n");
	fprintf(nusmv_stderr, "TJ: completeness, k=%d, solver time=%.4f, "
		"cumulative time=%.4f.\n",
		current_k,
		time_diff(&time_solver_start, &time_temp),
		time_diff(&time_bmc_start, &time_temp));
      }
#endif

      
      /* Processes the result: */
      switch (satResult)
	{
	case BMC_INCRSAT_SATISFIABLE_PROBLEM:
	  {
	    int i;

	    fprintf(nusmv_stderr,
		    "-- could not terminate yet with bound %d",
		    current_k);
	    if (opt_verbose_level_gt(options, 2)) {
	      fprintf(nusmv_stderr, " for "); 		  
	      print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	    }
	    fprintf(nusmv_stderr, "\n");
	    
	    if (opt_verbose_level_gt(options, 9)) {
	      for(i = 0; i <= _real_k(current_k); i++)
		{
		  lsGen    iterator;
		  node_ptr var;

		  fprintf(nusmv_stderr, "Time ");
		  _real_k_print(nusmv_stderr, i);
		  fprintf(nusmv_stderr, "\n");
		  
		  lsForEachItem(HJL_state_vars.original_state_vars, iterator, var)
		    {
		      be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							    _real_k(i),
							    _real_k(i));
		      int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		      fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		      print_node(nusmv_stderr, var);
		      Be_DumpSexprTJ(vars_mgr, be_var, nusmv_stderr);
		      fprintf(nusmv_stderr, " ");
		    }
		  
		  lsForEachItem(HJL_state_vars.translation_vars_pd0,
				iterator, var)
		    {
		      be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr,
							    var,
							    _real_k(i),
							    _real_k(i));
		      int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		      fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		      print_node(nusmv_stderr, var);
		      fprintf(nusmv_stderr, " ");
		    }
		  
		  lsForEachItem(HJL_state_vars.translation_vars_pdx,
				iterator, var)
		    {
		      be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr,
							    var,
							    _real_k(i),
							    _real_k(i));
		      int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		      fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		      print_node(nusmv_stderr, var);
		      fprintf(nusmv_stderr, " ");
		    }
		  
		  lsForEachItem(HJL_state_vars.translation_vars_aux,
				iterator, var)
		    {
		      be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr,
							    var,
							    _real_k(i),
							    _real_k(i));
		      int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		      fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		      print_node(nusmv_stderr, var);
		      fprintf(nusmv_stderr, " ");
		    }
		  
		  fprintf(nusmv_stderr, "\n");
		}
	    }

	    break;
	  }
	case BMC_INCRSAT_UNSATISFIABLE_PROBLEM:
	  fprintf(nusmv_stderr, "-- ");
	  print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	  fprintf(nusmv_stderr, "  is true\n");
	  Prop_set_status(ltlprop, Prop_True);
	  
	  found_solution = true;
	  break;
	  
	case BMC_INCRSAT_INTERNAL_ERROR:
	  internal_error("Sorry, solver answered with a fatal Internal " 
			 "Failure during problem solving.\n");
	  break;
	  
	case BMC_INCRSAT_TIMEOUT:
	case BMC_INCRSAT_MEMOUT:
	  internal_error("Sorry, solver ran out of resources and aborted "
			 "the execution.\n");
	  break;
	  
	default:
	  internal_error("Bmc_GenSolveLtlTJ: Unexpected value in satResult");
	  
	} /* switch */
      

      /*
       * Deallocate stuff
       */
      array_free(InLoop_array); InLoop_array = 0;
      Bmc_IncrSat_release(solver); solver = 0;
      _HJL_free_main_trans(bltlspec, info_map);

      if(found_solution)
	break;


    counterexample_part:



      /********************************************************************
       *
       * Counterexample part
       *
       ********************************************************************/

      /********************************************************************
       *
       * The base and initial parts of the translation
       *
       ********************************************************************/

#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:generating problem\n");
      times(&time_gen_start);
#endif  

      /* solver construction: */
      solver = Bmc_IncrSat_init(get_sat_solver(options));
      if (solver == 0) {
	fprintf(nusmv_stderr, "Incremental sat solver '%s' is not available.\n", 
		get_sat_solver(options));
	return 1;
      }

      /*
       * Initialize the InLoop array
       */
      InLoop_array = array_alloc(be_ptr, 1);
      
      
      /*
       * Initialize state vectors of aux states L and E
       */
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _L_state());
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _E_state());
      
      
      /*
       * Initialize state vector of time 0
       */
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map,  _real_k(0));
      
      
      /*
       * Insert initial conditions into the sat solver permanently
       */
      {
	be_ptr be_init = Bmc_TJ_GetInitI(be_fsm, _real_k(0));
	Bmc_IncrSat_insert(solver, be_init);
	Bmc_IncrSat_force_true(solver, be_init);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Initial condition: ");
	  Be_DumpSexprTJ(vars_mgr, be_init, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
  
  
      /***
       * Initialize l_0 to false
       */
      {
	be_ptr be_l_0 = BmcVarsMgr_Name2Timed(vars_mgr,
					      HJL_state_vars.l_var,
					      _real_k(0), _real_k(0));
	be_ptr be_constraint = Be_Iff(be_mgr, be_l_0, Be_Falsity(be_mgr));
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Forced (l_0 <=> false): ");
	  Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
      
      /***
       * Initialize InLoop_0 := false
       */
      {
	be_ptr be_InLoop_0 = Be_Falsity(be_mgr);
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_InLoop_0);
	array_insert(be_ptr, InLoop_array, 0, be_InLoop_0);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Constructed InLoop_0: ");
	  Be_DumpSexprTJ(vars_mgr, be_InLoop_0, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
      
      /*
       * Make base stuff
       */
      {
	lsGen  iterator;
	be_ptr be_constraint;
	
	lsList new_constraints = _HJL_unroll_base(vars_mgr, bltlspec,
						  info_map, be_LoopExists,
						  opt_do_optimization);
	
	/* Force constraints to be true in the fixed frame */
	lsForEachItem(new_constraints, iterator, be_constraint) {
	  Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	  Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	}
	lsDestroy(new_constraints, 0);
      }
      
      
      
      /********************************************************************
       *
       * The k-invariant part of the translation
       *
       ********************************************************************/

      
      /*
       * Initialize new state vectors up to k+1
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = previous_k + 2; i <= current_k + 1; i++)
	    {
	      _HJL_init_state_vector(vars_mgr,bltlspec,info_map,_real_k(i));
	    }
	}


      /*
       * Unroll the model transition relation to the fixed frame 0
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = max(previous_k,0); i < current_k; i++)
	    {
	      be_ptr be_TR = Bmc_Model_GetUnrolling(be_fsm, 
						    _real_k(i),
						    _real_k(i+1));
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_TR);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_TR);

	      if(opt_verbose_level_gt(options, 1))
		fprintf(nusmv_stderr, "Forced T(%d,%d)", i, i+1);
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, ": ");
		Be_DumpSexprTJ(vars_mgr, be_TR, nusmv_stderr);
	      }
	      if(opt_verbose_level_gt(options, 1))
		fprintf(nusmv_stderr, "\n");
	    }
	}
      

      /*
       * Unroll the (l_i => (s_{i-1} = s_E)) constraint to the fixed frame 0
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = previous_k + 1; i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      if(i < 1)
		continue;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));

	      be_constraint =
		_equal_vectors_formula(vars_mgr,
				       HJL_state_vars.simple_path_system_vars,
				       _real_k(i-1),_E_state());
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_constraint);

	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);

	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => (s_%d = s_E): ",
			current_k, current_k-1);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      if(opt_do_optimization && (previous_k < current_k))
	{
	  /*
	   * Unroll the (l_i => LoopExists) constraint to the fixed frame 0
	   */
	  int i;
	  for(i = previous_k + 1; i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_LoopExists);
	      
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => LoopExists: ",
			current_k);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      
      _HJL_unroll_InLoop(previous_k,
			 current_k + 1,
			 InLoop_array,
			 vars_mgr,
			 solver);
      

       _HJL_unroll_invariant(previous_k,
			    current_k,
			    InLoop_array,
			    vars_mgr,
			    solver,
			    bltlspec,
			    info_map,
			    opt_do_optimization);


      /*
       * Force negated NNF formula to true if not already done
       */
      if(1)
	{
	  _HJL_node_info* info;
	  array_t*         past_array;
	  be_ptr           be_f_0_0;
	  
	  /* Get info */
	  info = (_HJL_node_info*)find_assoc(info_map, bltlspec);
	  assert(info);
	  
	  /* Get past_array */
	  assert(info->main_trans);
	  assert(array_n(info->main_trans) > 0);
	  past_array = array_fetch(array_t*, info->main_trans, _real_k(0));
	  assert(past_array);
	  
	  be_f_0_0 = array_fetch(be_ptr, past_array, 0);
	  assert(be_f_0_0);
	  
	  Bmc_IncrSat_insert(solver, be_f_0_0);
	  Bmc_IncrSat_force_true(solver, be_f_0_0);
	  
	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "Forced [[f]]_0^0 to true: ");
	    Be_DumpSexprTJ(vars_mgr, be_f_0_0, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}




      /********************************************************************
       *
       * The k-dependent part of the translation
       *
       ********************************************************************/
      
      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Adding k-dependent stuff\n");
      
      /*
       * Add constraints
       */
      _HJL_dependent(current_k,
		     vars_mgr,
		     solver,
		     InLoop_array,
		     be_LoopExists,
		     bltlspec,
		     info_map);

#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":STOP:generating problem\n");
	fprintf(nusmv_stderr, "TJ: generation, k=%d, time=%.4f.\n",
		current_k,
		time_diff(&time_gen_start, &time_temp));
      }
#endif

#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:benchmarking Solving\n");
      times(&time_solver_start);
#endif  


      /*
       * Actually solve the problem
       */
      satResult = Bmc_IncrSat_solve(solver);
      

#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":UTIME = %.4f secs.\n",
		time_diff(&time_solver_start, &time_temp));
	fprintf(nusmv_stderr, ":STOP:benchmarking Solving\n");
	fprintf(nusmv_stderr, "TJ: counterexample, k=%d, solver time=%.4f, "
		"cumulative time=%.4f.\n",
		current_k,
		time_diff(&time_solver_start, &time_temp),
		time_diff(&time_bmc_start, &time_temp));
      }
#endif


     /* Processes the result: */
      switch (satResult)
	{
	case BMC_INCRSAT_UNSATISFIABLE_PROBLEM:
	  {
	    fprintf(nusmv_stderr,
		    "-- no counterexample found with bound %d",
		  current_k);
	    if (opt_verbose_level_gt(options, 2)) {
	      fprintf(nusmv_stderr, " for "); 		  
	      print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	    }
	    fprintf(nusmv_stderr, "\n");
	    
	    break;
	  }
	case BMC_INCRSAT_SATISFIABLE_PROBLEM:
	  fprintf(nusmv_stderr, "-- ");
	  print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	  fprintf(nusmv_stderr, "  is false\n");
	  Prop_set_status(ltlprop, Prop_False);
	  
	  found_solution = true;
	  
	  if (false && opt_counter_examples(options)) {
	    /* TJ: NYI */
	    Trace_ptr trace = 
	      Bmc_Utils_generate_and_print_cntexample(vars_mgr, 
						      SAT_SOLVER(solver), 
						      Enc_get_bdd_encoding(), 
						      beProb, 
						      current_k, 
						      "BMC Counterexample");
	    Prop_set_trace(ltlprop, Trace_get_id(trace));
	  }

	  if (opt_verbose_level_gt(options, 4)) {
	    unsigned int i_real;

	    fprintf(nusmv_stderr, "LoopExists = %s\n",
		    Bmc_IncrSat_get_solution_value(solver, be_LoopExists)==0?"false":"true");

	    for(i_real = 0; i_real <= _real_k(current_k); i_real++)
	      {
		lsGen    iterator;
		node_ptr var;
		be_ptr be_l_i;

		fprintf(nusmv_stderr, "Time ");
		_real_k_print(nusmv_stderr, i_real);
		fprintf(nusmv_stderr, "\n");
		
		if(i_real >= _real_k(0)) {
		  be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
						 HJL_state_vars.l_var,
						 i_real, i_real);
		  assert(be_l_i);
		  fprintf(nusmv_stderr, "%sl_%d ",
			  Bmc_IncrSat_get_solution_value(solver, be_l_i)==0?"!":" ", _model_k(i_real));
		}
		
		lsForEachItem(HJL_state_vars.original_state_vars, iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		lsForEachItem(HJL_state_vars.translation_vars_pd0,
			      iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		lsForEachItem(HJL_state_vars.translation_vars_pdx,
			      iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		lsForEachItem(HJL_state_vars.translation_vars_aux,
			      iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		fprintf(nusmv_stderr, "\n");
	      }
	  }
	  
	  break;
	  
	case BMC_INCRSAT_INTERNAL_ERROR:
	  internal_error("Sorry, solver answered with a fatal Internal " 
			 "Failure during problem solving.\n");
	  break;
	  
	case BMC_INCRSAT_TIMEOUT:
	case BMC_INCRSAT_MEMOUT:
	  internal_error("Sorry, solver ran out of resources and aborted "
			 "the execution.\n");
	  break;
	  
	default:
	  internal_error("Bmc_GenSolveLtlTJ: Unexpected value in satResult");
	  
	} /* switch */



      /*
       * Deallocate stuff
       */
      array_free(InLoop_array); InLoop_array = 0;
      Bmc_IncrSat_release(solver); solver = 0;
      _HJL_free_main_trans(bltlspec, info_map);
    }


  /*
   * Deallocate info map
   */
  clear_assoc_and_free_entries(info_map, _HJL_node_info_free_func);
  info_map = 0;
  
  
  
  return 0;
}


#if 0
int Bmc_HJL_counterexample_nonincr(Prop_ptr ltlprop, const int max_k)
{
  node_ptr bltlspec;  /* Its booleanization */
  Bmc_Fsm_ptr be_fsm; /* The corresponding be fsm */
  BmcVarsMgr_ptr vars_mgr = NULL;
  be_ptr beProb; /* A problem in BE format */
  /* ---------------------------------------------------------------------- */
  /* Here a property was selected                                           */
  /* ---------------------------------------------------------------------- */
  int current_k;
  int previous_k = -1; /* used to create Be of execution from time 0 */
  boolean found_solution = 0;
  static const int TJ_print_dotty = 0;
  Be_Manager_ptr be_mgr = NULL;
  
  hash_ptr info_map = 0;
  
  be_ptr be_LoopExists = 0;
  
  /* Array of be_ptr, InLoop_array[i] has the be for InLoop_i
   * (where i is a model index) */
  array_t* InLoop_array;
  

  /* Force each subformula to have its own state variable.
   * Costly, only for debugging/cross validation */
  const int opt_force_state_vars = 0;
  
  const int opt_do_optimization = 1;

  const int opt_do_virtual_unrolling = 1;

#ifdef BENCHMARKING
  times(&time_bmc_start);
#endif  


  /* checks that a property was selected: */
  nusmv_assert(ltlprop != PROP(NULL));
  
  /* checks if it has already been checked: */
  if (Prop_get_status(ltlprop) != Prop_Unchecked) {
    /* aborts this check */
    return 0;
  }
  
  if (opt_cone_of_influence(options) == true) {
    Prop_apply_coi_for_bmc(ltlprop, global_fsm_builder);
  }
  
  be_fsm = Prop_get_be_fsm(ltlprop);
  if (be_fsm == (Bmc_Fsm_ptr) NULL) {
    Prop_set_fsm_to_master(ltlprop);
    be_fsm = Prop_get_be_fsm(ltlprop);
    nusmv_assert(be_fsm != (Bmc_Fsm_ptr) NULL);
  }
  
  
  /*
   * Get vars managers
   */
  vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  be_mgr = BmcVarsMgr_GetBeMgr(vars_mgr);
  
  /*
   * Booleanized, negated and NNFed formula with fairness constraints
   */
  bltlspec = _HJL_make_boolean_formula(ltlprop, TJ_print_dotty);


  /*
   * Initialize the state variable structure
   */
  _init_HJL_state_vars(vars_mgr);
  if (opt_verbose_level_gt(options, 4))
    _print_HJL_state_vars(nusmv_stderr);
  
  
  /*
   * Build info map for the subformulae of the booleanized LTL spec.
   */
  {
    lsList free_state_vars = lsCopy(HJL_state_vars.translation_vars_all, 0);
    info_map = _HJL_init_LTL_info(bltlspec,
				  free_state_vars,
				  HJL_state_vars.translation_vars_pd0,
				  HJL_state_vars.translation_vars_pdx,
				  HJL_state_vars.translation_vars_aux,
				  opt_force_state_vars,
				  opt_do_virtual_unrolling);
    lsJoin(HJL_state_vars.translation_vars_unused, free_state_vars, 0);
    lsDestroy(free_state_vars, 0);
  }


   /*
   * Find state and input variables that occurr in the transition relation
   * and in the formula.
   *
   * HJL_state_vars.transition_state_vars will have the state vars occurring
   *   in the transition relation
   * HJL_state_vars.transition_input_vars will have the input vars occurring
   *   in the transition relation
   * HJL_state_vars.formula_state_vars will have the state vars occurring
   *   in the formula bltlspec
   * HJL_state_vars.formula_input_vars will have the input vars occurring
   *   in the formula bltlspec
   * HJL_state_vars.simple_path_system_vars will be the union of
   *   HJL_state_vars.transition_state_vars,
   *   HJL_state_vars.formula_state_vars, and
   *   HJL_state_vars.formula_input_vars,
   */
  _HJL_find_relevant_vars(vars_mgr, be_fsm, bltlspec);


  if (opt_verbose_level_gt(options, 4))
    _print_HJL_state_vars(nusmv_stderr);



  /*
   * Create the LoopExists variable be
   */
  be_LoopExists = BmcVarsMgr_Name2Timed(vars_mgr,
					HJL_state_vars.LoopExists_var,
					0, 0);
  
  


  /*
   * Start problem generation:
   */
  previous_k = -1;
  for(current_k = 1;
      current_k <= max_k && !found_solution;
      ++current_k)
    {
      Bmc_IncrSat_result satResult;
      
      /* sat solver instance */
      Bmc_IncrSat_ptr solver;
      
      /* prints a verbose message: */
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr,
		"\nGenerating problem with bound %d, all possible loopbacks...\n",
                current_k);
      }
      
      /********************************************************************
       *
       * Counterexample part
       *
       ********************************************************************/

      /********************************************************************
       *
       * The base and initial parts of the translation
       *
       ********************************************************************/

#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:generating problem\n");
      times(&time_gen_start);
#endif  

      /* solver construction: */
      solver = Bmc_IncrSat_init(get_sat_solver(options));
      if (solver == 0) {
	fprintf(nusmv_stderr, "Incremental sat solver '%s' is not available.\n", 
		get_sat_solver(options));
	return 1;
      }

      /*
       * Initialize the InLoop array
       */
      InLoop_array = array_alloc(be_ptr, 1);
      
      
      /*
       * Initialize state vectors of aux states L and E
       */
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _L_state());
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map, _E_state());
      
      
      /*
       * Initialize state vector of time 0
       */
      _HJL_init_state_vector(vars_mgr, bltlspec, info_map,  _real_k(0));
      
      
      /*
       * Insert initial conditions into the sat solver permanently
       */
      {
	be_ptr be_init = Bmc_TJ_GetInitI(be_fsm, _real_k(0));
	Bmc_IncrSat_insert(solver, be_init);
	Bmc_IncrSat_force_true(solver, be_init);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Initial condition: ");
	  Be_DumpSexprTJ(vars_mgr, be_init, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
  
  
      /***
       * Initialize l_0 to false
       */
      {
	be_ptr be_l_0 = BmcVarsMgr_Name2Timed(vars_mgr,
					      HJL_state_vars.l_var,
					      _real_k(0), _real_k(0));
	be_ptr be_constraint = Be_Iff(be_mgr, be_l_0, Be_Falsity(be_mgr));
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Forced (l_0 <=> false): ");
	  Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
      
      /***
       * Initialize InLoop_0 := false
       */
      {
	be_ptr be_InLoop_0 = Be_Falsity(be_mgr);
	Bmc_IncrSat_insert(solver, (Rbc_t*)be_InLoop_0);
	array_insert(be_ptr, InLoop_array, 0, be_InLoop_0);
	
	if(opt_verbose_level_gt(options, 5)) {
	  fprintf(nusmv_stderr, "Constructed InLoop_0: ");
	  Be_DumpSexprTJ(vars_mgr, be_InLoop_0, nusmv_stderr);
	  fprintf(nusmv_stderr, "\n");
	}
      }
      
      
      /*
       * Make base stuff
       */
      {
	lsGen  iterator;
	be_ptr be_constraint;
	
	lsList new_constraints = _HJL_unroll_base(vars_mgr, bltlspec,
						  info_map, be_LoopExists,
						  opt_do_optimization);
	
	/* Force constraints to be true in the fixed frame */
	lsForEachItem(new_constraints, iterator, be_constraint) {
	  Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	  Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	}
	lsDestroy(new_constraints, 0);
      }
      
      
      
      /********************************************************************
       *
       * The k-invariant part of the translation
       *
       ********************************************************************/

      
      /*
       * Initialize new state vectors up to k+1
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = previous_k + 2; i <= current_k + 1; i++)
	    {
	      _HJL_init_state_vector(vars_mgr,bltlspec,info_map,_real_k(i));
	    }
	}


      /*
       * Unroll the model transition relation to the fixed frame 0
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = max(previous_k,0); i < current_k; i++)
	    {
	      be_ptr be_TR = Bmc_Model_GetUnrolling(be_fsm, 
						    _real_k(i),
						    _real_k(i+1));
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_TR);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_TR);

	      if(opt_verbose_level_gt(options, 1))
		fprintf(nusmv_stderr, "Forced T(%d,%d)", i, i+1);
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, ": ");
		Be_DumpSexprTJ(vars_mgr, be_TR, nusmv_stderr);
	      }
	      if(opt_verbose_level_gt(options, 1))
		fprintf(nusmv_stderr, "\n");
	    }
	}
      

      /*
       * Unroll the (l_i => (s_{i-1} = s_E)) constraint to the fixed frame 0
       */
      if(previous_k < current_k)
	{
	  int i;
	  for(i = previous_k + 1; i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      if(i < 1)
		continue;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));

	      be_constraint =
		_equal_vectors_formula(vars_mgr,
				       HJL_state_vars.simple_path_system_vars,
				       _real_k(i-1),_E_state());
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_constraint);

	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);

	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => (s_%d = s_E): ",
			current_k, current_k-1);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      if(opt_do_optimization && (previous_k < current_k))
	{
	  /*
	   * Unroll the (l_i => LoopExists) constraint to the fixed frame 0
	   */
	  int i;
	  for(i = previous_k + 1; i <= current_k; i++)
	    {
	      be_ptr be_l_i;
	      be_ptr be_constraint;
	      
	      be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
					     HJL_state_vars.l_var,
					     _real_k(i), _real_k(i));
	      
	      be_constraint =
		Be_Implies(be_mgr, be_l_i, be_LoopExists);
	      
	      Bmc_IncrSat_insert(solver, (Rbc_t*)be_constraint);
	      Bmc_IncrSat_force_true(solver, (Rbc_t*)be_constraint);
	      
	      if(opt_verbose_level_gt(options, 5)) {
		fprintf(nusmv_stderr, "Added constraint l_%d => LoopExists: ",
			current_k);
		Be_DumpSexprTJ(vars_mgr, be_constraint, nusmv_stderr);
		fprintf(nusmv_stderr, "\n");
	      }
	    }
	}


      
      _HJL_unroll_InLoop(previous_k,
			 current_k + 1,
			 InLoop_array,
			 vars_mgr,
			 solver);
      

      _HJL_unroll_invariant(previous_k,
			    current_k,
			    InLoop_array,
			    vars_mgr,
			    solver,
			    bltlspec,
			    info_map,
			    opt_do_optimization);

      /*
       * Force negated NNF formula to true if not already done
       */
      if(1)
	{
	  _HJL_node_info* info;
	  array_t*         past_array;
	  be_ptr           be_f_0_0;
	  
	  /* Get info */
	  info = (_HJL_node_info*)find_assoc(info_map, bltlspec);
	  assert(info);
	  
	  /* Get past_array */
	  assert(info->main_trans);
	  assert(array_n(info->main_trans) > 0);
	  past_array = array_fetch(array_t*, info->main_trans, _real_k(0));
	  assert(past_array);
	  
	  be_f_0_0 = array_fetch(be_ptr, past_array, 0);
	  assert(be_f_0_0);
	  
	  Bmc_IncrSat_insert(solver, be_f_0_0);
	  Bmc_IncrSat_force_true(solver, be_f_0_0);
	  
	  if(opt_verbose_level_gt(options, 5)) {
	    fprintf(nusmv_stderr, "Forced [[f]]_0^0 to true: ");
	    Be_DumpSexprTJ(vars_mgr, be_f_0_0, nusmv_stderr);
	    fprintf(nusmv_stderr, "\n");
	  }
	}




      /********************************************************************
       *
       * The k-dependent part of the translation
       *
       ********************************************************************/

      if(opt_verbose_level_gt(options, 1))
	fprintf(nusmv_stderr, "Adding k-dependent stuff\n");

      /*
       * Add constraints
       */
      _HJL_dependent(current_k,
		     vars_mgr,
		     solver,
		     InLoop_array,
		     be_LoopExists,
		     bltlspec,
		     info_map);


#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":STOP:generating problem\n");
	fprintf(nusmv_stderr, "TJ: generation, k=%d, time=%.4f.\n",
		current_k,
		time_diff(&time_gen_start, &time_temp));
      }
#endif

#ifdef BENCHMARKING
      fprintf(nusmv_stderr, ":START:benchmarking Solving\n");
      times(&time_solver_start);
#endif  


      /*
       * Actually solve the problem
       */
      satResult = Bmc_IncrSat_solve(solver);
      

#ifdef BENCHMARKING
      {
	times(&time_temp);
	fprintf(nusmv_stderr, ":UTIME = %.4f secs.\n",
		time_diff(&time_solver_start, &time_temp));
	fprintf(nusmv_stderr, ":STOP:benchmarking Solving\n");
	fprintf(nusmv_stderr, "TJ: counterexample, k=%d, solver time=%.4f, "
		"cumulative time=%.4f.\n",
		current_k,
		time_diff(&time_solver_start, &time_temp),
		time_diff(&time_bmc_start, &time_temp));
      }
#endif


     /* Processes the result: */
      switch (satResult)
	{
	case BMC_INCRSAT_UNSATISFIABLE_PROBLEM:
	  {
	    fprintf(nusmv_stderr,
		    "-- no counterexample found with bound %d",
		  current_k);
	    if (opt_verbose_level_gt(options, 2)) {
	      fprintf(nusmv_stderr, " for "); 		  
	      print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	    }
	    fprintf(nusmv_stderr, "\n");
	    
	    break;
	  }
	case BMC_INCRSAT_SATISFIABLE_PROBLEM:
	  fprintf(nusmv_stderr, "-- ");
	  print_spec(nusmv_stderr, Prop_get_expr(ltlprop));
	  fprintf(nusmv_stderr, "  is false\n");
	  Prop_set_status(ltlprop, Prop_False);
	  
	  found_solution = true;
	  
	  if (false && opt_counter_examples(options)) {
	    /* TJ: NYI */
	    Trace_ptr trace = 
	      Bmc_Utils_generate_and_print_cntexample(vars_mgr, 
						      SAT_SOLVER(solver), 
						      Enc_get_bdd_encoding(), 
						      beProb, 
						      current_k, 
						      "BMC Counterexample");
	    Prop_set_trace(ltlprop, Trace_get_id(trace));
	  }

	  if (opt_verbose_level_gt(options, 4)) {
	    unsigned int i_real;

	    fprintf(nusmv_stderr, "LoopExists = %s\n",
		    Bmc_IncrSat_get_solution_value(solver, be_LoopExists)==0?"false":"true");

	    for(i_real = 0; i_real <= _real_k(current_k); i_real++)
	      {
		lsGen    iterator;
		node_ptr var;
		be_ptr be_l_i;

		fprintf(nusmv_stderr, "Time ");
		_real_k_print(nusmv_stderr, i_real);
		fprintf(nusmv_stderr, "\n");
		
		if(i_real >= _real_k(0)) {
		  be_l_i = BmcVarsMgr_Name2Timed(vars_mgr,
						 HJL_state_vars.l_var,
						 i_real, i_real);
		  assert(be_l_i);
		  fprintf(nusmv_stderr, "%sl_%d ",
			  Bmc_IncrSat_get_solution_value(solver, be_l_i)==0?"!":" ", _model_k(i_real));
		}
		
		lsForEachItem(HJL_state_vars.original_state_vars, iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		lsForEachItem(HJL_state_vars.translation_vars_pd0,
			      iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		lsForEachItem(HJL_state_vars.translation_vars_pdx,
			      iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		lsForEachItem(HJL_state_vars.translation_vars_aux,
			      iterator, var)
		  {
		    be_ptr be_var = BmcVarsMgr_Name2Timed(vars_mgr, var,
							  i_real, i_real);
		    int val = Bmc_IncrSat_get_solution_value(solver, be_var);
		    fprintf(nusmv_stderr, "%s", (val==0)?"!":" ");
		    print_node(nusmv_stderr, var);
		    fprintf(nusmv_stderr, " ");
		  }
		
		fprintf(nusmv_stderr, "\n");
	      }
	  }
	  
	  break;
	  
	case BMC_INCRSAT_INTERNAL_ERROR:
	  internal_error("Sorry, solver answered with a fatal Internal " 
			 "Failure during problem solving.\n");
	  break;
	  
	case BMC_INCRSAT_TIMEOUT:
	case BMC_INCRSAT_MEMOUT:
	  internal_error("Sorry, solver ran out of resources and aborted "
			 "the execution.\n");
	  break;
	  
	default:
	  internal_error("Bmc_GenSolveLtlTJ: Unexpected value in satResult");
	  
	} /* switch */



      /*
       * Deallocate stuff
       */
      array_free(InLoop_array); InLoop_array = 0;
      Bmc_IncrSat_release(solver); solver = 0;
      _HJL_free_main_trans(bltlspec, info_map);
    }


  /*
   * Deallocate info map
   */
  clear_assoc_and_free_entries(info_map, _HJL_node_info_free_func);
  info_map = 0;
  
  
  
  return 0;
}
#endif




/*
 * Similar to Bmc_Utils_generate_and_print_cntexample but modified
 * for my incremental SAT solver interface
 */
static Trace_ptr _generate_and_print_cntexample(BmcVarsMgr_ptr vars_mgr,
						Bmc_IncrSat_ptr solver, 
						BddEnc_ptr bdd_enc, 
						const int k, 
						const char* trace_name,
						node_ptr l_var_node)
{
  BmcSatTrace_ptr sat_trace;
  Trace_ptr trace;
  lsList be_model;
  node_ptr path;
  int bmc_tr;

  assert(l_var_node);

  /*	  
  The original
  be_model = Be_CnfModelToBeModel(BeEnc_get_be_manager(vars_mgr), 
				  SatSolver_get_model(solver));
  is replaced by the following:
  */
  be_model = lsCreate();
  {
    int i = 0;
    /* For each time step ... */
    for(i = _real_k(0); i <= _real_k(k); i++)
      {
	/* get values of state and input variables */
	int index;
	for(index = 0;
	    index < BmcVarsMgr_GetStateVarsNum(vars_mgr) +
	      BmcVarsMgr_GetInputVarsNum(vars_mgr);
	    index++)
	  {
	    be_ptr var_real_i;
	    int result;

	    var_real_i = BmcVarsMgr_VarIndex2Timed(vars_mgr, index, i, i+1);

	    result = Bmc_IncrSat_get_solution_value(solver, var_real_i);
	    if(result == 0 || result == 1)
	      {
		int rbc_index = BmcVarsMgr_VarIndex2BeIndex(vars_mgr,
							    index,
							    _model_k(i),
							    _model_k(i)+1);
		if(result == 0)
		  rbc_index = -rbc_index;
		lsNewEnd(be_model, (lsGeneric)rbc_index, LS_NH);
	      }
	  }
      }
  }

  /* As no real be_prob was ever generated,
     it is replaced by a dummy, non-constant be */
  sat_trace = BmcSatTrace_create(BmcVarsMgr_Name2Timed(vars_mgr, l_var_node, 0, 1), be_model);
  /*sat_trace = BmcSatTrace_create(be_prob, be_model);*/
  path = NodeList_to_node_ptr(
	    BmcSatTrace_get_symbolic_model(sat_trace, vars_mgr, k));
	  
  trace = Trace_create_from_state_input_list(bdd_enc,
					     trace_name,
					     TRACE_TYPE_CNTEXAMPLE,
					     path);

  bmc_tr = TraceManager_register_trace(global_trace_manager, trace);
	  
  /* Print the trace using default plugin */
  fprintf(nusmv_stdout,
	  "-- as demonstrated by the following execution sequence\n");

  /* New code: find and print the loopback point */
  {
    int i = 0;
    int loopback = -1;
    for(i = 0; i <= k; i++)
      {
	be_ptr l_var_real_i = BmcVarsMgr_Name2Timed(vars_mgr, l_var_node, _real_k(i), _real_k(i)+1);
	int result = Bmc_IncrSat_get_solution_value(solver, l_var_real_i);
	assert(result == 0 || result == 1);
	if(result == 1)
	  {
	    assert(loopback == -1);
	    loopback = i;
	  }
      }
    if(loopback != -1)
      {
	/* NuSMV prints traces so that first state is numbered 1 */
	fprintf(nusmv_stdout,
		"   the loop starts at state %d (that is redundantly printed also as state %d)\n", loopback+1-1, k+1);
      }
    else
      {
	fprintf(nusmv_stdout, "   the execution has no loop\n");
      }
  }

  TraceManager_execute_plugin(global_trace_manager, 
		      TraceManager_get_default_plugin(global_trace_manager),
		      bmc_tr);

  BmcSatTrace_destroy(&sat_trace);
  lsDestroy(be_model, 0);

  return trace;
}
