#include <assert.h>
#include "bmcInt.h"
#include "bmcIncrSat.h"
#include "rbc/rbcInt.h"
#include "utils/assoc.h"
#include "utils/utils.h"


/*
 * The MiniSat interface from Solver_C.h
 */
#define MiniSat_ptr void *

extern MiniSat_ptr MiniSat_Create();
extern void MiniSat_Delete(MiniSat_ptr);
extern int MiniSat_Nof_Variables(MiniSat_ptr);
extern int MiniSat_Nof_Clauses(MiniSat_ptr);
extern int MiniSat_New_Variable(MiniSat_ptr);
extern int MiniSat_Add_Clause(MiniSat_ptr, int *clause_lits, int num_lits);
extern int MiniSat_Solve(MiniSat_ptr);
extern int MiniSat_Solve_Assume(MiniSat_ptr, int nof_assumed_lits, int *assumed_lits);
extern int MiniSat_simplifyDB(MiniSat_ptr);
extern int MiniSat_Get_Value(MiniSat_ptr, int var_num);


/*
 * The ZChaff interface from SAT_C.h
 */
#define SAT_Manager void *
enum SAT_StatusT {
    UNDETERMINED,
    UNSATISFIABLE,
    SATISFIABLE,
    TIME_OUT,
    MEM_OUT,
    ABORTED
};
extern SAT_Manager SAT_InitManager(void);
extern void SAT_ReleaseManager (SAT_Manager    mng);
extern int SAT_AddVariable     (SAT_Manager    mng);
extern void SAT_AddClause      (SAT_Manager    mng,
				int *          clause_lits,
				int            num_lits,
				int            gid);


/*
 *
 */
static const char *SNH_error_text = "%s:%d: should not happen";

/*
 * The incremental sat solver "class"
 */
typedef struct Bmc_IncrSat_TAG
{
  /* Member functions */
  void (*release_fun)(void *data);
  void (*new_frame_fun)(void *data);
  void (*pop_frame_fun)(void *data);
  void (*insert_fun)(void *data, Rbc_t *);
  void (*force_false_fun)(void *data, Rbc_t *);
  void (*force_true_fun)(void *data, Rbc_t *);
  Bmc_IncrSat_result (*solve_fun)(void *data);
  int (*get_solution_value)(void *data, Rbc_t *);

  /* Data */
  void *solver_data;
} Bmc_IncrSat;



/*
 * ZChaff related data structures and function prototypes
 */

typedef struct Bmc_IncrSat_ZChaff_TAG
{
  /* Pointer to the actual solver */
  SAT_Manager zchaff;
  /* A hash table associating each be node with its CNF variable number */
  hash_ptr be2cnf_hash;
  /* CNF variables tha are allocated in the solver but not currently used */
  lsList free_cnf_variables;
  /* The frames */
  lsList frame_stack;
} Bmc_IncrSat_ZChaff;

typedef struct Bmc_FrameData_ZChaff_TAG
{
  unsigned int frame_num;
  lsList new_be_nodes;
  lsList be_exprs;
} Bmc_FrameData_ZChaff;

static Bmc_IncrSat_ZChaff *zchaff_init();
static void zchaff_release(void *data);
static void zchaff_new_frame(void *data);
static void zchaff_pop_frame(void *data);
static void zchaff_insert(void *data, Rbc_t *);
static void zchaff_force_false(void *data, Rbc_t *);
static void zchaff_force_true(void *data, Rbc_t *);
static Bmc_IncrSat_result zchaff_solve(void *data);
static int zchaff_get_solution_value(void *data, Rbc_t *);


/*
 * MiniSat related data structures and function prototypes
 */

typedef struct Bmc_FrameData_MiniSat_TAG
{
  int assumption_var;
  lsList new_be_nodes;
  lsList be_exprs;
} Bmc_FrameData_MiniSat;

typedef struct Bmc_IncrSat_MiniSat_TAG
{
  /* Pointer to the actual solver */
  MiniSat_ptr minisat;

  /* This is non-zero when the problem becomes permanently unsatisfiable
     (Minisat_Add_Clause or Minisat_simplifyDB have returned false) */
  int is_permanently_unsat;

  /* A hash table associating each be node with its CNF variable number */
  hash_ptr be2cnf_hash;
  /* CNF variables tha are allocated in the solver but not currently used */
  lsList free_cnf_variables;
  /* The frames */
  lsList frame_stack;
} Bmc_IncrSat_MiniSat;

static Bmc_IncrSat_MiniSat *minisat_init();
static void minisat_release(void *data);
static void minisat_new_frame(void *data);
static void minisat_pop_frame(void *data);
static void minisat_insert(void *data, Rbc_t *);
static void minisat_force_false(void *data, Rbc_t *);
static void minisat_force_true(void *data, Rbc_t *);
static Bmc_IncrSat_result minisat_solve(void *data);
static int minisat_get_solution_value(void *data, Rbc_t *);




/*
 * The generic interface functions
 */
Bmc_IncrSat_ptr Bmc_IncrSat_init(const char *solver_name)
{
  Bmc_IncrSat_ptr isat = ALLOC(Bmc_IncrSat, 1);

  if(strcasecmp("minisat", solver_name) == 0)
    {
      isat->release_fun = minisat_release;
      isat->new_frame_fun = minisat_new_frame;
      isat->pop_frame_fun = minisat_pop_frame;
      isat->insert_fun = minisat_insert;
      isat->force_false_fun = minisat_force_false;
      isat->force_true_fun = minisat_force_true;
      isat->solve_fun = minisat_solve;
      isat->get_solution_value = minisat_get_solution_value;
      isat->solver_data = minisat_init();
    }
  else
    {
      /* Default: use zchaff */
      if(strcasecmp("zchaff", solver_name) != 0)
	{
	  fprintf(nusmv_stderr,
		  "The solver %s has no incrementality support, using ZChaff\n",
		  solver_name);
	}
      isat->release_fun = zchaff_release;
      isat->new_frame_fun = zchaff_new_frame;
      isat->pop_frame_fun = zchaff_pop_frame;
      isat->insert_fun = zchaff_insert;
      isat->force_false_fun = zchaff_force_false;
      isat->force_true_fun = zchaff_force_true;
      isat->solve_fun = zchaff_solve;
      isat->get_solution_value = zchaff_get_solution_value;
      isat->solver_data = zchaff_init();
    }
  
  assert(isat);
  return isat;
}

void Bmc_IncrSat_release(Bmc_IncrSat_ptr isat)
{
  isat->release_fun(isat->solver_data);
  FREE(isat);
}

void Bmc_IncrSat_new_frame(Bmc_IncrSat_ptr isat)
{
  isat->new_frame_fun(isat->solver_data);
}

void Bmc_IncrSat_pop_frame(Bmc_IncrSat_ptr isat)
{
  isat->pop_frame_fun(isat->solver_data);
}

/*
 * Inset rbc into the current frame
 */
void Bmc_IncrSat_insert(Bmc_IncrSat_ptr isat, Rbc_t *rbc)
{
  isat->insert_fun(isat->solver_data, rbc);
}

/*
 * Force rbc to false
 */
void Bmc_IncrSat_force_false(Bmc_IncrSat_ptr isat, Rbc_t *rbc)
{
  isat->force_false_fun(isat->solver_data, rbc);
}

/*
 * Force rbc to true
 */
void Bmc_IncrSat_force_true(Bmc_IncrSat_ptr isat, Rbc_t *rbc)
{
  isat->force_true_fun(isat->solver_data, rbc);
}

/*
 * Find whether the current problem is satisfiable
 */
Bmc_IncrSat_result Bmc_IncrSat_solve(Bmc_IncrSat_ptr isat)
{
  return isat->solve_fun(isat->solver_data);

}

/*
 * Get the solution after a call to solve that returned satisfiable
 * Return 1 if rbc node is true
 * Return 0 if rbc node is false
 * Return -1 if rbc node is not used in the current problem
 */
int Bmc_IncrSat_get_solution_value(Bmc_IncrSat_ptr isat, Rbc_t *rbc)
{
  return isat->get_solution_value(isat->solver_data, rbc);
}



/**************************************************************************
 *
 * ZChaff related routines
 *
 **************************************************************************/

static void _zchaff_frame_release(Bmc_IncrSat_ZChaff *data,
				  Bmc_FrameData_ZChaff *frame);
static void _zchaff_increment_cnf(Rbc_t *rbc,
				  lsList free_cnf_variables,
				  hash_ptr be2cnf_hash,
				  Bmc_FrameData_ZChaff *frame,
				  SAT_Manager zchaff);

static Bmc_IncrSat_ZChaff *zchaff_init()
{
  Bmc_IncrSat_ZChaff *data = ALLOC(Bmc_IncrSat_ZChaff, 1);
  
  data->zchaff = SAT_InitManager();

  data->be2cnf_hash = new_assoc();
  
  data->free_cnf_variables = lsCreate();

  data->frame_stack = lsCreate();
  Bmc_FrameData_ZChaff *frame = ALLOC(Bmc_FrameData_ZChaff, 1);
  frame->frame_num = 0;
  frame->new_be_nodes = lsCreate();
  frame->be_exprs = lsCreate();
  lsNewEnd(data->frame_stack, (lsGeneric)frame, 0);

  return data;
}

static void zchaff_release(void *solver_data)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  Bmc_FrameData_ZChaff *frame;

  while(lsDelEnd(data->frame_stack, (lsGeneric *)&frame) == LS_OK)
    _zchaff_frame_release(data, frame);
  lsDestroy(data->frame_stack, 0); data->frame_stack = 0;

  free_assoc(data->be2cnf_hash); data->be2cnf_hash = 0;

  lsDestroy(data->free_cnf_variables, 0); data->free_cnf_variables = 0;

  SAT_ReleaseManager(data->zchaff); data->zchaff = 0;

  FREE(data);
}

static void _zchaff_frame_release(Bmc_IncrSat_ZChaff *data,
				  Bmc_FrameData_ZChaff *frame)
{
  Rbc_t *rbc_node;
  /* remove new_be_nodes here and release their associations */
  while(lsDelEnd(frame->new_be_nodes, (lsGeneric*) &rbc_node) == LS_OK)
    {
      int cnf_var = (int)find_assoc(data->be2cnf_hash,
				    (node_ptr)Dag_VertexGetRef(rbc_node));
      nusmv_assert(cnf_var != (int)Nil);
      nusmv_assert(cnf_var > 0 && cnf_var <= SAT_NumVariables(data->zchaff));
      lsNewBegin(data->free_cnf_variables, (lsGeneric)cnf_var, LS_NH);
      remove_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(rbc_node), 0);
    }

  /* unlock all be nodes in be_exprs */
  
  lsDestroy(frame->new_be_nodes, 0);
  lsDestroy(frame->be_exprs, 0);

  /*SAT_Reset(data->zchaff);*/

  if(frame->frame_num > 0)
    {
      /*fprintf(stderr, "SAT_DeleteClauseGroup %d\n", frame->frame_num);*/
      SAT_DeleteClauseGroup(data->zchaff, frame->frame_num);
    }

  /*SAT_Reset(data->zchaff);*/

  FREE(frame);
}


static void zchaff_new_frame(void *solver_data)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  int gid = SAT_AllocClauseGroupID(data->zchaff);
  /*fprintf(stderr, "SAT_AllocClauseGroupID %d\n", gid);*/
  if(gid <= 0)
    internal_error("Out of ZChaff group identifiers");
  
  Bmc_FrameData_ZChaff *frame = ALLOC(Bmc_FrameData_ZChaff, 1);
  frame->frame_num = gid;
  frame->new_be_nodes = lsCreate();
  frame->be_exprs = lsCreate();
  lsNewEnd(data->frame_stack, (lsGeneric)frame, 0);  
}


static void zchaff_pop_frame(void *solver_data)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  Bmc_FrameData_ZChaff *frame;
  
  /* Get frame */
  if(lsDelEnd(data->frame_stack, (lsGeneric *)&frame) != LS_OK)
    internal_error("%s:%d: Popping non-existent frame",__FILE__,__LINE__);
  /* Attempt to pop the fixed frame? */
  if(frame->frame_num <= 0)
    internal_error("%s:%d: Popping the fixed frame",__FILE__,__LINE__);
  
  _zchaff_frame_release(data, frame);
  
  /*fprintf(stderr, "CNF: %d vars, %d clauses\n",
    SAT_NumVariables(data->zchaff), SAT_NumClauses(data->zchaff));*/
  /*print_zchaff_clauses(stderr, data->zchaff);*/
}



static void zchaff_insert(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  Bmc_FrameData_ZChaff *frame = 0;
  lsStatus status;
  int cnf_var;
  int lits[1];

  /* Fetch frame */
  status = lsLastItem(data->frame_stack, (lsGeneric*)&frame, LS_NH);
  nusmv_assert(status == LS_OK);
  assert(frame);

  /* Lock rbc here */

  /* Add rbc into the set of exprs in the frame */
  status = lsNewEnd(frame->be_exprs, (lsGeneric)rbc, LS_NH);
  nusmv_assert(status == LS_OK);

  /* CNFize */
  _zchaff_increment_cnf(rbc, data->free_cnf_variables, data->be2cnf_hash,
			frame, data->zchaff);
}


static void zchaff_force_false(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  Bmc_FrameData_ZChaff *frame = 0;
  lsStatus status;
  int cnf_var;
  int lits[1];

  /* Fetch frame */
  status = lsLastItem(data->frame_stack, (lsGeneric*)&frame, LS_NH);
  nusmv_assert(status == LS_OK);
  assert(frame);

  /* Force the root of the rbc to be false */
  cnf_var = (int)find_assoc(data->be2cnf_hash,(node_ptr)Dag_VertexGetRef(rbc));
  nusmv_assert(cnf_var > 0);
  lits[0] = cnf_var * 2 + ((!Dag_VertexIsSet(rbc))?1:0);
  SAT_AddClause(data->zchaff, lits, 1, frame->frame_num);
}


static void zchaff_force_true(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  Bmc_FrameData_ZChaff *frame = 0;
  lsStatus status;
  int cnf_var;
  int lits[1];

  /* Fetch frame */
  status = lsLastItem(data->frame_stack, (lsGeneric*)&frame, LS_NH);
  nusmv_assert(status == LS_OK);
  assert(frame);

  /* Force the root of the rbc to be true */
  cnf_var = (int)find_assoc(data->be2cnf_hash,(node_ptr)Dag_VertexGetRef(rbc));
  nusmv_assert(cnf_var > 0);
  lits[0] = cnf_var * 2 + ((!Dag_VertexIsSet(rbc))?0:1);
  SAT_AddClause(data->zchaff, lits, 1, frame->frame_num);
}


static Bmc_IncrSat_result zchaff_solve(void *solver_data)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  int zchaffResult;
  long start = util_cpu_time();
  long solvingTime;

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, " CNF instance has %d vars and %d clauses\n",
	    SAT_NumVariables(data->zchaff), SAT_NumClauses(data->zchaff));
  }
  
#ifdef BENCHMARKING
  fprintf(nusmv_stderr, "TJ: cnf, vars=%d, clauses=%d\n",
	  SAT_NumVariables(data->zchaff), SAT_NumClauses(data->zchaff));
#endif

  /*print_zchaff_clauses(stderr, data->zchaff);*/

  /*
  SAT_SetClsDeletionInterval(data->zchaff, 1000);
  SAT_SetMaxUnrelevance(data->zchaff, 10);
  SAT_SetMinClsLenForDelete(data->zchaff, 2);
  SAT_SetMaxConfClsLenAllowed(data->zchaff, 10);
  */
		     
  SAT_Reset(data->zchaff);

  zchaffResult = SAT_Solve(data->zchaff);
 
  solvingTime = util_cpu_time() - start;
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, " Solver returned after %f secs, \n",
            solvingTime/1000.0);
  }
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, " Solver memory usage estimate: %d\n",
	    SAT_EstimateMemUsage(data->zchaff));
  }
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, " Solver search: %d decisions\n",
	    SAT_NumDecisions(data->zchaff)
	    );
  }

  switch(zchaffResult) {
  case UNSATISFIABLE:
    return BMC_INCRSAT_UNSATISFIABLE_PROBLEM;

  case SATISFIABLE:
    return BMC_INCRSAT_SATISFIABLE_PROBLEM;

  case TIME_OUT:
    return BMC_INCRSAT_TIMEOUT;

  case MEM_OUT:
    return BMC_INCRSAT_MEMOUT;

  case UNDETERMINED:
  case ABORTED:
    return BMC_INCRSAT_INTERNAL_ERROR;

  default:
    assert(false); /* no other cases are allowed */
  }

  assert(false);
  return BMC_INCRSAT_INTERNAL_ERROR;
}


static int zchaff_get_solution_value(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_ZChaff *data = (Bmc_IncrSat_ZChaff *)solver_data;
  int cnf_var;

  /* Fetch the cnf var num of rbc */
  cnf_var = (int)find_assoc(data->be2cnf_hash,(node_ptr)Dag_VertexGetRef(rbc));
  if(cnf_var <= 0)
    return -1;

  int val = SAT_GetVarAsgnment(data->zchaff, cnf_var);
  if(val == -1)
    return -1;

  /* Rbc node not negated */
  if(!Dag_VertexIsSet(rbc))
    return(val?1:0);
  /* Rbc node negated */
  return val?0:1;
}



static void _zchaff_increment_cnf(Rbc_t *rbc,
				  lsList free_cnf_variables,
				  hash_ptr be2cnf_hash,
				  Bmc_FrameData_ZChaff *frame,
				  SAT_Manager zchaff)
{
  int cnf_var = 0;
  Rbc_t *child;
  lsGen gen;

  /*Rbc_OutputSexpr(0, rbc, stderr); fprintf(stderr, "\n");*/

  cnf_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(rbc));

  /* Already translated? */
  if(cnf_var > 0)
    return;

  /* Translate all children */
  if(Dag_VertexGetRef(rbc)->outList) {
    gen = lsStart(Dag_VertexGetRef(rbc)->outList);
    while(lsNext(gen, (lsGeneric*) &child, LS_NH) == LS_OK)
      _zchaff_increment_cnf(child, free_cnf_variables, be2cnf_hash, frame, zchaff);
    lsFinish(gen);
  }

  /* Allocate a new variable here */
  if(lsDelBegin(free_cnf_variables, (lsGeneric*) &cnf_var) != LS_OK)
    {
      cnf_var = SAT_AddVariable(zchaff);
    }
  nusmv_assert(cnf_var > 0 && cnf_var <= SAT_NumVariables(zchaff));
  lsNewBegin(frame->new_be_nodes, (lsGeneric)rbc, LS_NH);
  insert_assoc(be2cnf_hash,
	       (node_ptr)Dag_VertexGetRef(rbc),
	       (node_ptr)cnf_var);
  
  switch(Dag_VertexGetRef(rbc)->symbol) {
  case RBCTOP:
    {
      static int lits[1]; 
      /* Add the clause {cnf_var} */
      lits[0] = cnf_var * 2;
      SAT_AddClause(zchaff, lits, 1, frame->frame_num);
      break;
    }
  case RBCAND:
    {
      static int lits[3]; 
      Rbc_t *c1 = 0, *c2 = 0, *dummy;
      int c1_var = 0, c2_var = 0;
      int c1_sign = 1, c2_sign = 1;
      int c1_lit, c2_lit;
      
      gen = lsStart(Dag_VertexGetRef(rbc)->outList);
      if(lsNext(gen, (lsGeneric*) &c1, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c1);
      c1_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(c1));
      c1_sign = (!Dag_VertexIsSet(c1))?0:1;
      assert(c1_var > 0 && c1_var <= SAT_NumVariables(zchaff));
      
      if(lsNext(gen, (lsGeneric*) &c2, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c2);
      c2_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(c2));
      c2_sign = (!Dag_VertexIsSet(c2))?0:1;
      assert(c2_var > 0 && c2_var <= SAT_NumVariables(zchaff));
      
      if(lsNext(gen, (lsGeneric*) &dummy, LS_NH) == LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      
      lsFinish(gen);
      
      assert(cnf_var != c1_var && cnf_var != c2_var && c1_var != c2_var);
      
      c1_lit = c1_var * 2 + c1_sign;
      c2_lit = c2_var * 2 + c2_sign;
      
      /* Add the binary clauses {-cnf_var c_i} */
      lits[0] = cnf_var * 2 + 1;
      lits[1] = c1_lit;
      SAT_AddClause(zchaff, lits, 2, frame->frame_num);
      lits[0] = cnf_var * 2 + 1;
      lits[1] = c2_lit;
      SAT_AddClause(zchaff, lits, 2, frame->frame_num);
      
      /* Add the clause {cnf_var -c_1 -c_2} */
      lits[0] = cnf_var * 2;
      lits[1] = c1_lit ^ 0x01;
      lits[2] = c2_lit ^ 0x01;
      SAT_AddClause(zchaff, lits, 3, frame->frame_num);
      break;
    }

  case RBCIFF:
    {
      static int lits[3]; 
      Rbc_t *c1 = 0, *c2 = 0, *dummy;
      int c1_var = 0, c2_var = 0;
      int c1_sign = 1, c2_sign = 1;
      int c1_lit, c2_lit;

      gen = lsStart(Dag_VertexGetRef(rbc)->outList);
      if(lsNext(gen, (lsGeneric*) &c1, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c1);
      c1_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(c1));
      c1_sign = (!Dag_VertexIsSet(c1))?0:1;
      assert(c1_var > 0 && c1_var <= SAT_NumVariables(zchaff));
      
      if(lsNext(gen, (lsGeneric*) &c2, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c2);
      c2_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(c2));
      c2_sign = (!Dag_VertexIsSet(c2))?0:1;
      assert(c2_var > 0 && c2_var <= SAT_NumVariables(zchaff));
      
      if(lsNext(gen, (lsGeneric*) &dummy, LS_NH) == LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      
      lsFinish(gen);
      
      assert(cnf_var != c1_var && cnf_var != c2_var && c1_var != c2_var);
      
      c1_lit = c1_var * 2 + c1_sign;
      c2_lit = c2_var * 2 + c2_sign;

      /* Add the clause {-cnf_var ~c_1 c_2} */
      lits[0] = cnf_var * 2 + 1;
      lits[1] = c1_lit ^ 0x01;
      lits[2] = c2_lit;
      SAT_AddClause(zchaff, lits, 3, frame->frame_num);
      /* Add the clause {-cnf_var c_1 ~c_2} */
      lits[0] = cnf_var * 2 + 1;
      lits[1] = c1_lit;
      lits[2] = c2_lit ^ 0x01;
      SAT_AddClause(zchaff, lits, 3, frame->frame_num);
      /* Add the clause {cnf_var ~c_1 ~c_2} */
      lits[0] = cnf_var * 2;
      lits[1] = c1_lit ^ 0x01;
      lits[2] = c2_lit ^ 0x01;
      SAT_AddClause(zchaff, lits, 3, frame->frame_num);
      /* Add the clause {cnf_var c_1 c_2} */
      lits[0] = cnf_var * 2;
      lits[1] = c1_lit;
      lits[2] = c2_lit;
      SAT_AddClause(zchaff, lits, 3, frame->frame_num);
      break;
    }

  case RBCITE: {
    static int lits[3]; 
    Rbc_t *ic = 0, *tc = 0, *ec = 0, *dummy;
    int ic_var = 0, tc_var = 0, ec_var = 0;
    int ic_sign = 1, tc_sign = 1, ec_sign = 1;
    int ic_lit = 1, tc_lit = 1, ec_lit = 1;
    
    gen = lsStart(Dag_VertexGetRef(rbc)->outList);
    if(lsNext(gen, (lsGeneric*) &ic, LS_NH) != LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    assert(ic);
    ic_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(ic));
    ic_sign = (!Dag_VertexIsSet(ic))?0:1;
    assert(ic_var > 0 && ic_var <= SAT_NumVariables(zchaff));
    
    if(lsNext(gen, (lsGeneric*) &tc, LS_NH) != LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    assert(tc);
    tc_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(tc));
    tc_sign = (!Dag_VertexIsSet(tc))?0:1;
    assert(tc_var > 0 && tc_var <= SAT_NumVariables(zchaff));
      
    if(lsNext(gen, (lsGeneric*) &ec, LS_NH) != LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    assert(ec);
    ec_var = (int)find_assoc(be2cnf_hash, (node_ptr)Dag_VertexGetRef(ec));
    ec_sign = (!Dag_VertexIsSet(ec))?0:1;
    assert(ec_var > 0 && ec_var <= SAT_NumVariables(zchaff));
    
    if(lsNext(gen, (lsGeneric*) &dummy, LS_NH) == LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    
    lsFinish(gen);
    
    assert(cnf_var != ic_var && cnf_var != tc_var &&
	   cnf_var != ec_var && ic_var != tc_var &&
	   ic_var != ec_var && tc_var != ec_var);
    
    ic_lit = ic_var * 2 + ic_sign;
    tc_lit = tc_var * 2 + tc_sign;
    ec_lit = ec_var * 2 + ec_sign;
    
    /* Add the clause {-cnf_var ~ic tc} */
    lits[0] = cnf_var * 2 + 1;
    lits[1] = ic_lit ^ 0x01;
    lits[2] = tc_lit;
    SAT_AddClause(zchaff, lits, 3, frame->frame_num);
    /* Add the clause {-cnf_var ic ec} */
    lits[0] = cnf_var * 2 + 1;
    lits[1] = ic_lit;
    lits[2] = ec_lit;
    SAT_AddClause(zchaff, lits, 3, frame->frame_num);
    /* Add the clause {cnf_var ~ic ~tc} */
    lits[0] = cnf_var * 2;
    lits[1] = ic_lit ^ 0x01;
    lits[2] = tc_lit ^ 0x01;
    SAT_AddClause(zchaff, lits, 3, frame->frame_num);
    /* Add the clause {cnf_var ic ~ec} */
    lits[0] = cnf_var * 2;
    lits[1] = ic_lit;
    lits[2] = ec_lit ^ 0x01;
    SAT_AddClause(zchaff, lits, 3, frame->frame_num);
    break;
  }

  case RBCVAR: {
    break;
  }

  default:
    internal_error("bmcIncrSat.c:bmc_increment_cnf: unknown RBC node type");
  }
}





/**************************************************************************
 *
 * MiniSat related routines
 *
 **************************************************************************/

static void _minisat_frame_release(Bmc_IncrSat_MiniSat *data,
				   Bmc_FrameData_MiniSat *frame);

static void _minisat_increment_cnf(Rbc_t *rbc,
				   Bmc_IncrSat_MiniSat *data,
				   Bmc_FrameData_MiniSat *frame,
				   MiniSat_ptr minisat);

static int _minisat_sign(Rbc_t *rbc)
{
  if(!Dag_VertexIsSet(rbc))
    return 1;
  return -1;
}

static Bmc_IncrSat_MiniSat *minisat_init()
{
  Bmc_IncrSat_MiniSat *data = ALLOC(Bmc_IncrSat_MiniSat, 1);
  
  data->minisat = MiniSat_Create();

  data->is_permanently_unsat = 0;

  data->be2cnf_hash = new_assoc();
  
  data->free_cnf_variables = lsCreate();

  /* Create the frame stack */
  data->frame_stack = lsCreate();

  /* Create the fixed frame */
  Bmc_FrameData_MiniSat *frame = ALLOC(Bmc_FrameData_MiniSat, 1);
  frame->assumption_var = 0;
  frame->new_be_nodes = lsCreate();
  frame->be_exprs = lsCreate();  
  lsNewEnd(data->frame_stack, (lsGeneric)frame, 0);

  return data;
}


static void minisat_release(void *solver_data)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  Bmc_FrameData_MiniSat *frame;

  /* Free frame stack */
  while(lsDelEnd(data->frame_stack, (lsGeneric *)&frame) == LS_OK)
    _minisat_frame_release(data, frame);
  lsDestroy(data->frame_stack, 0); data->frame_stack = 0;

  free_assoc(data->be2cnf_hash); data->be2cnf_hash = 0;

  lsDestroy(data->free_cnf_variables, 0); data->free_cnf_variables = 0;

  MiniSat_Delete(data->minisat); data->minisat = 0;

  FREE(data);
}


static void _minisat_frame_release(Bmc_IncrSat_MiniSat *data,
				   Bmc_FrameData_MiniSat *frame)
{
  Rbc_t *rbc_node;
  /* remove new_be_nodes here and release their associations */
  while(lsDelEnd(frame->new_be_nodes, (lsGeneric*) &rbc_node) == LS_OK)
    {
      /* Free the cnf variable */
      int cnf_var = (int)find_assoc(data->be2cnf_hash,
				    (node_ptr)Dag_VertexGetRef(rbc_node));
      nusmv_assert(cnf_var > 0);
      nusmv_assert(cnf_var <= MiniSat_Nof_Variables(data->minisat));
      lsNewBegin(data->free_cnf_variables, (lsGeneric)cnf_var, LS_NH);
      /* Remove the association */
      remove_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(rbc_node), 0);
    }
  
  /* unlock all be nodes in be_exprs */
  
  lsDestroy(frame->new_be_nodes, 0);
  lsDestroy(frame->be_exprs, 0);

  if(frame->assumption_var > 0)
    {
      int lits[1];
      int ret;
      lits[0] = frame->assumption_var;

      if(data->is_permanently_unsat)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      
      /* Force assumption variable to be true */
      if(MiniSat_Add_Clause(data->minisat, lits, 1) == 0)
	internal_error(SNH_error_text,__FILE__,__LINE__);

      /* Remove the clauses that depended on assumption_var */
      if(MiniSat_simplifyDB(data->minisat) == 0)
	internal_error(SNH_error_text,__FILE__,__LINE__);

      /* Note: the assumption variable is not freed as it is fixed to true */
      /* Thus some variables are "wasted" */
    }

  FREE(frame);
}


static void minisat_new_frame(void *solver_data)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  int assumption_var;

  /* Once permanently unsat, no new frames can be allocated */
  if(data->is_permanently_unsat)
    {
      nusmv_assert(lsLength(data->frame_stack) == 1);
      return;
    }

  /* Allocate a new assumption variable here */
  if(lsDelBegin(data->free_cnf_variables,(lsGeneric*)&assumption_var) != LS_OK)
    assumption_var = MiniSat_New_Variable(data->minisat);
  nusmv_assert(assumption_var > 0);
  
  Bmc_FrameData_MiniSat *frame = ALLOC(Bmc_FrameData_MiniSat, 1);
  frame->assumption_var = assumption_var;
  frame->new_be_nodes = lsCreate();
  frame->be_exprs = lsCreate();
  lsNewEnd(data->frame_stack, (lsGeneric)frame, 0);  
}


static void minisat_pop_frame(void *solver_data)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  Bmc_FrameData_MiniSat *frame;
  
  /* Once permanently unsat, no frames can be popped */
  if(data->is_permanently_unsat)
    {
      nusmv_assert(lsLength(data->frame_stack) == 1);
      return;
    }

  /* Get frame */
  if(lsDelEnd(data->frame_stack, (lsGeneric *)&frame) != LS_OK)
    internal_error("%s:%d: Popping non-existent frame",__FILE__,__LINE__);
  /* Attempt to pop the fixed frame? */
  if(frame->assumption_var <= 0)
    internal_error("%s:%d: Popping the fixed frame",__FILE__,__LINE__);
  
  _minisat_frame_release(data, frame);
}



static void minisat_insert(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  Bmc_FrameData_MiniSat *frame = 0;
  lsStatus status;

  /* Once permanently unsat, no RBCs can be inserted */
  if(data->is_permanently_unsat)
    {
      nusmv_assert(lsLength(data->frame_stack) == 1);
      return;
    }

  /* Fetch frame */
  status = lsLastItem(data->frame_stack, (lsGeneric*)&frame, LS_NH);
  nusmv_assert(status == LS_OK);
  assert(frame);

  /* Lock rbc here */

  /* Add rbc into the set of exprs in the frame */
  status = lsNewEnd(frame->be_exprs, (lsGeneric)rbc, LS_NH);
  nusmv_assert(status == LS_OK);

  /* CNFize */
  _minisat_increment_cnf(rbc, data, frame, data->minisat);
}


static void minisat_force_false(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  Bmc_FrameData_MiniSat *frame = 0;
  lsStatus status;
  int cnf_var;
  int lits[2];
  int nof_lits = 1;
  
  /* Instance already unsat and solver in undetermined state? */
  if(data->is_permanently_unsat)
    {
      nusmv_assert(lsLength(data->frame_stack) == 1);
      return;
    }

  /* Fetch frame */
  status = lsLastItem(data->frame_stack, (lsGeneric*)&frame, LS_NH);
  nusmv_assert(status == LS_OK);
  assert(frame);

  /* Force the root of the rbc to be false */
  cnf_var = (int)find_assoc(data->be2cnf_hash,(node_ptr)Dag_VertexGetRef(rbc));
  nusmv_assert(cnf_var > 0);
  lits[0] = cnf_var * _minisat_sign(rbc) * -1;
  if(frame->assumption_var > 0) {
    lits[1] = frame->assumption_var;
    nof_lits = 2;
  }
  if(MiniSat_Add_Clause(data->minisat, lits, nof_lits) == 0) {
    if(frame->assumption_var > 0)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    /* Fixed frame and the instance is now permanently unsat */
    nusmv_assert(lsLength(data->frame_stack) == 1);
    data->is_permanently_unsat = 1;
  }
}


static void minisat_force_true(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  Bmc_FrameData_MiniSat *frame = 0;
  lsStatus status;
  int cnf_var;
  int lits[2];
  int nof_lits = 1;

  /* Instance already unsat and solver in undetermined state? */
  if(data->is_permanently_unsat)
    {
      nusmv_assert(lsLength(data->frame_stack) == 1);
      return;
    }

  /* Fetch frame */
  status = lsLastItem(data->frame_stack, (lsGeneric*)&frame, LS_NH);
  nusmv_assert(status == LS_OK);
  assert(frame);

  /* Force the root of the rbc to be true */
  cnf_var = (int)find_assoc(data->be2cnf_hash,(node_ptr)Dag_VertexGetRef(rbc));
  nusmv_assert(cnf_var > 0);
  lits[0] = cnf_var * _minisat_sign(rbc);
  if(frame->assumption_var > 0) {
    lits[1] = frame->assumption_var;
    nof_lits = 2;
  }
  if(MiniSat_Add_Clause(data->minisat, lits, nof_lits) == 0) {
    if(frame->assumption_var > 0)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    /* Fixed frame and the instance is now permanently unsat */
    nusmv_assert(lsLength(data->frame_stack) == 1);
    data->is_permanently_unsat = 1;
  }
}


static Bmc_IncrSat_result minisat_solve(void *solver_data)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  int minisatResult;
  long start = util_cpu_time();
  long solvingTime;
  int *assumed_lits, nof_assumed_lits; 
  int i;
  Bmc_FrameData_MiniSat *frame;
  lsGen gen;

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, " CNF instance has %d vars and %d clauses\n",
	    MiniSat_Nof_Variables(data->minisat),
	    MiniSat_Nof_Clauses(data->minisat));
  }
  
#ifdef BENCHMARKING
  fprintf(nusmv_stderr, "TJ: cnf, vars=%d, clauses=%d\n",
	  MiniSat_Nof_Variables(data->minisat),
	  MiniSat_Nof_Clauses(data->minisat));
#endif

  if(data->is_permanently_unsat)
    {
      nusmv_assert(lsLength(data->frame_stack) == 1);
      solvingTime = util_cpu_time() - start;
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, " Solver returned after %f secs, \n",
		solvingTime/1000.0);
      }
      return BMC_INCRSAT_UNSATISFIABLE_PROBLEM;
    }

  /* Build assumptions */
  nof_assumed_lits = lsLength(data->frame_stack) - 1;
  assert(nof_assumed_lits >= 0);
  assumed_lits = ALLOC(int, nof_assumed_lits+1);
  i = 0;
  gen = lsStart(data->frame_stack);
  while(lsNext(gen, (lsGeneric*) &frame, LS_NH) == LS_OK) {
    /* For each non-fixed frame, force the assumption var to be false */
    if(frame->assumption_var > 0)
      assumed_lits[i++] = - frame->assumption_var;
  }
  lsFinish(gen);
  assert(i == nof_assumed_lits);


  /* Solve with assumptions */
  minisatResult = MiniSat_Solve_Assume(data->minisat,
				       nof_assumed_lits,
				       assumed_lits);
  
  FREE(assumed_lits);

  solvingTime = util_cpu_time() - start;
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, " Solver returned after %f secs, \n",
            solvingTime/1000.0);
  }

  switch(minisatResult) {
  case 0:
    return BMC_INCRSAT_UNSATISFIABLE_PROBLEM;

  case 1:
    return BMC_INCRSAT_SATISFIABLE_PROBLEM;

  default:
    internal_error(SNH_error_text,__FILE__,__LINE__);
  }

  assert(false);
  return BMC_INCRSAT_INTERNAL_ERROR;
}


static int minisat_get_solution_value(void *solver_data, Rbc_t *rbc)
{
  Bmc_IncrSat_MiniSat *data = (Bmc_IncrSat_MiniSat *)solver_data;
  int cnf_var;

  /*Rbc_OutputSexpr(0, rbc, stderr); fprintf(stderr, "\n");*/

  /* Trying to read a solution of permanently unsat instance? */
  if(data->is_permanently_unsat)
    internal_error(SNH_error_text,__FILE__,__LINE__);

  /* Fetch the cnf var num of rbc */
  cnf_var = (int)find_assoc(data->be2cnf_hash,(node_ptr)Dag_VertexGetRef(rbc));
  if(cnf_var <= 0)
    return -1;

  int val = MiniSat_Get_Value(data->minisat, cnf_var); /* - 1);*/

  /* Rbc node not negated */
  if(_minisat_sign(rbc) == 1)
    return(val?1:0);
  /* Rbc node negated */
  return val?0:1;
}


/* Add a clause to minisat */
/* Lits should have one free place for the assumption var */
static void _minisat_add_clause(Bmc_IncrSat_MiniSat *data,
				int nof_lits,
				int *lits,
				int assumption_var)
{
  nusmv_assert(assumption_var >= 0);

  if(assumption_var == 0)
    {
      /* Fixed frame */
      nusmv_assert(lsLength(data->frame_stack) == 1);
      if(MiniSat_Add_Clause(data->minisat, lits, nof_lits) == 0)
	data->is_permanently_unsat = 1;
    }
  else
    {
      /* Not the fixed frame */
      nusmv_assert(lsLength(data->frame_stack) > 1);
      nusmv_assert(data->is_permanently_unsat == 0);
      lits[nof_lits++] = assumption_var;
      if(MiniSat_Add_Clause(data->minisat, lits, nof_lits) == 0)
	internal_error(SNH_error_text,__FILE__,__LINE__);
    }
}

static void _minisat_increment_cnf(Rbc_t *rbc,
				   Bmc_IncrSat_MiniSat *data,
				   Bmc_FrameData_MiniSat *frame,
				   MiniSat_ptr minisat)
{
  int cnf_var = 0;
  Rbc_t *child;
  lsGen gen;
  int assumption_var;
  int lits[4]; 

  nusmv_assert(!(data->is_permanently_unsat &&
		 lsLength(data->frame_stack) > 1));

  cnf_var = (int)find_assoc(data->be2cnf_hash,
			    (node_ptr)Dag_VertexGetRef(rbc));

  /* Already translated? */
  if(cnf_var > 0)
    return;

  /*Rbc_OutputSexpr(0, rbc, stderr); fprintf(stderr, "\n");*/

  /* Translate all children */
  if(Dag_VertexGetRef(rbc)->outList) {
    gen = lsStart(Dag_VertexGetRef(rbc)->outList);
    while(lsNext(gen, (lsGeneric*) &child, LS_NH) == LS_OK)
      _minisat_increment_cnf(child, data, frame, minisat);
    lsFinish(gen);
  }

  /* Allocate a new variable here */
  if(lsDelBegin(data->free_cnf_variables, (lsGeneric*) &cnf_var) != LS_OK)
    cnf_var = MiniSat_New_Variable(minisat);
  assert(cnf_var > 0);
  assert(cnf_var <= MiniSat_Nof_Variables(minisat));
  lsNewBegin(frame->new_be_nodes, (lsGeneric)rbc, LS_NH);
  insert_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(rbc),
	       (node_ptr)cnf_var);

  /* Fetch the assumed variable */
  assumption_var = frame->assumption_var;
  assert(assumption_var >= 0);
  assert(assumption_var <= MiniSat_Nof_Variables(minisat));
    
  switch(Dag_VertexGetRef(rbc)->symbol) {
  case RBCTOP:
    {
      /* Add the clause {cnf_var assumption_var} */
      lits[0] = cnf_var;
      _minisat_add_clause(data, 1, lits, assumption_var);
      break;
    }
  case RBCAND:
    {
      Rbc_t *c1 = 0, *c2 = 0, *dummy;
      int c1_var = 0, c2_var = 0;
      int c1_sign = 1, c2_sign = 1;
      int c1_lit, c2_lit;
      
      gen = lsStart(Dag_VertexGetRef(rbc)->outList);
      if(lsNext(gen, (lsGeneric*) &c1, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c1);
      c1_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(c1));
      c1_sign = _minisat_sign(c1);
      assert(c1_var > 0 && c1_var <= MiniSat_Nof_Variables(minisat));
      
      if(lsNext(gen, (lsGeneric*) &c2, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c2);
      c2_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(c2));
      c2_sign = _minisat_sign(c2);
      assert(c2_var > 0 && c2_var <= MiniSat_Nof_Variables(minisat));
      
      if(lsNext(gen, (lsGeneric*) &dummy, LS_NH) == LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      
      lsFinish(gen);
      
      assert(cnf_var != c1_var && cnf_var != c2_var && c1_var != c2_var);
      
      c1_lit = c1_var * c1_sign;
      c2_lit = c2_var * c2_sign;
      
      /* Add the binary clauses {-cnf_var c_i assumption_var} */
      lits[0] = -cnf_var;
      lits[1] = c1_lit;
      _minisat_add_clause(data, 2, lits, assumption_var);
      lits[0] = -cnf_var;
      lits[1] = c2_lit;
      _minisat_add_clause(data, 2, lits, assumption_var);
      /* Add the clause {cnf_var -c_1 -c_2 assumption_var} */
      lits[0] = cnf_var;
      lits[1] = -c1_lit;
      lits[2] = -c2_lit;
      _minisat_add_clause(data, 3, lits, assumption_var);
      break;
    }

  case RBCIFF:
    {
      Rbc_t *c1 = 0, *c2 = 0, *dummy;
      int c1_var = 0, c2_var = 0;
      int c1_sign = 1, c2_sign = 1;
      int c1_lit, c2_lit;

      gen = lsStart(Dag_VertexGetRef(rbc)->outList);
      if(lsNext(gen, (lsGeneric*) &c1, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c1);
      c1_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(c1));
      c1_sign = _minisat_sign(c1);
      assert(c1_var > 0 && c1_var <= MiniSat_Nof_Variables(minisat));
      
      if(lsNext(gen, (lsGeneric*) &c2, LS_NH) != LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      assert(c2);
      c2_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(c2));
      c2_sign = _minisat_sign(c2);
      assert(c2_var > 0 && c2_var <= MiniSat_Nof_Variables(minisat));
      
      if(lsNext(gen, (lsGeneric*) &dummy, LS_NH) == LS_OK)
	internal_error(SNH_error_text,__FILE__,__LINE__);
      
      lsFinish(gen);
      
      assert(cnf_var != c1_var && cnf_var != c2_var && c1_var != c2_var);
      
      c1_lit = c1_var * c1_sign;
      c2_lit = c2_var * c2_sign;

      /* Add the clause {-cnf_var ~c_1 c_2 assumption_var} */
      lits[0] = -cnf_var;
      lits[1] = -c1_lit;
      lits[2] = c2_lit;
      _minisat_add_clause(data, 3, lits, assumption_var);
      /* Add the clause {-cnf_var c_1 ~c_2 assumption_var} */
      lits[0] = -cnf_var;
      lits[1] = c1_lit;
      lits[2] = -c2_lit;
      _minisat_add_clause(data, 3, lits, assumption_var);
      /* Add the clause {cnf_var ~c_1 ~c_2 assumption_var} */
      lits[0] = cnf_var;
      lits[1] = -c1_lit;
      lits[2] = -c2_lit;
      _minisat_add_clause(data, 3, lits, assumption_var);
      /* Add the clause {cnf_var c_1 c_2 assumption_var} */
      lits[0] = cnf_var;
      lits[1] = c1_lit;
      lits[2] = c2_lit;
      _minisat_add_clause(data, 3, lits, assumption_var);
      break;
    }

  case RBCITE: {
    Rbc_t *ic = 0, *tc = 0, *ec = 0, *dummy;
    int ic_var = 0, tc_var = 0, ec_var = 0;
    int ic_sign = 1, tc_sign = 1, ec_sign = 1;
    int ic_lit = 1, tc_lit = 1, ec_lit = 1;
    
    gen = lsStart(Dag_VertexGetRef(rbc)->outList);
    if(lsNext(gen, (lsGeneric*) &ic, LS_NH) != LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    assert(ic);
    ic_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(ic));
    ic_sign = _minisat_sign(ic);
    assert(ic_var > 0 && ic_var <= MiniSat_Nof_Variables(minisat));
    
    if(lsNext(gen, (lsGeneric*) &tc, LS_NH) != LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    assert(tc);
    tc_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(tc));
    tc_sign = _minisat_sign(tc);
    assert(tc_var > 0 && tc_var <= MiniSat_Nof_Variables(minisat));
      
    if(lsNext(gen, (lsGeneric*) &ec, LS_NH) != LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    assert(ec);
    ec_var = (int)find_assoc(data->be2cnf_hash, (node_ptr)Dag_VertexGetRef(ec));
    ec_sign = _minisat_sign(ec);
    assert(ec_var > 0 && ec_var <= MiniSat_Nof_Variables(minisat));
    
    if(lsNext(gen, (lsGeneric*) &dummy, LS_NH) == LS_OK)
      internal_error(SNH_error_text,__FILE__,__LINE__);
    
    lsFinish(gen);
    
    assert(cnf_var != ic_var && cnf_var != tc_var &&
	   cnf_var != ec_var && ic_var != tc_var &&
	   ic_var != ec_var && tc_var != ec_var);
    
    ic_lit = ic_var * ic_sign;
    tc_lit = tc_var * tc_sign;
    ec_lit = ec_var * ec_sign;
    
    /* Add the clause {-cnf_var ~ic tc assumption_var} */
    lits[0] = -cnf_var;
    lits[1] = -ic_lit;
    lits[2] = tc_lit;
    _minisat_add_clause(data, 3, lits, assumption_var);
    /* Add the clause {-cnf_var ic ec assumption_var} */
    lits[0] = -cnf_var;
    lits[1] = ic_lit;
    lits[2] = ec_lit;
    _minisat_add_clause(data, 3, lits, assumption_var);
    /* Add the clause {cnf_var ~ic ~tc assumption_var} */
    lits[0] = cnf_var;
    lits[1] = -ic_lit;
    lits[2] = -tc_lit;
    _minisat_add_clause(data, 3, lits, assumption_var);
    /* Add the clause {cnf_var ic ~ec assumption_var} */
    lits[0] = cnf_var;
    lits[1] = ic_lit;
    lits[2] = -ec_lit;
    _minisat_add_clause(data, 3, lits, assumption_var);
    break;
  }

  case RBCVAR: {
    break;
  }

  default:
    internal_error("bmcIncrSat.c:bmc_increment_cnf: unknown RBC node type");
  }
}
